from numpy import array, zeros, argmin, inf, equal, ndim
from scipy.spatial.distance import cdist
import time

def dtw(x, y, dist, local=False, nonlinear=1, normalize=None):
    """
    Computes Dynamic Time Warping (DTW) of two sequences.

    :param array x: N1*M array
    :param array y: N2*M array
    :param func dist: distance used as cost measure

    Returns the minimum distance, the cost matrix, the accumulated cost matrix, and the wrap path.
    """
    assert len(x)
    assert len(y)
    r, c = len(x), len(y)
    D0 = zeros((r + 1, c + 1))
    D1 = D0[1:, 1:] # view
    if not local:
        D0[1:, 0] = inf
    D0[0, 1:] = inf
    for i in range(r):
        for j in range(c):
            D1[i, j] = dist(x[i], y[j])
    C = D1.copy()
    for i in range(r):
        for j in range(c):
            D1[i, j] += min(D0[i, j], D0[i, j+1]*nonlinear, D0[i+1, j]*nonlinear)
    if len(x)==1:
        path = zeros(len(y)), range(len(y))
    elif len(y) == 1:
        path = range(len(x)), zeros(len(x))
    elif local:
        pos = argmin(D1[:,-1])+1
        D0 = D0[:pos+1, :]
        D1 = D1[:pos, :]
        path = _traceback(D0)
    else:
        path = _traceback(D0)
    res = D1[-1, -1]

    if normalize == 'min':
        res /= min(r, c)
    elif normalize == 'max':
        res /= max(r, c)
    elif normalize == 'sum':
        res /= r+c
    elif normalize == 'path':
        res /= len(path[0])
    return res, C, D1, path

def _traceback(D):
    i, j = array(D.shape) - 2
    p, q = [i], [j]
    while ((i > 0) or (j > 0)):
        tb = argmin((D[i, j], D[i, j+1], D[i+1, j]))
        if i == -1:
            tb = 2
        if j == -1:
            tb = 1
        if (tb == 0):
            i -= 1
            j -= 1
        elif (tb == 1):
            i -= 1
        else: # (tb == 2):
            j -= 1
        p.insert(0, i)
        q.insert(0, j)
    a = [x for x in zip(p, q) if min(x) >=0]
    p, q = [x[0] for x in a], [x[1] for x in a]
    return array(p), array(q)

if __name__ == '__main__':
    from sklearn.metrics.pairwise import manhattan_distances
    #x = [0, 0, 1, 1, 2, 4, 2, 1, 2, 0]
    #y = [1, 1, 1, 2, 2, 2, 2, 3, 2, 0]
    x = [1, 1, 1, 1, 1, 1, 1, 4, 1, 1, 1, 1]
    y = [3, 4, 4, 4, 4, 5, 6]
    dist_fun = manhattan_distances

    dist, cost, acc, path = dtw(x, y, dist_fun)
    print(dist, path)
    # vizualize
    from matplotlib import pyplot as plt
    plt.imshow(cost.T, origin='lower', cmap=plt.cm.Reds, interpolation='nearest')
    plt.plot(path[0], path[1], '-o') # relation
    plt.xticks(range(len(x)), x)
    plt.yticks(range(len(y)), y)
    plt.xlabel('x')
    plt.ylabel('y')
    plt.axis('tight')
    plt.title('Minimum distance: {}'.format(dist))
    plt.show()
