'''
Created on Apr 10, 2012

@author: keeper
'''

class FontProperty(object):
    
    def __init__(self, key, value, children):
        
        self._key = key
        self._value = value
        self._children = children
        
    def get_key(self):
        return self._key
    
    def get_value(self):
        return self._value
    
    def get_children(self):
        return self._children
    
class SetCharFontProperty(FontProperty):    
    @classmethod
    def from_item(cls, item):
        return cls('SETCHAR', ''.join(['SETCHAR', item.get_raw_value()]), [])
        
    
class SelectFontProperty(FontProperty):      
    @classmethod
    def from_font(cls, font):
        return cls("SELECTFONT", ''.join(["SELECTFONT", font.get_select_font_data()]), [])

    
class MapCharFontProperty(FontProperty):
    
    def __init__(self, key, value, children):
        FontProperty.__init__(self, key, value, children)
        children = self.get_children()
        
    @classmethod
    def from_font_and_item(cls, font, item):
        children = []
        children.append(SelectFontProperty.from_font(font))
        children.append(SetCharFontProperty.from_item(item))
        
        return cls("MAP", "MAP", children)
        
    
class MapFontProperty(FontProperty):
    def get_select_font_data(self):
        return ''.join(map(lambda s: ''.join([' ', s]), self.get_value().split()[1:3]))        

class CharacterFontProperty(FontProperty):
    
    def __init__(self, key, value, children):
        FontProperty.__init__(self, key, value, children)
        self._mapped_to = -1
    
    def get_ord_val(self):
        params = self.get_value().split()
        if len(params) < 3:
            return -1
        if params[1] == "C":
            return ord(params[2])
        elif params[1] == "O":
            return int(params[2], 8)
        else:
            return -1
        
    def get_raw_value(self):
        return ''.join(map(lambda s: ''.join([' ', s]), self.get_value().split()[1:3]))
        
    def remove_mapping(self):
        children = self.get_children()
        i = 0
        while i < len(children):
            if isinstance(children[i], MapCharFontProperty):
                del children[i]
            else:
                i += 1
        
        self._mapped_to = -1
                
        return self
        
    def map_to(self, item, font):
        if not isinstance(item, type(self)): return
        if not isinstance(font, MapFontProperty): return
        
        self.remove_mapping().get_children().append(MapCharFontProperty.from_font_and_item(font, item))
        self._mapped_to = item.get_ord_val()
        
    def mapped_to(self):
        return self._mapped_to

class MetricFile(object):
    
    def __init__(self, filename, fontname):
        
        self._root = []
        self.load_pl(filename, fontname)
    
    def parse_data(self, f, root):
        c = f.read(1)
        
        while c != "":
            
            if c == "(":
                d = f.read(1)
                s = []
                children = []
                
                while d != ")":
                    if d == "(":
                        f.seek(-1, 1)
                        self.parse_data(f, children)
                    else:                
                        s.append(d)
                        
                    d = f.read(1)
                    
                d = ''.join(s)
                root.append(self.get_font_property(d.split()[0], d.strip(), children))
                
            elif c == ")":
                f.seek(-1, 1)
                return
            
            c = f.read(1)
            
    def get_font_property(self, key, value, children):
        if key == "MAPFONT":
            return MapFontProperty(key, value, children)
        elif key == "CHARACTER":
            return CharacterFontProperty(key, value, children)
        
        return FontProperty(key, value, children)
        
    def output_to_file(self, f, root):
        for item in root:
            f.write(''.join(['(', item.get_value()]))
            self.output_to_file(f, item.get_children())
            f.write(')\n')
    
    def load_pl(self, filename, fontname):
        f = open(filename, "r+")
        
        data = f.read()
        f.seek(0)
        f.truncate()
        f.write(''.join(['(MAPFONT D 0 (FONTNAME ', fontname, '))', chr(10)]))
        f.write(data)
        f.seek(0)
                
        self.parse_data(f, self._root)
        
        f.close()
        
    def save_pl(self, filename):
        f = open(filename, "w")
        
        self.output_to_file(f, self._root)
        
        f.close()
        
    def get_data(self):
        return self._root
    
    def get_character_vector(self):
        return [item.get_ord_val() for item in self.get_characters()]
    
    def get_encoding_vector(self):
        return [item.mapped_to() for item in self.get_characters()]
    
    def map_characters(self, encoding_vector):
        char_vector = self.get_character_vector()
        chars = self.get_characters()
        font = self.get_mapfont()
        
        if len(char_vector) != len(encoding_vector):
            return False
        
        for i in range(len(chars)):
            target = self.get_character_by_ord_val(encoding_vector[i])
            if not target is None:
                chars[i].map_to(target, font)
    
    def get_characters(self):
        return [item for item in self._root if isinstance(item, CharacterFontProperty)]

    def get_character_by_ord_val(self, value):
        chars = self.get_characters()
        
        for item in chars:
            if value == item.get_ord_val():
                return item
            
        return None

    def get_mapfont(self):       
        for item in self._root:
            if isinstance(item, MapFontProperty):
                return item
            
        return None
        