'''
Created on Apr 8, 2012

@author: Igor Liska
'''

import StringIO
import string

class PfbSegmentHeader(object):
    
    def __init__(self, data, offset = 0):
        self.HEADER_START = 0x80
        self.SMALL_HEADER_LENGTH = 2
        self.HEADER_LENGTH = 6
        self.TYPE_ASCII = 1
        self.TYPE_BINARY = 2
        self.TYPE_EOF = 3
    
        self.type = 0
        self.size = 0
        self.start = 0
        
        self.parse_data(data, offset)
    
    '''
     Parses the PFB file's segment's header. The header consists of 6 bytes, first is always 0x80 (128).
     Second byte is type of the segment (1 = ASCII, 2 = binary, 3 = eof). Remaining 4 bytes are size of the
     segment in little-endian.
    '''
    def parse_data(self, data, offset = 0):
        if ((offset < 0) or len(data) < (self.SMALL_HEADER_LENGTH + offset) or ((ord(data[offset]) & self.HEADER_START) != self.HEADER_START)):
            return False
        
        self.type = ord(data[offset + 1]) & 0xff;
        
        if (self.type == self.TYPE_ASCII or self.type == self.TYPE_BINARY):
            if (len(data) < (self.HEADER_LENGTH + offset)):
                return False
            
            b0 = ord(data[offset + 2]) & 0xff
            b1 = ord(data[offset + 3]) & 0xff
            b2 = ord(data[offset + 4]) & 0xff
            b3 = ord(data[offset + 5]) & 0xff
            
            self.size = b0 + (b1 << 8) + (b2 << 16) + (b3 << 24)
        else:
            self.size = 0
        
        self.start = offset + self.get_header_size()
        
        return True
    
    def get_type(self):
        return self.type
    
    def get_size(self):
        return self.size
    
    def get_header_size(self):
        if (self.type == self.TYPE_ASCII or self.type == self.TYPE_BINARY):
            return self.HEADER_LENGTH
        return self.SMALL_HEADER_LENGTH
    
    def get_size_with_header(self):
        return self.get_size() + self.get_header_size()
    
    def get_start(self):
        return self.start 
    
    def set_size(self, size):
        self.size = size
        
    def output(self):
        out_bytes = [self.HEADER_START, self.type]
        
        if (self.type == self.TYPE_ASCII or self.type == self.TYPE_BINARY):
            size = self.get_size()            
            out_bytes.append(size & 0xff)
            for _ in range(3):
                size >>= 8
                out_bytes.append(size & 0xff)
        
        return ''.join(map(chr, out_bytes))
    

class PfbSegment(object):
    
    def __init__(self, data, header):
        self.header = header
        offset = header.get_start()
        length = header.get_size()
        self.data = data[offset:(offset + length)]
 
    def get_size(self):
        return len(self.data)
 
    def get_data(self):
        return self.data   
    
    def set_data(self, data):
        self.data = data
        self.get_header().set_size(len(data))
        
    def get_header(self):
        return self.header
        
    
class FirstAsciiPfbSegment(PfbSegment):
    
    def process_encoding(self):
        fd = StringIO.StringIO(self.get_data())
        
        encoding = PfbEncodingDictionary(fd)
        
        fd.close()
        
        return encoding
    
    def save_encoding(self, encoding):
        fd = StringIO.StringIO(self.get_data())
        
        fd.truncate(encoding.get_start())
        fd.seek(encoding.get_start())
        
        for item in encoding.get_dict():
            fd.write(item.output())
            fd.write("\n")

        fd.write(encoding.get_rest())
        
        self.set_data(fd.getvalue())
        
        fd.close()
    
    
class BinaryPfbSegment(PfbSegment):
    
    def __init__(self, data, header):
        PfbSegment.__init__(self, data, header)
        
        self.PADDING_LENGTH = 4
        self.C1 = 52845
        self.C2 = 22719
        self.IV = 55665
        
        self.decrypted_data = None

    def bytes_to_hex(self, data):
        return ''.join(["%02x" % ord(b) for b in data]).strip()
    
    def to_hex(self):
        return self.bytes_to_hex(self.get_data())
    
    def decrypt_eexec(self):
        r = self.IV
        c1 = self.C1
        c2 = self.C2
        data = self.get_data()
        decrypted_data = []
        
        for i in range(self.get_size()):
            c = ord(data[i]) & 0xff
            t = (r >> 8) & 0xff
            decrypted_data.append(chr(t ^ c))
            r = (((r + c) * c1) + c2) % 65536
            
        self.decrypted_data = ''.join(decrypted_data[self.PADDING_LENGTH:])
        
        return True
    
    def encrypt_eexec(self):
        r = self.IV
        c1 = self.C1
        c2 = self.C2
        data = ''.join([chr(0), chr(0), chr(0), chr(0), self.get_decrypted_data()])
        encrypted_data = []
        
        for i in range(len(data)):
            p = ord(data[i]) & 0xff
            t = (r >> 8) & 0xff
            c = t ^ p
            encrypted_data.append(chr(c))
            r = (((r + c) * c1) + c2) % 65536
            
        self.set_data(''.join(encrypted_data))
        
        return True
    
    def get_decrypted_data(self):
        if self.decrypted_data is None:
            self.decrypt_eexec()
        
        return self.decrypted_data
    
    def set_decrypted_data(self, new_data):
        self.decrypted_data = new_data
        self.encrypt_eexec()
    
    def process_segment(self):
        fd = StringIO.StringIO(self.get_decrypted_data())
        
        subrs = PfbDictionary("/Subrs", PfbDictionaryItem, fd)
        charstrings = PfbDictionary("/CharStrings", PfbDictionaryItem, fd)
        
        fd.close()
        
        return [subrs, charstrings]
    
    def save_charstrings(self, charstrings):
        fd = StringIO.StringIO(self.get_decrypted_data())
        
        fd.truncate(charstrings.get_start())
        fd.seek(charstrings.get_start())
        
        fd.write(charstrings.get_header())
        fd.write("\n")
        
        for item in charstrings.get_dict():
            fd.write(item.output())
            fd.write("\n")

        fd.write(charstrings.get_rest())
        
        self.set_decrypted_data(fd.getvalue())
        
        fd.close()
        
    
class PfbDictionaryItem(object):
    def __init__(self, d_type):
        self.d_type = d_type
        self.name = ''
        self.proc = ''
        self.original_proc = ''
        self.proc_len = 0
        self.original_proc_len = 0
        self.mapped_to = None
    
    def parse(self, fd):        
        # trim whitespaces
        c = fd.read(1)
        
        while string.whitespace.find(c) > -1:
            c = fd.read(1)
        
        fd.seek(-1, 1)
        
        if self.d_type == "array":
            s = fd.read(4)
            if (s != 'dup '): return False
            
        self.name = self.read_until_space(fd)
        self.proc_len = int(self.read_until_space(fd))
        self.original_proc_len = self.proc_len
        
        rd = self.read_until_space(fd)
        proc = fd.read(self.proc_len)
        # remove space after proc
        fd.read(1)
        np = self.read_until_space(fd)
        
        self.proc = ''.join([rd, ' ', proc, ' ', np])
        self.original_proc = self.proc
        
    def read_until_space(self, fd):
        c = fd.read(1)
        command = []
        
        while string.whitespace.find(c) == -1:
            command.append(c)
            c = fd.read(1)
            
        return ''.join(command)         

    def output(self):
        pref = 'dup ' if self.d_type == "array" else ""
        return ''.join([pref, self.get_name(), ' ', `self.get_proc_len()`, ' ', self.get_proc()])
    
    def get_name(self):
        return self.name
    
    def set_name(self, name):
        self.name = name
        
    def copy_from(self, item):
        if isinstance(item, type(self)):
            self.set_name(item.get_name())
            self.proc = item.get_proc()
            self.proc_len = item.get_proc_len()
            self.original_proc = item.get_original_proc()
            self.original_proc_len = item.get_original_proc_len()
    
    def get_proc(self):
        return self.proc
    
    def get_proc_len(self):
        return self.proc_len
    
    def get_original_proc(self):
        return self.original_proc
    
    def get_original_proc_len(self):
        return self.original_proc_len
        
    def get_dict_type(self):
        return self.d_type
    
    def is_mapped(self):
        return not self.mapped_to is None
    
    def map_item(self, item):
        self.mapped_to = item
        self.proc = item.get_original_proc()
        self.proc_len = item.get_original_proc_len()
    
class PfbEncodingItem(object):
    def __init__(self, fd, d_type):
        self.fd = fd
        self.d_type = d_type
        self.char = -1
        self._char = None
        self.name = ''
        self.original_name = ''
        
        self.broken = not self.parse()
    
    def parse(self):
        fd = self.fd
        
        line = fd.readline()

        params = line.split('%')[0].split()
        
        try:
            d_index = params.index('dup')   
        except ValueError:
            d_index = -1   
            
        try:  
            p_index = params.index('put')
        except ValueError:
            p_index = -1
        
        if len(params) == 0 or d_index == -1 or p_index == -1: return False
        
        enc_data = ''.join(params[(d_index + 1):p_index]).split('/')
        
        self.char = int(enc_data[0])
        self.name = ''.join(['/', enc_data[1]])
        self.original_name = self.name
    
        return True
            

    def output(self):
        return ''.join(['dup ', `self.char`, ' ', self.name, ' put'])
    
    def get_name(self):
        return self.name
    
    def set_name(self, value):
        self.name = value
    
    def get_original_name(self):
        return self.original_name
    
    def get_char(self):
        return self.char
    
    def map_char(self, char):
        self._char = self.char
        self.char = char
    
    def mapped_to(self):
        return self.get_char() if self.is_mapped() else -1
    
    def is_mapped(self):
        return self._char is None
    
    def is_broken(self):
        return self.broken
    
class PfbDictionary(object):
       
    def __init__(self, dict_name, item_class, fd):       
        self.name = dict_name
        self.fd = fd
        self.dict = []
        self.count = 0
        self.start = 0
        self.rest = ""
        self.header = ""
        self.item_class = item_class

        self.process()
    
    def process(self):
        self.start = self.fd.tell()
        line = self.fd.readline()
        
        while line != "":
            pos = line.find(self.name)
            if pos > -1:
                self.header = line.split()
                line = line[(pos + len(self.name) + 1):]
                args = line.split()
                self.count = int(args[0])
                
                for _ in range(self.count):
                    item = self.item_class(args[1])
                    item.parse(self.fd)
                    self.dict.append(item)
                    
                p = self.fd.tell()
                self.rest = self.fd.read()
                self.fd.seek(p)
                break
            
            self.start = self.fd.tell()
            line = self.fd.readline()
    
    def length(self):
        return len(self.dict)
    
    def get_start(self):
        return self.start
    
    def get_rest(self):
        return self.rest     
    
    def get_dict(self):
        return self.dict  
    
    def set_dict(self, d):
        self.dict = d
        
    def set_item_count(self, c):
        pos = self.header.index(self.name)
        self.header[pos + 1] = `c`
        
    def refresh_item_count(self):
        self.set_item_count(len(self.get_dict()))
        
    def get_header(self):
        return ' '.join(self.header)
        
    def get_item_class(self):
        return self.item_class
    
    def get_item_by_name(self, name):
        for item in self.get_dict():
            if item.get_name() == name:
                return item
        return None
    
    def add_item(self, item):
        self.get_dict().append(item)
        self.refresh_item_count()
        return self
        
    def remove_item_by_name(self, name):
        item = self.get_item_by_name(name)
        if not item is None:
            self.get_dict().remove(item)
            
        self.refresh_item_count()
        return item
    
    
class PfbEncodingDictionary(PfbDictionary):
       
    def __init__(self, fd):
        PfbDictionary.__init__(self, "/Encoding", PfbEncodingItem, fd)
    
    def process(self):
        line = self.fd.readline()
        
        while line != "":
            pos = line.find(self.name)
            if pos > -1:
                self.start = self.fd.tell()
                line = line[(pos + len(self.name) + 1):]
                args = line.split()
                self.count = int(args[0])
                
                item = self.item_class(self.fd, args[1])
                while item.is_broken():
                    self.start = self.fd.tell()
                    item = self.item_class(self.fd, args[1])
                
                p = self.fd.tell()
                
                while not item.is_broken():
                    self.dict.append(item)
                    p = self.fd.tell()
                    item = self.item_class(self.fd, args[1])
                    
                self.fd.seek(p)
                self.rest = self.fd.read()
                self.fd.seek(p)
                break
            
            line = self.fd.readline() 
            
    def get_character_vector(self):
        return map(lambda x: x.get_char(), self.get_dict())
    
    def get_encoding_vector(self):
        return map(lambda x: x.mapped_to(), self.get_dict())
    
    def resolve_duplicates(self, charstrings):
        counter = {}
        pool = set()
        
        for item in self.get_dict():
            name = item.get_name()
            if name in counter:
                counter[name] += 1
            else:
                counter[name] = 1
            
            new_name = ''.join([name, 'V', `counter[name]`])
            item.set_name(new_name)
            ch_item = charstrings.get_item_by_name(item.get_original_name())
            
            if not ch_item is None:
                new_item = charstrings.get_item_class()(ch_item.get_dict_type())
                new_item.copy_from(ch_item)
                new_item.set_name(new_name)
                charstrings.add_item(new_item)
                pool.add(item.get_original_name())
        
        for item in pool:
            charstrings.remove_item_by_name(item)
    
    
    def get_character_names(self):
        return map(lambda x: x.get_name(), self.get_dict())
    
    def get_name_by_char(self, char):
        for item in self.get_dict():
            if item.get_char() == char:
                return item.get_name()
        return None
    
    def map_characters(self, charstrings, encoding_vector):
        vector = self.get_encoding_vector()
        if len(vector) != len(encoding_vector): return False

        for i in range(len(self.get_dict())):
            proc_item_name = self.get_name_by_char(vector[i])
            new_item_name = self.get_name_by_char(encoding_vector[i])
            
            proc_item = charstrings.get_item_by_name(proc_item_name)
            new_item = charstrings.get_item_by_name(new_item_name)
            
            new_item.map_item(proc_item)
    
    
class Type1FontFile(object):
    
    def __init__(self):
        self.firstAsciiSegmentHeader = None
        self.binarySegmentHeader = None
        self.secondAsciiSegmentHeader = None
        self.eofSegmentHeader = None
    
        self.fileData = None
        self.loaded = False
    
        self.firstAsciiSegment = None
        self.binarySegment = None
        self.secondAsciiSegment = None
    
    def read_pfb_file(self, input_file):       
        f = open(input_file, "rb")
        file_data = f.read()
        f.close()        
        return file_data
    
    def load_from_pfb_file(self, input_file):
        
        data = self.read_pfb_file(input_file) 
        
        if not self.read_segment_headers(data):
            return False
        
        self.firstAsciiSegment = FirstAsciiPfbSegment(data, self.firstAsciiSegmentHeader)
        self.binarySegment = BinaryPfbSegment(data, self.binarySegmentHeader)
        self.secondAsciiSegment = PfbSegment(data, self.secondAsciiSegmentHeader)
    
        self.set_loaded()
        
        return True
    
    def save_as_pfa_file(self, output_file):
        if not self.is_loaded():
            return False
        
        f = open(output_file, "wb")
        
        f.write(self.firstAsciiSegment.get_data())
        f.write(self.binarySegment.to_hex())
        f.write(self.secondAsciiSegment.get_data())
        
        f.close()
        
        return True
    
    def save_as_pfb_file(self, output_file):
        if not self.is_loaded():
            return False
        
        f = open(output_file, "wb")
        
        f.write(self.firstAsciiSegmentHeader.output())
        f.write(self.firstAsciiSegment.get_data())
        f.write(self.binarySegmentHeader.output())
        f.write(self.binarySegment.get_data())
        f.write(self.secondAsciiSegmentHeader.output())
        f.write(self.secondAsciiSegment.get_data())
        f.write(self.eofSegmentHeader.output())
        
        f.close()
        
        return True
    
    def save_as_raw_file(self, output_file):
        if not self.is_loaded():
            return False
        
        f = open(output_file, "wb")
        
        f.write(self.firstAsciiSegment.get_data())
        f.write(self.binarySegment.get_decrypted_data())
        f.write(self.secondAsciiSegment.get_data())
        
        f.close()
        
        return True
    
    def read_segment_headers(self, data):
        offset = 0
        
        firstSegmentHeader = PfbSegmentHeader(data, offset)
        offset += firstSegmentHeader.get_size_with_header()
        
        secondSegmentHeader = PfbSegmentHeader(data, offset)
        offset += secondSegmentHeader.get_size_with_header()
        
        thirdSegmentHeader = PfbSegmentHeader(data, offset)
        offset += thirdSegmentHeader.get_size_with_header()
        
        lastSegmentHeader = PfbSegmentHeader(data, offset)
        offset += lastSegmentHeader.get_size_with_header()
        
        if (len(data) != offset):
            return False
        
        self.firstAsciiSegmentHeader = firstSegmentHeader
        self.binarySegmentHeader = secondSegmentHeader
        self.secondAsciiSegmentHeader = thirdSegmentHeader
        self.eofSegmentHeader = lastSegmentHeader
        
        return True
    
    def set_loaded(self):
        self.loaded = True
    
    def is_loaded(self):
        return self.loaded
    
    