#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#include "debug.h"
#include "node.h"
#include "message.h"
#include "transaction.h"
#include "p2pcb.h"
#include "config.h"

#include "tpl.h"

pthread_t listener;

static pthread_mutex_t mutex_message;
static pthread_cond_t cond_message;

static int me_running, shutdown;
static int my_id;
static int nodes_count;
static int running[MAX_NODES];
static int received_shutdown_message;
static transaction_list *transactions;
static int node_transaction_counter;
static int transaction_all, transaction_processed;
static message *message_first, *message_last;

static int node_check_someone_running(void);
static void node_try_shutdown(void);

static int node_match(int sender, int tag, void *message)
{
    return 1;
}

static void node_init(void)
{
    int i;

    me_running = TRUE;
    for (i=1; i<=nodes_count; i++)
    {
        running[i] = TRUE;
    }
    node_check_someone_running();
    transactions = transaction_list_create();
    node_transaction_counter = 0;
    transaction_all = -1;
    transaction_processed = 0;
    received_shutdown_message = FALSE;

    message_first = NULL;
    message_last = message_first;

    pthread_mutex_init(&mutex_message, NULL);
    pthread_cond_init(&cond_message, NULL);
}

int node_get_id(void)
{
    return my_id;
}

transaction_list* node_get_transaction_list(void)
{
    return transactions;
}

void node_print_message_list(void)
{
    message *temp;
    temp = message_first;

    while (temp != NULL)
    {
        debug_print(DEBUG_MESSAGE_LIST, "ML%d: (S)%d (T)%d\n", my_id, temp->sender, temp->tag);
        temp = temp->next;
    }
}

static message node_get_message(void)
{
    message m, *temp;
    if (message_first != NULL)
    {
        m = *message_first;
        temp = message_first;
        message_first = message_first->next;
        free(temp);
    }
    else
    {
        m.tag = MESSAGE_NULL;
        m.sender = NODE_NULL;
    }
    return m;
}

static int node_set_message(int sender, int tag, void *msg)
{
    message *message_new = (message*)malloc(sizeof(message));
    if (message_new == NULL)
    {
        /*node_print_message_list();*/
        return 0;
    }
    else
    {
        message_new->sender = sender;
        message_new->tag = tag;
        message_new->message = msg;
        message_new->next = NULL;
        if (message_first == NULL)
        {
            message_first = message_new;
            message_last = message_new;
        }
        else
        {
            message_last->next = message_new;
            message_last = message_last->next;
        }
        /*node_print_message_list();*/
        return 1;
    }
}

void node_send_all(message m)
{
    void *message;
    int i;

    for (i=1; i<=nodes_count; i++)
    {
        if (i!=my_id)
        {
            tpl_create(&message);
            tpl_send(&i, 1, m.tag, message);
        }
    }
}

void node_send_all_simple_message(int tag)
{
    message m;
    m.sender = my_id;
    m.tag = tag;
    m.message = NULL;
    m.next = NULL;
    node_send_all(m);
}

void node_send_all_transaction_message(int tag, long transaction_id)
{
    message m;
    void *message;
    int i;

    m.sender = my_id;
    m.tag = tag;
    m.message = NULL;
    m.next = NULL;

    for (i=1; i<=nodes_count; i++)
    {
        if (i!=my_id)
        {
            tpl_create(&message);
            tpl_pkbyte(message, &transaction_id, sizeof(long));
            tpl_send(&i, 1, m.tag, message);
        }
    }

}

long node_get_transaction_id(void)
{
    long id_transaction = node_transaction_counter*100+my_id;
    node_transaction_counter++;
    return id_transaction;
}

static int node_check_someone_running(void)
{
    int i;
    int result = FALSE;

    for (i=1; i<=nodes_count; i++)
    {
        if (running[i])
        {
            result = TRUE;
        }
    }
    return result;
}

static void node_try_shutdown(void)
{

    if ( transaction_processed == transaction_all )
    {
        debug_print(DEBUG_TRANSACTION_COMPLETE, "Server %d: All transactions complete\n", my_id);
        running[my_id] = FALSE;
        /* Send others shutdown message */
        node_send_all_simple_message(MESSAGE_SHUTDOWN);

        shutdown = TRUE;
    }
    else
    {
       debug_print(DEBUG_TRANSACTION_COMPLETE, "Server %d: Processed: %d from: %d transactions\n", my_id, transaction_processed, transaction_all);
    }
}

static void *node_listen(void *arg)
{
    void *msg, *msg_content;
    int tag, sender;

    me_running = TRUE;
    while (me_running)
    {
        tpl_recv(node_match, &sender, &tag, &msg);
        tpl_upkbyte(msg, &msg_content, sizeof(long));
        tpl_destroy(msg);
        if (sender >= 0)
        {
            debug_print(DEBUG_MESSAGE, "DB Server %d recieved message %d %d\n", my_id, sender, tag);

            switch (tag)
            {
                case CMESSAGE_SHUTDOWN :
                    received_shutdown_message = TRUE;
                    running[my_id] = FALSE;
                    break;
                case MESSAGE_SHUTDOWN :
                    running[sender] = FALSE;
                    break;

                default:
                    break;
            }

            debug_print(DEBUG_MUTEX, "Listener locked mutex %d\n", my_id);
            pthread_mutex_lock(&mutex_message);
            /* notify compute thread */
            node_set_message(sender, tag, msg_content);

            debug_print(DEBUG_SLEEP_WAKE, "DB Server %d was waked up\n", my_id);
            pthread_cond_signal(&cond_message);
            pthread_mutex_unlock(&mutex_message);
            debug_print(DEBUG_MUTEX, "Listener released mutex %d\n", my_id);

            me_running = node_check_someone_running();
        }
    }
    debug_print(DEBUG_START_STOP, "Listener %d exits\n", my_id);

    return NULL;
}

void node_transaction_end(long transaction_id)
{
    transaction_processed++;
    node_try_shutdown();
}

void node_start_server(int id, int nodes)
{
    message m;
    shutdown = FALSE;

    my_id = id;
    nodes_count = nodes;
    node_init();

    debug_print(DEBUG_START_STOP, "DB Server %d started\n", my_id);

    debug_print(DEBUG_START_STOP, "Listener %d created\n", my_id);
    pthread_create(&listener, NULL, node_listen, NULL);

    while(!shutdown)
    {
        m = node_get_message();
        debug_print(DEBUG_MESSAGE, "Compute %d recieved message from %d with tag %d\n", my_id, m.sender, m.tag);
        switch (m.tag)
        {
            case MESSAGE_NULL :
                pthread_mutex_lock(&mutex_message);
                debug_print(DEBUG_SLEEP_WAKE, "DB Server %d is sleeping\n", my_id);
                pthread_cond_wait(&cond_message, &mutex_message);
                pthread_mutex_unlock(&mutex_message);
                break;
            case CMESSAGE_SHUTDOWN :
                debug_print(DEBUG_MESSAGE, "Compute %d recieved shutdown message\n", my_id);
                transaction_all = (int)m.message;
                node_try_shutdown();
                break;
            default :
                if (p2pcb_is_protocol_message(m.tag))
                {
                    p2pcb_handle_message(m);
                }
                break;
        }
    }

    node_stop_server();
}

void node_stop_server(void)
{
    void *message;
    int receiver;

    debug_print(DEBUG_START_STOP, "DB Server %d stopped\n", my_id);

    pthread_join(listener, NULL);
    receiver = ID_CONTROLLER;
    tpl_create(&message);
    tpl_send(&receiver, 1, CMESSAGE_SHUTDOWN_ACKNOWLEDGMENT, message);
}

