#include <stdio.h>
#include <stdlib.h>
#include <time.h>

#include "p2pcb.h"
#include "message.h"
#include "node.h"
#include "debug.h"
#include "config.h"

#include "tpl.h"

short votes[MAX_TRANSACTIONS][MAX_NODES];

static int p2pcb_get_vote_message_tag(short vote);
static void p2pcb_check_votes(long transaction_id);
static void p2pcb_transaction_commit(long transaction_id);
static void p2pcb_transaction_abort(long transaction_id);
static short p2pcb_vote(long transaction_id);
static void p2pcb_clear_votes(long transaction_id);
static void p2pcb_save_vote(long transaction_id, int node_id, short vote);
static void p2pcb_debug_votes(long transaction_id);

int p2pcb_is_protocol_message(int tag)
{
    if (MESSAGE_2PC_VOTE_REQ <= tag && tag <= MESSAGE_2PC_ABORT)
    {
        return TRUE;
    }
    if (tag == CMESSAGE_TRANSACTION_BEGIN)
    {
        return TRUE;
    }
    return FALSE;
}

int p2pcb_get_message_round(int tag)
{
    switch (tag)
    {
        case MESSAGE_2PC_VOTE_REQ:
            return 1;
        case MESSAGE_2PC_VOTE_YES:
        case MESSAGE_2PC_VOTE_NO:
            return 2;
        case MESSAGE_2PC_ABORT:
        case MESSAGE_2PC_COMMIT:
            return 3;
        default :
            return 0;
    }
}

void p2pcb_handle_message(message m)
{
    transaction_list *list;
    transaction *t = NULL;
    long transaction_id;
    int node_id;

    node_id = node_get_id();
    list = node_get_transaction_list();

    switch (m.tag)
    {
        case CMESSAGE_TRANSACTION_BEGIN:
            transaction_id = node_get_transaction_id();
            t = transaction_begin(transaction_id, node_id, node_id);
            transaction_list_add(list, t);
            /* transaction_list_print(list); */
            /*p2pcb_clear_votes(transaction_id);*/
            p2pcb_save_vote(transaction_id, node_id, VOTE_YES);
            debug_print(DEBUG_COMMIT_ABORT ,"Server %d: BEGIN TRANSACTION %d\n", node_id, transaction_id);
            node_send_all_transaction_message(MESSAGE_2PC_VOTE_REQ, transaction_id);
            return;
        case MESSAGE_2PC_VOTE_REQ:
            transaction_id = (long)(m.message);
            message_print(&m);
            debug_print(DEBUG_TEMP, "Message belongs to transcation %d\n", transaction_id);
            t = transaction_begin(transaction_id, node_id, m.sender);
            transaction_list_add(list, t);
            /*p2pcb_clear_votes(transaction_id);*/
            p2pcb_save_vote(transaction_id, m.sender, VOTE_YES);
            p2pcb_debug_votes(transaction_id);
            node_send_all_transaction_message(p2pcb_get_vote_message_tag(p2pcb_vote(transaction_id)), transaction_id);
            return;
        case MESSAGE_2PC_VOTE_YES:
            transaction_id = (long)(m.message);
            message_print(&m);
            /*debug_print(DEBUG_TEMP, "Server %d voted YES for transaction %d\n", m.sender, transaction_id);*/
            p2pcb_save_vote(transaction_id, m.sender, VOTE_YES);
            p2pcb_debug_votes(transaction_id);
            return;
        case MESSAGE_2PC_VOTE_NO:
            transaction_id = (long)(m.message);
            message_print(&m);
            /*debug_print(DEBUG_TEMP, "Server %d voted NO for transaction %d\n", m.sender, transaction_id);*/
            p2pcb_save_vote(transaction_id, m.sender, VOTE_NO);
            p2pcb_debug_votes(transaction_id);
            return;
        case MESSAGE_2PC_ABORT:
        case MESSAGE_2PC_COMMIT:
            return;
        default :
            return;
    }
}

static int p2pcb_get_vote_message_tag(short vote)
{
    switch (vote)
    {
        case VOTE_YES:
            return MESSAGE_2PC_VOTE_YES;
        case VOTE_NO:
            return MESSAGE_2PC_VOTE_NO;
        default :
            return MESSAGE_NULL;
    }
}


static void p2pcb_clear_votes(long transaction_id)
{
    int j;

    for ( j=0; j<MAX_NODES; j++)
    {
        votes[transaction_id][j] = VOTE_NONE;
    }
}

static void p2pcb_debug_votes(long transaction_id)
{
    int j;
    int my_id, nodescount;
    my_id = node_get_id();
    nodescount = config_get_nodes_count();

    debug_print(DEBUG_VOTE_DATA, "Node %d votes for %d: ", my_id, transaction_id);
    for ( j=1; j<=nodescount; j++)
    {
        debug_print(DEBUG_VOTE_DATA, "%d:%d, ", j, votes[transaction_id][j]);
    }
    debug_print(DEBUG_VOTE_DATA, "\n");
}

static void p2pcb_save_vote(long transaction_id, int node_id, short vote)
{
    votes[transaction_id][node_id] = vote;
    p2pcb_check_votes(transaction_id);
}

static void p2pcb_check_votes(long transaction_id)
{
    int i, commit, abort, nodescount;
    commit = TRUE;
    abort = FALSE;
    nodescount = config_get_nodes_count();

    for ( i=1; i<=nodescount; i++)
    {
        if (votes[transaction_id][i]!=VOTE_YES)
        {
            commit = FALSE;
        }
        if (votes[transaction_id][i]==VOTE_NO)
        {
            abort = TRUE;
        }
    }

    if (commit)
    {
        p2pcb_transaction_commit(transaction_id);
    }

    if (abort)
    {
        p2pcb_transaction_abort(transaction_id);
   }
}

static void p2pcb_transaction_commit(long transaction_id)
{
    transaction_list *list;
    transaction *t;
    int my_id;

    my_id = node_get_id();
    list = node_get_transaction_list();
    t = transaction_list_find(list, transaction_id);
    t->status = TRANSACTION_COMMITED;
    debug_print(DEBUG_COMMIT_ABORT ,"Server %d: COMMIT TRANSACTION %d\n", my_id, transaction_id);
    node_transaction_end(transaction_id);
}

static void p2pcb_transaction_abort(long transaction_id)
{
    transaction_list *list;
    transaction *t;
    int my_id;

    my_id = node_get_id();
    list = node_get_transaction_list();
    t = transaction_list_find(list, transaction_id);
    if (t->status == TRANSACTION_CREATED)
    {
        t->status = TRANSACTION_ABORTED;
        debug_print(DEBUG_COMMIT_ABORT ,"Server %d: ABORT TRANSACTION %d\n", my_id, transaction_id);
        node_transaction_end(transaction_id);
    }
}


static short p2pcb_vote(long transaction_id)
{
    short vote;
    int nodescount;
    int my_id = node_get_id();

    nodescount = config_get_nodes_count();

    srand(time(NULL));

    if (rand() % (2*nodescount) != 0)
    {
        vote = VOTE_YES;
    }
    else
    {
        vote = VOTE_NO;
    }
    p2pcb_save_vote(transaction_id, my_id, vote);

    if (vote == VOTE_YES)
    {
        debug_print(DEBUG_VOTES, "Server %d: YES TRANSACTION %d\n", my_id, transaction_id);
    }
    else
    {
        debug_print(DEBUG_VOTES, "Server %d: NO TRANSACTION %d\n", my_id, transaction_id);
    }

    return vote;
}
