#include "CControlFractal.h"
#include "CMandelbrot.h"
#include "CJulia.h"
#include "CAnimObject.h"
#include "CAnimationPath.h"
#include "stdio.h"
#include "AFractalDlg.h"
#include "CTimer.h"

LRESULT CALLBACK CControlFractal::CustWndProcWrapper(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
	CControlFractal *pThis = (CControlFractal*)GetProp(hwnd,"ClassPointer");
	if(pThis) 
		return pThis->CustWndProc(hwnd,msg,wParam,lParam);
	return DefWindowProc(hwnd,msg,wParam,lParam);
}

LRESULT CControlFractal::CustWndProc(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
	PAINTSTRUCT ps; 
	HDC hdc; 

	switch(msg)
	{
	case WM_PAINT:
		if(wParam == 0)
			hdc = BeginPaint(hwnd, &ps);
		else
			hdc = (HDC)wParam;

		OnPaint(hdc);

		if(wParam == 0)
			EndPaint(hwnd, &ps);
	break;
	case WM_RBUTTONDOWN:
		OnRButtonDown(wParam, lParam);
	break;
	case WM_LBUTTONDOWN:
		OnLButtonDown(wParam, lParam);
	break;
	case WM_LBUTTONUP:
		OnLButtonUp(wParam, lParam);
	break;
	case WM_MOUSEMOVE:
		OnMouseMove(wParam, lParam);
	break;
	default:
        break;
    }

    return DefWindowProc(hwnd, msg, wParam, lParam);
}

CControlFractal::CControlFractal(HWND hwndParent, CMandelbrot *def)
: CCustomControl(hwndParent),
  m_path(NULL),
  m_fractalType(FRACTAL_TYPE_MAND),
  m_drag_x_e(0),m_drag_x_s(0),m_drag_y_e(0),m_drag_y_s(0),
  m_dragging(false)
{
	m_className = _T("ControlFractal");
	m_width = 640;
	m_height = 512;

	InitCustomControl();
	CreateCustomControl();

	m_fractalObject = new CMandelbrot(hwndParent);
	
	m_bitmapInfo.biSize = sizeof(BITMAPINFOHEADER);
	m_bitmapInfo.biWidth = m_width;
	m_bitmapInfo.biHeight = m_height;
	m_bitmapInfo.biPlanes = 1;
	m_bitmapInfo.biBitCount = 24;
	m_bitmapInfo.biCompression = BI_RGB;
	m_bitmapInfo.biSizeImage = 0;
	m_bitmapInfo.biXPelsPerMeter = 0;
	m_bitmapInfo.biYPelsPerMeter = 0;
	m_bitmapInfo.biClrUsed = 0;
	m_bitmapInfo.biClrImportant = 0;

	HDC hdc = GetWindowDC(NULL);

	m_hBitmap = CreateDIBSection(hdc, (LPBITMAPINFO)&m_bitmapInfo,DIB_RGB_COLORS,(LPVOID *)&m_image,NULL,0);

	m_fractalObject->CopyToLocation(m_image);

	ReleaseDC(NULL,hdc);
}

CControlFractal::~CControlFractal()
{
	DeleteObject(m_hBitmap);
}

void CControlFractal::InitCustomControl()
{
    WNDCLASSEX wc;
    
    wc.cbSize         = sizeof(wc);
    wc.lpszClassName  = m_className;
    wc.hInstance      = GetModuleHandle(0);
    wc.lpfnWndProc    = CustWndProcWrapper;
    wc.hCursor        = LoadCursor (NULL, IDC_ARROW);
    wc.hIcon          = 0;
    wc.lpszMenuName   = 0;
    wc.hbrBackground  = (HBRUSH)GetSysColorBrush(COLOR_BTNFACE);
    wc.style          = 0;
    wc.cbClsExtra     = 0;
    wc.cbWndExtra     = 0;
    wc.hIconSm        = 0;

    RegisterClassEx(&wc);
}


void CControlFractal::CreateCustomControl()
{
	m_hwndCtrl = CreateWindowEx(
                 0, // give it a standard border
                 m_className,
                 _T("Fractal control"),
                 WS_VISIBLE | WS_CHILD,
                 0, 0, m_width, m_height,
                 m_hwndParent,
                 NULL, GetModuleHandle(0), NULL
               );

	SetProp(m_hwndCtrl,"ClassPointer",(HANDLE)this);
}

void CControlFractal::OnMouseMove(WPARAM wParam, LPARAM lParam)
{
	if(m_dragging)
	{
		m_drag_x_e = LOWORD(lParam);
		m_drag_y_e = HIWORD(lParam);
		InvalidateRect(m_hwndCtrl, NULL, false);
		UpdateWindow(m_hwndCtrl);
	}
}

void CControlFractal::OnLButtonDown(WPARAM wParam, LPARAM lParam) 
{
	m_drag_x_s = LOWORD(lParam);
	m_drag_y_s = HIWORD(lParam);
	m_drag_x_e = m_drag_x_s;
	m_drag_y_e = m_drag_y_s;
	m_dragging = true;

	if(m_fractalObject->IsDone())
		InvalidateRect(m_hwndCtrl, NULL, false);
}

void CControlFractal::OnLButtonUp(WPARAM wParam, LPARAM lParam) 
{
	m_dragging = false;
	if(m_fractalObject->IsDone() && m_fractalObject->ZoomIn(m_drag_x_s, m_drag_y_s, m_drag_x_e, m_drag_y_e))
	{
		timer->Start();
		m_fractalObject->Compute();
	
		if(m_path && m_path->m_active)
			m_path->Add(m_fractalObject->m_zoomDepth, m_fractalObject->m_xMin, m_fractalObject->m_xMax, m_fractalObject->m_yMin, m_fractalObject->m_yMax, m_fractalObject->m_exp, m_fractalObject->m_iterationCount);
	}
}

void CControlFractal::OnRButtonDown(WPARAM wParam, LPARAM lParam) 
{
	if(m_fractalObject->ZoomOut())
	{
		timer->Start();
		m_fractalObject->Compute();
	}

	if(m_path && m_path->m_active)
		if(m_path->GetRecordFullPath())
			m_path->Add(m_fractalObject->m_zoomDepth, m_fractalObject->m_xMin, m_fractalObject->m_xMax, m_fractalObject->m_yMin, m_fractalObject->m_yMax, m_fractalObject->m_exp, m_fractalObject->m_iterationCount);
		else
			m_path->RemoveLast();

	InvalidateRect(m_hwndCtrl, NULL, false);
}

void CControlFractal::OnKeyPress(WPARAM wParam){
	switch((int)wParam){
		case VK_SUBTRACT:
			DecreaseIterationCount();
		break;
		case VK_ADD:
			IncreaseIterationCount();
		break;
		case VK_BACK:
			m_fractalObject->Reset();
			timer->Start();
			m_fractalObject->Compute();
			InvalidateRect(m_hwndCtrl, NULL, false);
		break;
		case VK_ESCAPE:
//			m_fractalObject->Abort();
		break;
	}
}

void CControlFractal::IncreaseIterationCount(){
	m_fractalObject->SetIterationCount(int(m_fractalObject->GetIterationCount()*1.5));
	timer->Start();
	m_fractalObject->Compute();
	InvalidateRect(m_hwndCtrl, NULL, false);
}

void CControlFractal::DecreaseIterationCount(){
	m_fractalObject->SetIterationCount(int(m_fractalObject->GetIterationCount()/1.5));
	timer->Start();
	m_fractalObject->Compute();
	InvalidateRect(m_hwndCtrl, NULL, false);
}

// Set Mandelbrot Set as current displayed fractal
void CControlFractal::SetCurrentMandelbrot(){
	delete m_fractalObject;
	m_fractalObject = new CMandelbrot(m_hwndParent);
	m_fractalObject->SetMemoryLocation(m_image);
	timer->Start();
	m_fractalObject->Compute(0,1);
	m_fractalType = FRACTAL_TYPE_MAND;
	InvalidateRect(m_hwndCtrl, NULL, false);
}

// Set Julia Set as current displayed fractal
// void CControlFractal::SetCurrentJulia(){
// 	delete m_fractalObject;
// 	m_fractalObject = new CJulia(m_hwndParent);
// 	m_fractalObject->SetMemoryLocation(m_image);
// 	m_fractalObject->Compute();
// 	m_fractalType = FRACTAL_TYPE_JULIA;
// 	InvalidateRect(m_hwndCtrl, NULL, false);
// }

void CControlFractal::SetCurrentJulia(double x, double y){
	delete m_fractalObject;
	m_fractalObject = new CJulia(m_hwndParent);
	((CJulia*)m_fractalObject)->SetSeed(x,y);
	m_fractalObject->SetMemoryLocation(m_image);
	timer->Start();
	m_fractalObject->Compute(0,1);
	m_fractalType = FRACTAL_TYPE_JULIA;
	InvalidateRect(m_hwndCtrl, NULL, false);
}

void CControlFractal::SaveToBmp(const char *fileName, unsigned char *texture, int width, int height){
	HANDLE file;
	BITMAPFILEHEADER fileHeader; 
	BITMAPINFOHEADER infoHeader;
	RGBTRIPLE rgb;
	DWORD writeSize;
	
	file = CreateFile(fileName, GENERIC_WRITE, FILE_SHARE_WRITE, (LPSECURITY_ATTRIBUTES)NULL, CREATE_ALWAYS, 0, (HANDLE)NULL);
	
	if(file == NULL){
		MessageBox(NULL,"Unable to load specified file","AFractal error",MB_OK|MB_ICONSTOP);
		return;
	}
	fileHeader.bfType = 19778;
	fileHeader.bfSize = 54 + 3*width*height;
	fileHeader.bfReserved1 = 0;
	fileHeader.bfReserved2 = 0;
	fileHeader.bfOffBits = 54;

	infoHeader.biSize = sizeof(BITMAPINFOHEADER);
    infoHeader.biWidth = width;
    infoHeader.biHeight = height;
    infoHeader.biPlanes = 1;
    infoHeader.biBitCount = 24;
    infoHeader.biCompression = BI_RGB;
    infoHeader.biSizeImage = width*height;
    infoHeader.biClrUsed = 0;
    infoHeader.biClrImportant = 0;

	WriteFile(file, &fileHeader, sizeof(fileHeader), &writeSize, NULL);
	WriteFile(file, &infoHeader, sizeof(infoHeader), &writeSize, NULL);

	for(int i = 0; i< 3*width*height; i+=3)
	{
		rgb.rgbtRed = texture[i+2];
		rgb.rgbtGreen = texture[i+1];
		rgb.rgbtBlue = texture[i+0];
		
		WriteFile(file, &rgb, sizeof(rgb), &writeSize, NULL);
	}
		
	CloseHandle(file);
}

void CControlFractal::DrawSelection(HDC hdc)
{
	if(!m_dragging)
		return;

	HPEN hPen, hpOld;
	hPen = CreatePen(PS_SOLID, 1, 0x00000000);
	hpOld = (HPEN)SelectObject(hdc, hPen);

	
	MoveToEx(hdc, m_drag_x_s, m_drag_y_s, NULL);
	LineTo(hdc, m_drag_x_e, m_drag_y_s);
	LineTo(hdc, m_drag_x_e, m_drag_y_e);
	LineTo(hdc, m_drag_x_s, m_drag_y_e);
	LineTo(hdc, m_drag_x_s, m_drag_y_s);

	SelectObject(hdc, hpOld);
	DeleteObject(hPen);  
}

void CControlFractal::SetData(CMandelbrot *data)
{
	m_fractalObject->SetTexture(data->GetImage(), data->GetSizeX()*data->GetSizeY()*sizeof(unsigned char)*3);
}