#include "CControlMandNURBS.h"
#include "NURBS.h"
#include "CMandelbrot.h"
#include "CJulia.h"
#include "stdio.h"

#define WM_UPDATEPOSITION	WM_APP + 0x162
#define WM_UPDATERANGE		WM_APP + 0x164

LRESULT CALLBACK CControlMandNURBS::MandWndProcWrapper(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
	CControlMandNURBS *pThis = (CControlMandNURBS*)GetProp(hwnd,"ClassPointer");
	if(pThis) 
		return pThis->CustWndProc(hwnd,msg,wParam,lParam);
	return DefWindowProc(hwnd,msg,wParam,lParam);
}

LRESULT CControlMandNURBS::CustWndProc(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
	PAINTSTRUCT ps; 
	HDC hdc; 

    switch(msg)
    {
	case WM_PAINT:
		if(wParam == 0)
			hdc = BeginPaint(hwnd, &ps);
		else
			hdc = (HDC)wParam;

		OnPaint(hdc);

		if(wParam == 0)
			EndPaint(hwnd, &ps);
	break;
	case WM_LBUTTONDOWN:
		OnLButtonDown(LOWORD(lParam), HIWORD(lParam));
	break;
	case WM_LBUTTONUP:
		OnLButtonUp(LOWORD(lParam), HIWORD(lParam));
	break;
	case WM_RBUTTONDOWN:
		OnRButtonDown(LOWORD(lParam), HIWORD(lParam));
	break;
	case WM_RBUTTONUP:
		OnRButtonUp(LOWORD(lParam), HIWORD(lParam));
	break;
	case WM_MOUSEMOVE:
		OnMouseMove(LOWORD(lParam), HIWORD(lParam));
	break;
	default:
        break;
    }

    return DefWindowProc(hwnd, msg, wParam, lParam);
}

CControlMandNURBS::CControlMandNURBS(HWND hwndParent, HWND hwndJulia, CFractal *julia, CNURBS *nurbs, CMandelbrot *def) 
: CCustomControl(hwndParent),
  m_juliaObject(julia),
  m_hwndJulia(hwndJulia),
  m_mouseX(197), m_mouseY(67),
  m_pointX(-0.4), m_pointY(0.6),
  m_positionX(m_pointX), m_positionY(m_pointY),
  m_dragging(false),
  m_draggedVertex(NULL),
  m_actualFrame(0)
{
	m_className = _T("ControlMandNURBS");
	m_width = 640;
	m_height = 512;

	if(nurbs == NULL)
	{
		m_NURBS = new CNURBS();
		m_NURBS->SetStepFrames(24);
		m_NURBS->AddPoint(-0.4, 0.6);
		m_NURBS->AddPoint(-0.2, 0.8);
	}
	else
	{
		m_NURBS = nurbs;
	}

	InitCustomControl();
	CreateCustomControl();

	m_fractalObject = new CMandelbrot();
	m_fractalObject->SetSize(m_width, m_height);

	m_bitmapInfo.biSize = sizeof(BITMAPINFOHEADER);
	m_bitmapInfo.biWidth = m_width;
	m_bitmapInfo.biHeight = m_height;
	m_bitmapInfo.biPlanes = 1;
	m_bitmapInfo.biBitCount = 24;
	m_bitmapInfo.biCompression = BI_RGB;
	m_bitmapInfo.biSizeImage = 0;
	m_bitmapInfo.biXPelsPerMeter = 0;
	m_bitmapInfo.biYPelsPerMeter = 0;
	m_bitmapInfo.biClrUsed = 0;
	m_bitmapInfo.biClrImportant = 0;

	HDC hdc = GetWindowDC(NULL);

	m_hBitmap = CreateDIBSection(hdc, (LPBITMAPINFO)&m_bitmapInfo,DIB_RGB_COLORS,(LPVOID *)&m_image,NULL,0);

	m_fractalObject->SetMemoryLocation(m_image);
	m_fractalObject->SetIterationCount(128);
	if(def != NULL)
		m_fractalObject->SetTexture(def->GetImage(), def->GetSizeX()*def->GetSizeY()*sizeof(unsigned char)*3);

	ReleaseDC(NULL,hdc);
}

CControlMandNURBS::~CControlMandNURBS()
{
	delete m_fractalObject;
}

void CControlMandNURBS::InitCustomControl()
{
    WNDCLASSEX wc;
    
    wc.cbSize         = sizeof(wc);
    wc.lpszClassName  = m_className;
    wc.hInstance      = GetModuleHandle(0);
    wc.lpfnWndProc    = MandWndProcWrapper;
    wc.hCursor        = LoadCursor (NULL, IDC_ARROW);
    wc.hIcon          = 0;
    wc.lpszMenuName   = 0;
    wc.hbrBackground  = (HBRUSH)GetSysColorBrush(COLOR_BTNFACE);
    wc.style          = 0;
    wc.cbClsExtra     = 0;
    wc.cbWndExtra     = 0;
    wc.hIconSm        = 0;

    RegisterClassEx(&wc);
}


void CControlMandNURBS::CreateCustomControl()
{
	m_hwndCtrl = CreateWindowEx(
                 WS_EX_CLIENTEDGE, 
                 m_className,
                 _T("control MandNURBS"),
                 WS_VISIBLE | WS_CHILD,
                 5, 5, m_width + 4, m_height + 4,
                 m_hwndParent,
                 NULL, GetModuleHandle(0), NULL
               );

	SetProp(m_hwndCtrl,"ClassPointer",(HANDLE)this);
}

void CControlMandNURBS::OnLButtonDown(int x, int y) 
{
	float tmpX = 3.0f*(float)x/(float)m_width-2.25f;
	float tmpY = -2.5f*(float)y/(float)m_height+1.25f;

	CVertex *tmpV = m_NURBS->GetNearestPoint(tmpX, tmpY);

	if(tmpV == NULL)
	{
		m_NURBS->AddPoint(tmpX, tmpY);
		PostMessage(m_hwndParent, WM_UPDATERANGE, (m_NURBS->GetVertexCount()-1)*m_NURBS->GetStepFrames(), NULL);
		InvalidateRect(m_hwndJulia, NULL, false);
		UpdateWindow(m_hwndJulia);
		InvalidateRect(m_hwndCtrl, NULL, false);
		UpdateWindow(m_hwndCtrl);
	}
	else
	{
		m_draggedVertex = tmpV;
		m_draggedVertex->SetSelected(true);
		m_dragging = true;
		InvalidateRect(m_hwndCtrl, NULL, false);
		UpdateWindow(m_hwndCtrl);
	}
}

void CControlMandNURBS::OnLButtonUp(int x, int y) 
{
	m_dragging = false;
	if(m_draggedVertex != NULL)
		m_draggedVertex->SetSelected(false);
	m_draggedVertex = NULL;
}

void CControlMandNURBS::OnRButtonDown(int x, int y) 
{
	float tmpX = 3.0f*(float)x/(float)m_width-2.25f;
	float tmpY = -2.5f*(float)y/(float)m_height+1.25f;

	CVertex *tmpV = m_NURBS->GetNearestPoint(tmpX, tmpY);

	if(tmpV != NULL)
	{
		m_toDeleteVertex = tmpV;
	}
}

void CControlMandNURBS::OnRButtonUp(int x, int y) 
{
	float tmpX = 3.0f*(float)x/(float)m_width-2.25f;
	float tmpY = -2.5f*(float)y/(float)m_height+1.25f;

	CVertex *tmpV = m_NURBS->GetNearestPoint(tmpX, tmpY);
	if(tmpV == m_toDeleteVertex)
	{
		m_NURBS->DeleteVertex(m_toDeleteVertex);
		InvalidateRect(m_hwndCtrl, NULL, false);
		UpdateWindow(m_hwndCtrl);
		PostMessage(m_hwndParent, WM_UPDATERANGE, (m_NURBS->GetVertexCount()-1)*m_NURBS->GetStepFrames(), NULL);
	}
}

void CControlMandNURBS::SetPointPossition(float x, float y)
{
	m_pointX = x;
	m_pointY = y;

	((CJulia *)m_juliaObject)->SetSeed(m_pointX, m_pointY);
	m_juliaObject->Compute(0,1);
	InvalidateRect(m_hwndJulia, NULL, false);
	UpdateWindow(m_hwndJulia);
	InvalidateRect(m_hwndCtrl, NULL, false);
	UpdateWindow(m_hwndCtrl);
}

void CControlMandNURBS::OnMouseMove(int x, int y)
{
	float tmpX = 3.0f*(float)x/(float)m_width-2.25f;
	float tmpY = -2.5f*(float)y/(float)m_height+1.25f;

	if(!m_dragging)
	{
		bool refreshNeeded = m_NURBS->ClearVerticesHover();
		CVertex *tmpV = m_NURBS->GetNearestPoint(tmpX, tmpY);

		if(tmpV != NULL)
		{
			tmpV->SetHover(true);
			refreshNeeded = true;		
		}
		if(refreshNeeded)		
		{
			InvalidateRect(m_hwndCtrl, NULL, false);
			UpdateWindow(m_hwndCtrl);
		}
	}
	else
	{
		m_draggedVertex->SetVertex(tmpX, tmpY);

		m_positionX = m_NURBS->GetFrame(m_actualFrame).r;
		m_positionY = m_NURBS->GetFrame(m_actualFrame).i;
	
		((CJulia *)m_juliaObject)->SetSeed(m_positionX, m_positionY);
		m_juliaObject->Compute(0,1);
		InvalidateRect(m_hwndJulia, NULL, false);
		UpdateWindow(m_hwndJulia);
		InvalidateRect(m_hwndCtrl, NULL, false);
		UpdateWindow(m_hwndCtrl);
	}
}

void CControlMandNURBS::SetStepFrames(int i)
{
	m_NURBS->SetStepFrames(i);
}

void CControlMandNURBS::SetFrame(int i)
{
	m_actualFrame = i;
	m_positionX = m_NURBS->GetFrame(i).r;
	m_positionY = m_NURBS->GetFrame(i).i;

	((CJulia *)m_juliaObject)->SetSeed(m_positionX, m_positionY);
	m_juliaObject->Compute(0,1);
	InvalidateRect(m_hwndJulia, NULL, false);
	UpdateWindow(m_hwndJulia);
	InvalidateRect(m_hwndCtrl, NULL, false);
	UpdateWindow(m_hwndCtrl);
}

int CControlMandNURBS::GetRange()
{
	return (m_NURBS->GetVertexCount()-1)*m_NURBS->GetStepFrames();
}

long double CControlMandNURBS::GetFrameSeedX()
{
	return m_NURBS->GetFrame(m_actualFrame).r;
}

long double CControlMandNURBS::GetFrameSeedY()
{
	return m_NURBS->GetFrame(m_actualFrame).i;
}

void CControlMandNURBS::DrawSelection(HDC hdc)
{
	HPEN hPen, hpOld;
	hPen = CreatePen(PS_SOLID, 3, 0x000000FF);
	hpOld = (HPEN)SelectObject(hdc, hPen);

	MoveToEx(hdc, (int)((m_NURBS->GetFrame(0).r+2.25f)/3*m_width), (int)(m_height - (m_NURBS->GetFrame(0).i+1.25f)/2.5f*m_height), NULL);
	for(int i = 0; i < m_NURBS->GetFramesCount(); i++)
	{
		LineTo(hdc, (int)((m_NURBS->GetFrame(i).r+2.25f)/3*m_width), (int)(m_height - (m_NURBS->GetFrame(i).i+1.25f)/2.5f*m_height));
	}

	DeleteObject(hPen);  
	hPen = CreatePen(PS_SOLID, 5, 0x000000FF);
	SelectObject(hdc, hPen);
	for(int i = 0; i < m_NURBS->GetVertexCount(); i++)
	{
 		if(m_NURBS->GetVertex(i)->GetHover() || m_NURBS->GetVertex(i)->GetSelected())
  		{
			MoveToEx(hdc, (int)((m_NURBS->GetVertex(i)->GetX()+2.25)/3*m_width), (int)(m_height - (m_NURBS->GetVertex(i)->GetY()+1.25)/2.5*m_height), NULL);
			LineTo(hdc, (int)((m_NURBS->GetVertex(i)->GetX()+2.25)/3*m_width + 1), (int)(m_height - (m_NURBS->GetVertex(i)->GetY()+1.25)/2.5*m_height));
		}
  	}

	DeleteObject(hPen);  
	hPen = CreatePen(PS_SOLID, 7, 0x000000FF);
	SelectObject(hdc, hPen);
	for(int i = 0; i < m_NURBS->GetVertexCount(); i++)
	{
		if(!m_NURBS->GetVertex(i)->GetHover() && !m_NURBS->GetVertex(i)->GetSelected())
		{
			MoveToEx(hdc, (int)((m_NURBS->GetVertex(i)->GetX()+2.25)/3*m_width) + 1, (int)(m_height - (m_NURBS->GetVertex(i)->GetY()+1.25)/2.5*m_height), NULL);
			LineTo(hdc, (int)((m_NURBS->GetVertex(i)->GetX()+2.25)/3*m_width + 1), (int)(m_height - (m_NURBS->GetVertex(i)->GetY()+1.25)/2.5*m_height));
		}
	}

	DeleteObject(hPen);  
	hPen = CreatePen(PS_SOLID, 3, 0x0000FFFF);
	SelectObject(hdc, hPen);
	for(int i = 0; i < m_NURBS->GetVertexCount(); i++)
	{
		if(!m_NURBS->GetVertex(i)->GetHover() && !m_NURBS->GetVertex(i)->GetSelected())
		{
			MoveToEx(hdc, (int)((m_NURBS->GetVertex(i)->GetX()+2.25)/3*m_width), (int)(m_height - (m_NURBS->GetVertex(i)->GetY()+1.25)/2.5*m_height), NULL);
			LineTo(hdc, (int)((m_NURBS->GetVertex(i)->GetX()+2.25)/3*m_width + 1), (int)(m_height - (m_NURBS->GetVertex(i)->GetY()+1.25)/2.5*m_height));
		}
	}

	DeleteObject(hPen);  
	hPen = CreatePen(PS_SOLID, 5, 0x0000FF00);
	SelectObject(hdc, hPen);
	MoveToEx(hdc, (int)((m_positionX+2.25)/3*m_width), (int)(m_height - (m_positionY+1.25)/2.5*m_height), NULL);
	LineTo(hdc, (int)((m_positionX+2.25)/3*m_width + 1), (int)(m_height - (m_positionY+1.25)/2.5*m_height));

	SelectObject(hdc, hpOld);
	DeleteObject(hPen);  
}