#include "CControlPalette.h"
#include "stdio.h"

LRESULT CALLBACK CControlPalette::CustWndProcWrapper(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
	CControlPalette *pThis = (CControlPalette*)GetProp(hwnd,"ClassPointer");
	if(pThis) 
		return pThis->CustWndProc(hwnd,msg,wParam,lParam);
	return DefWindowProc(hwnd,msg,wParam,lParam);
}

LRESULT CControlPalette::CustWndProc(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
	PAINTSTRUCT ps; 
	HDC hdc; 

    switch(msg)
    {
	case WM_PAINT:
		if(wParam == 0)
			hdc = BeginPaint(hwnd, &ps);
		else
			hdc = (HDC)wParam;

		OnPaint(hdc);

		if(wParam == 0)
			EndPaint(hwnd, &ps);
	break;
	case WM_LBUTTONDOWN:
		RECT r;
		GetWindowRect(hwnd, &r);
		r.bottom--;
		r.top++;
		r.left++;
		r.right--;
		ClipCursor(&r);
		OnLButtonDown(wParam, lParam);
	break;
	case WM_LBUTTONUP:
		ClipCursor(NULL);
		OnLButtonUp(wParam, lParam);
	break;
	case WM_MOUSEMOVE:
		OnMouseMove(wParam, lParam);
	break;
	default:
        break;
    }

    return DefWindowProc(hwnd, msg, wParam, lParam);
}

CControlPalette::CControlPalette(HWND hwndParent)
: CCustomColorControl(hwndParent),
  m_dragging(false)
{
	m_className = _T("CControlPalette");
	m_h = 0.0f;
	m_s = 1.0f;
	m_v = 0.8f;

	m_width = 256;
	m_height = 256;

	InitCustomControl();
	CreateCustomControl();

	m_bitmapInfo.biSize = sizeof(BITMAPINFOHEADER);
	m_bitmapInfo.biWidth = m_width;
	m_bitmapInfo.biHeight = m_height;
	m_bitmapInfo.biPlanes = 1;
	m_bitmapInfo.biBitCount = 24;
	m_bitmapInfo.biCompression = BI_RGB;
	m_bitmapInfo.biSizeImage = 0;
	m_bitmapInfo.biXPelsPerMeter = 0;
	m_bitmapInfo.biYPelsPerMeter = 0;
	m_bitmapInfo.biClrUsed = 0;
	m_bitmapInfo.biClrImportant = 0;

	HDC hdc = GetWindowDC(NULL);

	m_hBitmap = CreateDIBSection(hdc, (LPBITMAPINFO)&m_bitmapInfo,DIB_RGB_COLORS,(LPVOID *)&m_palette,NULL,0);

	GenerateBitmap();

	ReleaseDC(NULL,hdc);
}

CControlPalette::~CControlPalette()
{
	DeleteObject(m_hBitmap);
}

void CControlPalette::InitCustomControl()
{
    WNDCLASSEX wc;
    
    wc.cbSize         = sizeof(wc);
    wc.lpszClassName  = m_className;
    wc.hInstance      = GetModuleHandle(0);
    wc.lpfnWndProc    = CustWndProcWrapper;
    wc.hCursor        = LoadCursor (NULL, IDC_ARROW);
    wc.hIcon          = 0;
    wc.lpszMenuName   = 0;
    wc.hbrBackground  = (HBRUSH)GetSysColorBrush(COLOR_BTNFACE);
    wc.style          = 0;
    wc.cbClsExtra     = 0;
    wc.cbWndExtra     = 0;
    wc.hIconSm        = 0;

    RegisterClassEx(&wc);
}


void CControlPalette::CreateCustomControl()
{
	m_hwndCtrl = CreateWindowEx(
                 WS_EX_STATICEDGE, // give it a standard border
                 m_className,
                 _T("Palette Control"),
                 WS_VISIBLE | WS_CHILD,
                 10, 145, m_width + 2, m_height + 2,
                 m_hwndParent,
                 NULL, GetModuleHandle(0), NULL
               );

	SetProp(m_hwndCtrl,"ClassPointer",(HANDLE)this);
}

void CControlPalette::OnMouseMove(WPARAM wParam, LPARAM lParam)
{
	if(m_dragging)
	{
		rgb c;
		m_h = (float)LOWORD(lParam)/255.0f;
		m_s = 1.0f - (float)HIWORD(lParam)/255.0f;
		InvalidateRect(m_hwndCtrl, NULL, false);
		UpdateWindow(m_hwndCtrl);
		PostMessage(m_hwndParent, WM_SET_HS, WPARAM(m_h*360), LPARAM(m_s*100));
		HSV2RGB(m_h, m_s, m_v, &c);
		unsigned int i = (c.r << 16) + (c.g << 8) + c.b;
		long j = ((unsigned short)(m_h*360) << 16) + ((unsigned char)(m_s*100) << 8) + (unsigned char)(m_v*100);
		SendMessage(m_hwndParent, WM_SET_RGB_HSV, WPARAM(i), LPARAM(j));
	}
}

void CControlPalette::OnLButtonDown(WPARAM wParam, LPARAM lParam) 
{
	m_dragging = true;

	rgb c;

	m_h = (float)LOWORD(lParam)/255.0f;
	m_s = 1.0f - (float)HIWORD(lParam)/255.0f;
	InvalidateRect(m_hwndCtrl, NULL, false);
	PostMessage(m_hwndParent, WM_SET_HS, WPARAM(m_h*360), LPARAM(m_s*100));

	HSV2RGB(m_h, m_s, m_v, &c);
	unsigned int i = (c.r << 16) + (c.g << 8) + c.b;
	long j = ((unsigned short)(m_h*360) << 16) + ((unsigned char)(m_s*100) << 8) + (unsigned char)(m_v*100);
	SendMessage(m_hwndParent, WM_SET_RGB_HSV, WPARAM(i), LPARAM(j));
}

void CControlPalette::OnLButtonUp(WPARAM wParam, LPARAM lParam) 
{
	m_dragging = false;
}

void CControlPalette::SetV(float v)
{
	m_v = v;
	GenerateBitmap();
	InvalidateRect(m_hwndCtrl, NULL, false);
	UpdateWindow(m_hwndCtrl);
}

void CControlPalette::SetRGB(unsigned char r, unsigned char g, unsigned char b)
{
	if(r > 255)
		r = 255;
	if(g > 255)
		g = 255;
	if(b > 255)
		b = 255;

	hsv c;
	RGB2HSV(r,g,b,&c);
	m_v = c.v;
	m_h = c.h;
	m_s = c.s;
	GenerateBitmap();
	InvalidateRect(m_hwndCtrl, NULL, false);
}

void CControlPalette::SetHSV(short int h, unsigned char s, unsigned char v)
{
	if(h > 360)
		h = 360;
	if(s > 100)
		s = 100;
	if(v > 100)
		v = 100;

	m_v = (float)v/100.0f;
	m_h = (float)h/360.0f;
	m_s = (float)s/100.0f;
	GenerateBitmap();
	InvalidateRect(m_hwndCtrl, NULL, false);
}

void CControlPalette::GenerateBitmap()
{
	rgb c;
	for(int s = 0; s < 256; s++)
		for(int h = 0; h < 256; h++)
		{
			HSV2RGB((float)h/256.0f, (float)s/256.0f, m_v, &c);
			m_palette[3*(256*s+h)] = c.b;
			m_palette[3*(256*s+h)+1] = c.g;
			m_palette[3*(256*s+h)+2] = c.r;
		}
}

void CControlPalette::DrawSelection(HDC hdc)
{
	HPEN hPen, hpOld;

	hPen = CreatePen(PS_SOLID, 3, 0);
	hpOld = (HPEN)SelectObject(hdc, hPen);

	MoveToEx(hdc, int(m_h*255) - 7, 255 - int(m_s*255), NULL);
	LineTo(hdc, int(m_h*255) - 2, 255 - int(m_s*255));
	MoveToEx(hdc, int(m_h*255) + 7, 255 - int(m_s*255), NULL);
	LineTo(hdc, int(m_h*255) + 2, 255 - int(m_s*255));
	MoveToEx(hdc, int(m_h*255), 255 - int(m_s*255) - 7, NULL);
	LineTo(hdc, int(m_h*255), 255 - int(m_s*255) - 2);
	MoveToEx(hdc, int(m_h*255), 255 - int(m_s*255) + 7, NULL);
	LineTo(hdc, int(m_h*255), 255 - int(m_s*255) + 2);

	DeleteObject(hPen);
	hPen = CreatePen(PS_SOLID, 1, 0x00FFFFFF);
	(HPEN)SelectObject(hdc, hPen);

	MoveToEx(hdc, int(m_h*255) - 7, 255 - int(m_s*255), NULL);
	LineTo(hdc, int(m_h*255) - 2, 255 - int(m_s*255));
	MoveToEx(hdc, int(m_h*255) + 7, 255 - int(m_s*255), NULL);
	LineTo(hdc, int(m_h*255) + 2, 255 - int(m_s*255));
	MoveToEx(hdc, int(m_h*255), 255 - int(m_s*255) - 7, NULL);
	LineTo(hdc, int(m_h*255), 255 - int(m_s*255) - 2);
	MoveToEx(hdc, int(m_h*255), 255 - int(m_s*255) + 7, NULL);
	LineTo(hdc, int(m_h*255), 255 - int(m_s*255) + 2);

	SelectObject(hdc, hpOld);
	DeleteObject(hPen);  
}