#include "COpenGLControlMand.h"
#include "CMandelbrot.h"
#include "CJulia.h"
#include "stdio.h"

#include <gl/gl.h>
#include <gl/glu.h>

#pragma comment (lib,"opengl32.lib")
#pragma comment (lib,"glu32.lib")
#pragma comment (lib,"glaux.lib")

#define WM_UPDATEPOSITION	WM_APP + 0x162

LRESULT CALLBACK COpenGLControlMand::MandWndProcWrapper(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
	COpenGLControlMand *pThis = (COpenGLControlMand*)GetProp(hwnd,"ClassPointer");
	if(pThis) 
		return pThis->CustWndProc(hwnd,msg,wParam,lParam);
	return DefWindowProc(hwnd,msg,wParam,lParam);
}

LRESULT COpenGLControlMand::CustWndProc(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
    switch(msg)
    {
	case WM_PAINT:
		wglMakeCurrent(m_hDC,m_hRC);
		OnPaint();
	break;
	case WM_LBUTTONDOWN:
		OnLButtonDown(LOWORD(lParam), HIWORD(lParam));
	break;
	case WM_LBUTTONUP:
		OnLButtonUp(LOWORD(lParam), HIWORD(lParam));
	break;
	case WM_MOUSEMOVE:
		if((wParam & MK_LBUTTON) != 0)						// is L left mouse button pressed?
			OnMouseMove(LOWORD(lParam), HIWORD(lParam));
	break;
	case WM_DESTROY:
	 	wglMakeCurrent(NULL,NULL);	//neaktvny render contex
	    wglDeleteContext(m_hRC);		//zmae rendering contex	
		::ReleaseDC(m_hwndCtrl,m_hDC);		//ukon vykreslovanie	
	break;
	default:
        break;
    }

    return DefWindowProc(hwnd, msg, wParam, lParam);
}

COpenGLControlMand::COpenGLControlMand(HWND hwndParent, HWND hwndJulia, CFractal *julia, CMandelbrot *def) 
: m_hwndParent(hwndParent),
  m_className(_T("OpenGLControlMand")),
  m_juliaObject(julia),
  m_hwndJulia(hwndJulia),
  m_mouseX(197), m_mouseY(67),
  m_pointX(-0.4), m_pointY(0.6),
  m_width(320), m_height(256)
{
	InitCustomControl();
	CreateCustomControl();
	OnCreate(def);
}

COpenGLControlMand::~COpenGLControlMand()
{
	delete m_fractalObject;
}

void COpenGLControlMand::InitCustomControl()
{
    WNDCLASSEX wc;
    
    wc.cbSize         = sizeof(wc);
    wc.lpszClassName  = m_className;
    wc.hInstance      = GetModuleHandle(0);
    wc.lpfnWndProc    = MandWndProcWrapper;
    wc.hCursor        = LoadCursor (NULL, IDC_ARROW);
    wc.hIcon          = 0;
    wc.lpszMenuName   = 0;
    wc.hbrBackground  = (HBRUSH)GetSysColorBrush(COLOR_BTNFACE);
    wc.style          = CS_OWNDC;
    wc.cbClsExtra     = 0;
    wc.cbWndExtra     = 0;
    wc.hIconSm        = 0;


    RegisterClassEx(&wc);
}


void COpenGLControlMand::CreateCustomControl()
{
	m_hwndCtrl = CreateWindowEx(
                 WS_EX_CLIENTEDGE, // give it a standard border
                 m_className,
                 _T("OpenGL control Mand"),
                 WS_VISIBLE | WS_CHILD,
                 5, 5, m_width + 4, m_height + 4,
                 m_hwndParent,
                 NULL, GetModuleHandle(0), NULL
               );

	SetProp(m_hwndCtrl,"ClassPointer",(HANDLE)this);
}

int COpenGLControlMand::MySetPixelFormat(HDC hdc)
{
	    PIXELFORMATDESCRIPTOR *ppfd; 
		int pixelformat; 
 
	    PIXELFORMATDESCRIPTOR pfd = { 
		sizeof(PIXELFORMATDESCRIPTOR),	
        1,									//slo verzie
        PFD_DRAW_TO_WINDOW |				//Pixel Format mus podporova okno...
        PFD_SUPPORT_OPENGL |				//...aj OpenGL
        PFD_DOUBLEBUFFER,					//mus podporova double buffering
        PFD_TYPE_RGBA,						//vyaduje RGBA Format
        32,									//nastav farebn hbku
        0,0,0,0,0,0,						//farebn hbka ignorovan
        8,									//iadny alpha buffer
        0,									//Shift Bit ignorovan
        8,									//iadny Accumulation buffer
        0,0,0,0,							//Accumulation bity ignorovan
        64,									//32 bitov Z-Buffer
        8,									//iadny stencil buffer
        8,									//iadny auxiliary buffer
        PFD_MAIN_PLANE,						//hlavn vykreslovacia vrstva
        0,									//rezervovan
        0,0,0								//Layer Masks ignorovan
    }; 
   
    ppfd = &pfd;
 
    if ( (pixelformat = ChoosePixelFormat(hdc, ppfd)) == 0 ) 
    { 
        ::MessageBox(NULL, "ChoosePixelFormat failed", "Error", MB_OK); 
        return FALSE; 
    } 
 
    if (SetPixelFormat(hdc, pixelformat, ppfd) == FALSE) 
    { 
        ::MessageBox(NULL, "SetPixelFormat failed", "Error", MB_OK); 
        return FALSE; 
    } 
 
    return TRUE; 
}

int COpenGLControlMand::OnCreate(CMandelbrot *def) 
{
    //zsk device context
	m_hDC = GetDC(m_hwndCtrl);
    //ak neiel nastavi pixel formt ukonenie + hlka
	if(!MySetPixelFormat(m_hDC))
    {
		::MessageBox(::GetFocus(),"MySetPixelFormat Failed!","Error",MB_OK);
		return -1;
    }

	//vytvorme a nastavme render contex
    m_hRC = wglCreateContext(m_hDC);
	
	wglMakeCurrent(m_hDC,m_hRC);
	
	m_fractalObject = new CMandelbrot();
	m_fractalObject->SetSize(m_width, m_height);
	m_fractalObject->SetIterationCount(64);
	
	if(def != NULL)
		m_fractalObject->SetTexture(def->GetImage(), def->GetSizeXc()*def->GetSizeYc()*sizeof(unsigned char)*3);
	else
		m_fractalObject->Compute();

	InitGL();	//inicializcia OpenGL
	
    return 0;
}

//OpenGL setup
int COpenGLControlMand::InitGL(void)									
{
	glEnable(GL_TEXTURE_2D);
	glDisable(GL_DEPTH_TEST);
	//glClearColor(0.0f, 0.0f, 0.0f, 0.5f);				// ierne pozadie
	//glClearDepth(1.0f);									// Nastav Depth Buffer na 1
	
	glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP);
    glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);

	glMatrixMode(GL_PROJECTION);						// Modeling transformation
	glLoadIdentity();									// Reset The View
	glOrtho(-2.25, 0.75, -1.25, 1.25, -1, 1);	
	
	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();

	wglMakeCurrent(m_hDC,m_hRC);
	//glGenTextures(1, &m_texture);
	m_texture = 0;
	glBindTexture(GL_TEXTURE_2D, m_texture);
	glTexImage2D(GL_TEXTURE_2D, 0, GL_RGB, m_fractalObject->GetSizeXc(), m_fractalObject->GetSizeYc(), 0, GL_RGB, GL_UNSIGNED_BYTE, m_fractalObject->GetImage());

	glPointSize(4);
	return TRUE;										
}

void COpenGLControlMand::OnPaint() 
{
	//glClear(GL_COLOR_BUFFER_BIT);	// Vyist obrazovku a Depth Buffer

	Draw();	//kreslenie fraktlu

	SwapBuffers(m_hDC);	//prehod buffery
}


void COpenGLControlMand::Draw()	//vykreslenie fraktlu
{
	glBindTexture(GL_TEXTURE_2D, m_texture);

	glBegin(GL_QUADS);
		glTexCoord2f(0.0f, 1.0f); glVertex2f(-2.25, 1.25);
		glTexCoord2f(0.0f, 0.0f); glVertex2f(-2.25, -1.25);
		glTexCoord2f(0.625f, 0.0f); glVertex2f(0.75, -1.25);
		glTexCoord2f(0.625f, 1.0f); glVertex2f(0.75, 1.25);
	glEnd();

	glColor3f(1.0f, 0.0f, 0.0f);
	glDisable(GL_TEXTURE_2D);
	glBegin(GL_POINTS);
		glVertex2f(m_pointX, m_pointY);
	glEnd();
	glColor3f(1.0f, 1.0f, 1.0f);
	glEnable(GL_TEXTURE_2D);
}


void COpenGLControlMand::OnLButtonDown(int x, int y) 
{
	m_pointX = 3.0f*(float)x/(float)m_width-2.25f;
	m_pointY = -2.5f*(float)y/(float)m_height+1.25f;

	((CJulia *)m_juliaObject)->SetSeed(m_pointX, m_pointY);
	m_juliaObject->Compute();
	PostMessage(m_hwndJulia, WM_PAINT, NULL, NULL);
	PostMessage(m_hwndCtrl, WM_PAINT, NULL, NULL);
	PostMessage(m_hwndParent, WM_UPDATEPOSITION, m_pointX*100000, m_pointY*100000);
}

void COpenGLControlMand::OnLButtonUp(int x, int y) 
{
	PostMessage(m_hwndCtrl, WM_PAINT, NULL, NULL);
}

void COpenGLControlMand::SetPointPossition(long double x, long double y)
{
	m_pointX = x;
	m_pointY = y;

	((CJulia *)m_juliaObject)->SetSeed(m_pointX, m_pointY);
	m_juliaObject->Compute();
	PostMessage(m_hwndJulia, WM_PAINT, NULL, NULL);
	PostMessage(m_hwndCtrl, WM_PAINT, NULL, NULL);
}

void COpenGLControlMand::OnMouseMove(int x, int y)
{
	m_pointX = 3.0f*(float)x/(float)m_width-2.25f;
	m_pointY = -2.5f*(float)y/(float)m_height+1.25f;

	((CJulia *)m_juliaObject)->SetSeed(m_pointX, m_pointY);
	m_juliaObject->Compute();
	PostMessage(m_hwndJulia, WM_PAINT, NULL, NULL);
	PostMessage(m_hwndCtrl, WM_PAINT, NULL, NULL);
	PostMessage(m_hwndParent, WM_UPDATEPOSITION, m_pointX*100000, m_pointY*100000);
}