#include "COpenGLControlMandNURBS.h"
#include "NURBS.h"
#include "CMandelbrot.h"
#include "CJulia.h"
#include "stdio.h"

#include <gl/gl.h>
#include <gl/glu.h>

#pragma comment (lib,"opengl32.lib")
#pragma comment (lib,"glu32.lib")
#pragma comment (lib,"glaux.lib")

#define WM_UPDATEPOSITION	WM_APP + 0x162
#define WM_UPDATERANGE		WM_APP + 0x164

LRESULT CALLBACK COpenGLControlMandNURBS::MandWndProcWrapper(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
	COpenGLControlMandNURBS *pThis = (COpenGLControlMandNURBS*)GetProp(hwnd,"ClassPointer");
	if(pThis) 
		return pThis->CustWndProc(hwnd,msg,wParam,lParam);
	return DefWindowProc(hwnd,msg,wParam,lParam);
}

LRESULT COpenGLControlMandNURBS::CustWndProc(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
    switch(msg)
    {
	case WM_PAINT:
		wglMakeCurrent(m_hDC,m_hRC);
		OnPaint();
	break;
	case WM_LBUTTONDOWN:
		OnLButtonDown(LOWORD(lParam), HIWORD(lParam));
	break;
	case WM_LBUTTONUP:
		OnLButtonUp(LOWORD(lParam), HIWORD(lParam));
	break;
	case WM_RBUTTONDOWN:
		OnRButtonDown(LOWORD(lParam), HIWORD(lParam));
	break;
	case WM_RBUTTONUP:
		OnRButtonUp(LOWORD(lParam), HIWORD(lParam));
	break;
	case WM_MOUSEMOVE:
		//if((wParam & MK_LBUTTON) != 0)						// is L left mouse button pressed?
		OnMouseMove(LOWORD(lParam), HIWORD(lParam));
	break;
	case WM_DESTROY:
	 	wglMakeCurrent(NULL,NULL);	//neaktvny render contex
	    wglDeleteContext(m_hRC);		//zmae rendering contex	
		::ReleaseDC(m_hwndCtrl,m_hDC);		//ukon vykreslovanie	
	break;
	default:
        break;
    }

    return DefWindowProc(hwnd, msg, wParam, lParam);
}

COpenGLControlMandNURBS::COpenGLControlMandNURBS(HWND hwndParent, HWND hwndJulia, CFractal *julia, CNURBS *nurbs, CMandelbrot *def) 
: m_hwndParent(hwndParent),
  m_className(_T("OpenGLControlMandNURBS")),
  m_juliaObject(julia),
  m_hwndJulia(hwndJulia),
  m_mouseX(197), m_mouseY(67),
  m_pointX(-0.4f), m_pointY(0.6f),
  m_positionX(m_pointX), m_positionY(m_pointY),
  m_width(640), m_height(512),
  m_dragging(false),
  m_draggedVertex(NULL),
  m_actualFrame(0)
{
	if(nurbs == NULL)
	{
		m_NURBS = new CNURBS();
		m_NURBS->SetStepFrames(24);
		m_NURBS->AddPoint(-0.4f, 0.6f);
		m_NURBS->AddPoint(-0.2f, 0.8f);
	}
	else
	{
		m_NURBS = nurbs;
	}
	//m_NURBS->AddPoint(0.0f, 0.6f);
	//m_NURBS->AddPoint(0.2f, 0.6f);
	InitCustomControl();
	CreateCustomControl();
	OnCreate(def);
}

COpenGLControlMandNURBS::~COpenGLControlMandNURBS()
{
	delete m_fractalObject;
}

void COpenGLControlMandNURBS::InitCustomControl()
{
    WNDCLASSEX wc;
    
    wc.cbSize         = sizeof(wc);
    wc.lpszClassName  = m_className;
    wc.hInstance      = GetModuleHandle(0);
    wc.lpfnWndProc    = MandWndProcWrapper;
    wc.hCursor        = LoadCursor (NULL, IDC_ARROW);
    wc.hIcon          = 0;
    wc.lpszMenuName   = 0;
    wc.hbrBackground  = (HBRUSH)GetSysColorBrush(COLOR_BTNFACE);
    wc.style          = CS_OWNDC;
    wc.cbClsExtra     = 0;
    wc.cbWndExtra     = 0;
    wc.hIconSm        = 0;


    RegisterClassEx(&wc);
}


void COpenGLControlMandNURBS::CreateCustomControl()
{
	m_hwndCtrl = CreateWindowEx(
                 WS_EX_CLIENTEDGE, 
                 m_className,
                 _T("OpenGL control MandNURBS"),
                 WS_VISIBLE | WS_CHILD,
                 5, 5, m_width + 4, m_height + 4,
                 m_hwndParent,
                 NULL, GetModuleHandle(0), NULL
               );

	SetProp(m_hwndCtrl,"ClassPointer",(HANDLE)this);
}

int COpenGLControlMandNURBS::MySetPixelFormat(HDC hdc)
{
	    PIXELFORMATDESCRIPTOR *ppfd; 
		int pixelformat; 
 
	    PIXELFORMATDESCRIPTOR pfd = { 
		sizeof(PIXELFORMATDESCRIPTOR),	
        1,									//slo verzie
        PFD_DRAW_TO_WINDOW |				//Pixel Format mus podporova okno...
        PFD_SUPPORT_OPENGL |				//...aj OpenGL
        PFD_DOUBLEBUFFER,					//mus podporova double buffering
        PFD_TYPE_RGBA,						//vyaduje RGBA Format
        32,									//nastav farebn hbku
        0,0,0,0,0,0,						//farebn hbka ignorovan
        8,									//iadny alpha buffer
        0,									//Shift Bit ignorovan
        8,									//iadny Accumulation buffer
        0,0,0,0,							//Accumulation bity ignorovan
        64,									//32 bitov Z-Buffer
        8,									//iadny stencil buffer
        8,									//iadny auxiliary buffer
        PFD_MAIN_PLANE,						//hlavn vykreslovacia vrstva
        0,									//rezervovan
        0,0,0								//Layer Masks ignorovan
    }; 
   
    ppfd = &pfd;
 
    if ( (pixelformat = ChoosePixelFormat(hdc, ppfd)) == 0 ) 
    { 
        ::MessageBox(NULL, "ChoosePixelFormat failed", "Error", MB_OK); 
        return FALSE; 
    } 
 
    if (SetPixelFormat(hdc, pixelformat, ppfd) == FALSE) 
    { 
        ::MessageBox(NULL, "SetPixelFormat failed", "Error", MB_OK); 
        return FALSE; 
    } 
 
    return TRUE; 
}

int COpenGLControlMandNURBS::OnCreate(CMandelbrot *def) 
{
    //zsk device context
	m_hDC = GetDC(m_hwndCtrl);
    //ak neiel nastavi pixel formt ukonenie + hlka
	if(!MySetPixelFormat(m_hDC))
    {
		::MessageBox(::GetFocus(),"MySetPixelFormat Failed!","Error",MB_OK);
		return -1;
    }

	//vytvorme a nastavme render contex
    m_hRC = wglCreateContext(m_hDC);
	
	wglMakeCurrent(m_hDC,m_hRC);
	
	m_fractalObject = new CMandelbrot();
	m_fractalObject->SetSize(m_width, m_height);
	m_fractalObject->SetIterationCount(128);
	
	if(def != NULL)
		m_fractalObject->SetTexture(def->GetImage(), def->GetSizeXc()*def->GetSizeYc()*sizeof(unsigned char)*3);
	else
		m_fractalObject->Compute();

	InitGL();
	
    return 0;
}

int COpenGLControlMandNURBS::InitGL(void)									
{
	glEnable(GL_TEXTURE_2D);
	glDisable(GL_DEPTH_TEST);
	
	glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP);
    glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);

	glMatrixMode(GL_PROJECTION);						// Modeling transformation
	glLoadIdentity();									// Reset The View
	glOrtho(-2.25, 0.75, -1.25, 1.25, -1, 1);	
	
	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();

	wglMakeCurrent(m_hDC,m_hRC);
	//glGenTextures(1, &m_texture);
	m_texture = 0;
	glBindTexture(GL_TEXTURE_2D, m_texture);
	glTexImage2D(GL_TEXTURE_2D, 0, GL_RGB, m_fractalObject->GetSizeXc(), m_fractalObject->GetSizeYc(), 0, GL_RGB, GL_UNSIGNED_BYTE, m_fractalObject->GetImage());

	glPointSize(4);
	return TRUE;										
}

void COpenGLControlMandNURBS::OnPaint() 
{
	Draw();
	SwapBuffers(m_hDC);
}


void COpenGLControlMandNURBS::Draw()
{
	glBindTexture(GL_TEXTURE_2D, m_texture);

	glBegin(GL_QUADS);
		glTexCoord2f(0.0f, 1.0f); glVertex2f(-2.25, 1.25);
		glTexCoord2f(0.0f, 0.0f); glVertex2f(-2.25, -1.25);
		glTexCoord2f(0.625f, 0.0f); glVertex2f(0.75, -1.25);
		glTexCoord2f(0.625f, 1.0f); glVertex2f(0.75, 1.25);
	glEnd();

	glDisable(GL_TEXTURE_2D);
	
	glColor3f(1.0f, 1.0f, 0.0f);
	glBegin(GL_LINE_STRIP);
		for(int i = 0; i < m_NURBS->GetFramesCount(); i++)
		{
			glVertex2f(m_NURBS->GetFrame(i).r, m_NURBS->GetFrame(i).i);
		}
	glEnd();
	
	glColor3f(1.0f, 0.0f, 0.0f);
	glBegin(GL_POINTS);
		for(int i = 0; i < m_NURBS->GetVertexCount(); i++)
		{
			if(m_NURBS->GetVertex(i)->GetHover() || m_NURBS->GetVertex(i)->GetSelected())
			{
				glColor3f(1.0f, 0.5f, 0.0f);
				glVertex2f(m_NURBS->GetVertex(i)->GetX(), m_NURBS->GetVertex(i)->GetY());
				glColor3f(1.0f, 0.0f, 0.0f);
			}
			else
				glVertex2f(m_NURBS->GetVertex(i)->GetX(), m_NURBS->GetVertex(i)->GetY());
		}
	glEnd();

	glColor3f(0.0f, 1.0f, 0.0f);
	glBegin(GL_POINTS);
		glVertex2f(m_positionX, m_positionY);
	glEnd();
	glColor3f(1.0f, 1.0f, 1.0f);
	glEnable(GL_TEXTURE_2D);
}


void COpenGLControlMandNURBS::OnLButtonDown(int x, int y) 
{
	float tmpX = 3.0f*(float)x/(float)m_width-2.25f;
	float tmpY = -2.5f*(float)y/(float)m_height+1.25f;

	CVertex *tmpV = m_NURBS->GetNearestPoint(tmpX, tmpY);

	if(tmpV == NULL)
	{
		m_NURBS->AddPoint(tmpX, tmpY);
		PostMessage(m_hwndParent, WM_UPDATERANGE, (m_NURBS->GetVertexCount()-1)*m_NURBS->GetStepFrames(), NULL);
		PostMessage(m_hwndJulia, WM_PAINT, NULL, NULL);
		wglMakeCurrent(m_hDC,m_hRC);
		OnPaint();
	}
	else
	{
		m_draggedVertex = tmpV;
		m_draggedVertex->SetSelected(true);
		m_dragging = true;
		wglMakeCurrent(m_hDC,m_hRC);
		OnPaint();
	}
}

void COpenGLControlMandNURBS::OnLButtonUp(int x, int y) 
{
	m_dragging = false;
	if(m_draggedVertex != NULL)
		m_draggedVertex->SetSelected(false);
	m_draggedVertex = NULL;
	wglMakeCurrent(m_hDC,m_hRC);
	OnPaint();
}

void COpenGLControlMandNURBS::OnRButtonDown(int x, int y) 
{
	float tmpX = 3.0f*(float)x/(float)m_width-2.25f;
	float tmpY = -2.5f*(float)y/(float)m_height+1.25f;

	CVertex *tmpV = m_NURBS->GetNearestPoint(tmpX, tmpY);

	if(tmpV != NULL)
	{
		m_toDeleteVertex = tmpV;
	}
}

void COpenGLControlMandNURBS::OnRButtonUp(int x, int y) 
{
	float tmpX = 3.0f*(float)x/(float)m_width-2.25f;
	float tmpY = -2.5f*(float)y/(float)m_height+1.25f;

	CVertex *tmpV = m_NURBS->GetNearestPoint(tmpX, tmpY);
	if(tmpV == m_toDeleteVertex)
	{
		m_NURBS->DeleteVertex(m_toDeleteVertex);
		wglMakeCurrent(m_hDC,m_hRC);
		OnPaint();
		PostMessage(m_hwndParent, WM_UPDATERANGE, (m_NURBS->GetVertexCount()-1)*m_NURBS->GetStepFrames(), NULL);
	}
}

void COpenGLControlMandNURBS::SetPointPossition(float x, float y)
{
	m_pointX = x;
	m_pointY = y;

	((CJulia *)m_juliaObject)->SetSeed(m_pointX, m_pointY);
	m_juliaObject->Compute();
	PostMessage(m_hwndJulia, WM_PAINT, NULL, NULL);
	wglMakeCurrent(m_hDC,m_hRC);
	OnPaint();
}

void COpenGLControlMandNURBS::OnMouseMove(int x, int y)
{
	float tmpX = 3.0f*(float)x/(float)m_width-2.25f;
	float tmpY = -2.5f*(float)y/(float)m_height+1.25f;

	//PostMessage(m_hwndParent, WM_UPDATEPOSITION, tmpX*100000, tmpY*100000);
	
	if(!m_dragging)
	{
		m_NURBS->ClearVerticesHover();
		CVertex *tmpV = m_NURBS->GetNearestPoint(tmpX, tmpY);

		if(tmpV != NULL)
		{
			tmpV->SetHover(true);
		}
		wglMakeCurrent(m_hDC,m_hRC);
		OnPaint();
	}
	else
	{
		m_draggedVertex->SetVertex(tmpX, tmpY);

		m_positionX = m_NURBS->GetFrame(m_actualFrame).r;
		m_positionY = m_NURBS->GetFrame(m_actualFrame).i;
	
		((CJulia *)m_juliaObject)->SetSeed(m_positionX, m_positionY);
		m_juliaObject->Compute();
		PostMessage(m_hwndJulia, WM_PAINT, NULL, NULL);
	
		wglMakeCurrent(m_hDC,m_hRC);
		OnPaint();
	}
}

void COpenGLControlMandNURBS::SetStepFrames(int i)
{
	m_NURBS->SetStepFrames(i);
}

void COpenGLControlMandNURBS::SetFrame(int i)
{
	m_actualFrame = i;
	m_positionX = m_NURBS->GetFrame(i).r;
	m_positionY = m_NURBS->GetFrame(i).i;

	((CJulia *)m_juliaObject)->SetSeed(m_positionX, m_positionY);
	m_juliaObject->Compute();
	PostMessage(m_hwndJulia, WM_PAINT, NULL, NULL);
	wglMakeCurrent(m_hDC,m_hRC);
	OnPaint();
}

int COpenGLControlMandNURBS::GetRange()
{
	return (m_NURBS->GetVertexCount()-1)*m_NURBS->GetStepFrames();
}

long double COpenGLControlMandNURBS::GetFrameSeedX()
{
	return m_NURBS->GetFrame(m_actualFrame).r;
}

long double COpenGLControlMandNURBS::GetFrameSeedY()
{
	return m_NURBS->GetFrame(m_actualFrame).i;
}