#include <utility>
#include <set>
#include <boost/concept_check.hpp>
#include <boost/tuple/tuple.hpp>
#include "external_graph.hpp"
#include "external_stack.hpp"
#include "external_sequence.hpp"

namespace boost {

  template <class EDFSVisitor>
  class EDFSVisitorConcept {
    public:
      void constraints() {
        visitor.visit(vertex);
      }
    private:
      EDFSVisitor visitor;
      int vertex;
  };

  // handle visiting of a single vertex in external_depth_first_search
  template <class EGraph>
  inline void visit(int v, std::set<int>& ss, external_stack& stack, EGraph& g) {
    ss.insert(v);
    if (ss.size() >= MEMORY_ITEMS) {
      g.remove_visited(ss);
      ss.clear();
    }
    std::pair<int, int> it2 = g.init_iterator(v);
    stack.push(it2.first); stack.push(it2.second);
  }

  template <class EGraph, class EDFSVisitor>
  void external_depth_first_search(EGraph &g, EDFSVisitor &visitor, int u) {
    typedef std::set<int> SearchStructure;
    
    function_requires<EDFSVisitorConcept<EDFSVisitor> >();
    function_requires<EGraphConcept<EGraph> >();
    function_requires<MultiRemoveEGraphConcept<EGraph, SearchStructure> >();
  
    external_stack stack;
    SearchStructure ss;
    
    visit(u, ss, stack, g); visitor.visit(u);
    while (!stack.empty()) {
      // restore iterator from current stack frame
      int second = stack.pop(), first = stack.pop();
      g.set_position(first, second);
      
      if (g.has_next()) {
        // find next unvisited vertex
        int t = g.get_next();
        while (g.has_next() && ss.count(t)) t = g.get_next();
        
        // store iterator position if more neighbours exist
        // not storing position is not equivalent to leaving current vertex
        if (g.has_next()) {
          tie(first, second) = g.get_position();
          stack.push(first); stack.push(second);
        }
        
        // visit next unvisited vertex, if exists
        if (!ss.count(t)) {
          visit(t, ss, stack, g);
          visitor.visit(t);
        }
      }
    }
    g.remove_visited(ss);
  }
  
  template <class EDFSVisitor>
  class dft_visitor {
  
  private:
    EDFSVisitor &visitor;
    bit_sequence &unvisited;
    
  public:
    dft_visitor(EDFSVisitor &v, bit_sequence &u) : visitor(v), unvisited(u) {
      function_requires<EDFSVisitorConcept<EDFSVisitor> >();
    }
    
    void visit(int vertex) {
      unvisited.remove(vertex);
      visitor.visit(vertex);
    }
    
    int end_mark;
  
  };

  template <class EGraph, class EDFSVisitor>
  void external_depth_first_traversal(EGraph &g, EDFSVisitor &visitor) {
    function_requires<EDFSVisitorConcept<EDFSVisitor> >();

    bit_sequence unvisited("unvisited.tmp", num_vertices(g));
    dft_visitor<EDFSVisitor> vis(visitor, unvisited);

    // call external_depth_first_search for the first unvisited vertex until no more such vertices exists
    while (unvisited.init_iterator(), unvisited.has_next()) external_depth_first_search(g, vis, unvisited.get_next());
  }


}
