#include "external_sequence.hpp"
#include <stdlib.h>
#include <math.h>
#include <limits.h>
#define BLOCK_COUNT ((int) ceil(double(count*sizeof(int)) / double(BLOCK_SIZE)))
#define MIN(X,Y)  ((X) < (Y) ? (X) : (Y))

namespace boost {
  
  class mergesort_buffer {
  friend block_file* external_mergesort(block_file *f, int count);
  private: 
    int *buffer, level, run_size, run_count, count;
    block_file *file;
    int *position;
    int nonempty_runs;
    int first_run;
    
    mergesort_buffer(int c) : position(0), count(c) {
      buffer = new int[MEMORY_ITEMS];
    }
    
    // initialize for the next level
    void set_level(int l, block_file *f, int r) {
      run_count = r;
      level = l; file = f;
      if (level==1) run_size = MEMORY_BLOCKS * BLOCK_SIZE;
      else run_size *= MEMORY_BLOCKS;
    }
    
    // initialize for the next tuple of runs
    void set_first_run(int f) {
      nonempty_runs = MIN(f+MEMORY_BLOCKS,run_count)-f;
      first_run = f;
      if (position) delete position;
      position = new int[MEMORY_ITEMS];
      for (int i=0; i<nonempty_runs; i++) {
        position[i] = 0;
        file->load_block((first_run+i)*run_size, 1, &(buffer[i*BLOCK_SIZE/sizeof(int)]));
      }
    }
    
    // is there a non-empty run?
    bool has_current() {
      return nonempty_runs > 0;
    }
    
    // is this run non-empty?
    bool has_current(int index) {
      if (index>=run_count) return false;
      int top = (index+first_run==run_count-1)?run_size-(run_size*run_count-count*sizeof(int)):run_size;
      return position[index] < top;
    }
    
    // get current number from this run
    int get_current(int index) {
      return buffer[index*BLOCK_SIZE/sizeof(int) + (position[index]%BLOCK_SIZE)/sizeof(int)];
    }
    
    // shift to the next number for this run, reading next block if necessary
    void fetch_next(int index) {
      position[index] += sizeof(int);
      int top = (index+first_run==run_count-1)?run_size-(run_size*run_count-count*sizeof(int)):run_size;
      if (position[index]==top) nonempty_runs--;
      else if ((position[index]-sizeof(int)) / BLOCK_SIZE != position[index] / BLOCK_SIZE) {
        file->load_block((first_run+index)*run_size+position[index], 1, &(buffer[index*BLOCK_SIZE/sizeof(int)]));
      }
    }
    
    ~mergesort_buffer() {
      delete buffer;
    }
  };
  
  int cmpintp(const void* aa, const void* bb) {
    int a = *(const int*) aa; int b = *(const int*) bb;
    if (a<b) return -1;
    else if (a>b) return 1;
    else return 0;
  }
  
  block_file* external_mergesort(block_file *f, int count) {
    int offset = 0;
    block_file *result = f;
    mergesort_buffer b = mergesort_buffer(count);
    
    // create runs by sorting in internal memory
    for (int remains = count; remains > 0; offset += MEMORY_BLOCKS*BLOCK_SIZE, remains -= MEMORY_ITEMS) {
      f->load_block(offset, MEMORY_BLOCKS, (void *) b.buffer);
      qsort((void *) b.buffer, MIN(MEMORY_ITEMS, remains), sizeof(int), cmpintp);
      f->write_block(offset, MEMORY_BLOCKS, (void *) b.buffer);
    }
    
    // iterate through levels
    for (int level=1, run_count = (int) ceil(double(BLOCK_COUNT)/double(MEMORY_BLOCKS));
      run_count>1;
      level++, run_count = (int) ceil(double(run_count)/double(MEMORY_BLOCKS))
    ) {
      b.set_level(level, result, run_count);
      list_sequence *output = new list_sequence("output.tmp", 0);
      
      // iterate through tuples of runs
      for (int first_run=0; first_run<run_count; first_run += MEMORY_BLOCKS) {
        b.set_first_run(first_run);
        
        // find lowest number
        while (b.has_current()) {
          int minv=INT_MAX, minind;
          for (int i=0; i<MEMORY_BLOCKS; i++) if (b.has_current(i) && b.get_current(i)<=minv) {
            minv = b.get_current(i); minind = i;
          }
          // send it to output
          b.fetch_next(minind);
          output->add(minv);
        }
      }
      
      // shift files for the next level
      delete output; delete result;
      rename("output.tmp", "input.tmp");
      result = new block_file("input.tmp", "r+", 0);
    }
    
    result->buf_pos = -1; // force next load_block() to do fread() next time
    return result;
  }
  
}
