#ifndef EXTERNAL_SEQUENCE
#define EXTERNAL_SEQUENCE

#include "block_file.hpp"

namespace boost {
  
  template <class T>
  class ESequenceConcept {
    public:
      void constraints() {
        n = s.get_next();
        b = s.has_next();
        s.init_iterator();
      }
    private:
      T s;
      int n;
      bool b;
  };

  class sequence {
    
  protected:
    int iter_pos, n_items;
    bool is_empty;
    
  public:
    block_file* f;
    int count;

    sequence() : count(0), iter_pos(0), is_empty(true), n_items(0) {
      f = new block_file();
    }
    
    bool empty() {
      return is_empty;
    }

    bool has_next() {
      return iter_pos < count;
    }
    
    ~sequence() {
      delete f;
    }
    
  };
  
  class list_sequence : public sequence {
  
  private:
    int buf_pos;
    bool changed;
    
  public:
    list_sequence() : changed(false) { }
    
    list_sequence(int v) : changed(false) {
      is_empty = false;
      ((int *) f->buffer)[count] = v;
      count += sizeof(int);
    }
    
    list_sequence(char *file, int del=2) : buf_pos(0), changed(false) {
      f = new block_file(file, "w+", del);
      f->load_block(0);
    }
    
    void flush() {
      f->write_block(); f->load_block(count+sizeof(int)); changed = false;
    }
    
    void init_iterator() {
      iter_pos = 0; if (changed) flush();
    }
    
    int get_next() {
      f->load_block(iter_pos);
		  int result = ((int*) f->buffer)[(iter_pos % BLOCK_SIZE) / sizeof(int)];
      iter_pos += sizeof(int);
      return result;
    }
    
    void add(int v) {
      ((int*) f->buffer)[(count % BLOCK_SIZE) / sizeof(int)] = v;
      changed = true;
      if ((count+sizeof(int))/BLOCK_SIZE != count/BLOCK_SIZE) flush();
      count += sizeof(int); n_items++;
      is_empty = false;
    }
    
    ~list_sequence() {
      if (changed) flush();
    }

  };
  
  class bit_sequence : public sequence {
  
  private:
    // move the iterator to the next number that is present
    void skip() {
      int result;
      do {
   		  f->load_block(iter_pos/8);
        result = f->buffer[(iter_pos/8) % BLOCK_SIZE];
  		  result &= 1 << (iter_pos%8);
  		  if (result) iter_pos++;
      } while (result && iter_pos < count);
    }
  
  public:
    bit_sequence() { }
    
    bit_sequence(char* file, int c, int del=2) {
      count = c;
      f = new block_file(file, "w+", del); 
    }
    
    void init_iterator() {
      iter_pos = 0; skip();
    }
    
    int get_next() {
      int result = iter_pos;
		  iter_pos++; skip();
		  return result;
    }
    
    void remove(int v) {
      f->load_block(v/8);
      f->buffer[(v/8) % BLOCK_SIZE] |= 1 << (v%8);
      f->write_block();
      is_empty = false;
      n_items++;
    }
    
  };
  
  template <class Sequence, class Sequence2>
  class unify_sequence : public sequence {
    
  private:
    Sequence* s0;
    Sequence2* s1;
    int front[2];
    bool s_empty[2];

    // fetch next number for this sequence, updating s_empty
    inline int fetch_next(int index) {
      int result = front[index];
      s_empty[index] = (index==0)?!s0->has_next():!s1->has_next();
      if (!s_empty[index]) front[index] = (index==0)?s0->get_next():s1->get_next();
      return result;
    }
    
  public:
    unify_sequence() { }
    
    unify_sequence(Sequence *a, Sequence2 *b) : s0(a), s1(b) {
      function_requires<ESequenceConcept<Sequence> >();
      function_requires<ESequenceConcept<Sequence2> >();
    }
    
    void init_iterator() {
      s0->init_iterator(); s1->init_iterator();
      fetch_next(0); fetch_next(1);
    }
    
    int get_next() {
      int result;
      if (s_empty[0]) result = fetch_next(1);
      else if (s_empty[1]) result = fetch_next(0); 
      else {
        if (front[0] < front[1]) result = fetch_next(0); else result = fetch_next(1);
      }

      return result;
    }
    
    bool has_next() {
      bool result = !s_empty[0] || !s_empty[1];
      return result;
    }
    
    bool empty() {
      return s0->empty() && s1->empty();
    }
    
  };
  
  template <class Sequence, class Sequence2>
  class complement_sequence : public sequence {

  private:
    Sequence* s0;
    Sequence2* s1;
    int front[2];
    bool s_empty[2], next;
    
    // fetch next number for this sequence, updating s_empty
    inline int fetch_next(int index) {
      int result = front[index];
      s_empty[index] = (index==0)?!s0->has_next():!s1->has_next();
      if (!s_empty[index]) front[index] = (index==0)?s0->get_next():s1->get_next();
      return result;
    }
    
    // move in input sequences until acceptable output number is found
    void skip() {
      while (!s_empty[1] && front[0] >= front[1]) {
        while (!s_empty[1] && front[1] < front[0]) fetch_next(1);
        while (!s_empty[0] && front[0] == front[1]) fetch_next(0);
        if (s_empty[0]) return;
      }
    }
    
  public:
    
    complement_sequence() { }
    
    complement_sequence(Sequence *a, Sequence2 *b) : s0(a), s1(b) {
      function_requires<ESequenceConcept<Sequence> >();
      function_requires<ESequenceConcept<Sequence2> >();
    }

    void init_iterator() {
      s0->init_iterator(); s1->init_iterator();
      fetch_next(0); fetch_next(1);
      skip();
      is_empty = s_empty[0];
      next = !s_empty[0];
    }
    
    int get_next() {
      int result = fetch_next(0);
      skip(); next = !s_empty[0];
      return result;
    }
    
    bool has_next() {
      return next;
    }
    
  };

  template <class Sequence>
  class remove_duplicates : public sequence {
    
  private:
    Sequence *s;
    int front;
    bool empty;

    // fetch next number from the input sequence
    inline void fetch_next() {
      empty = !s->has_next();
      if (!empty) front = s->get_next();
    }
    
  public:
    remove_duplicates(Sequence *ss) : s(ss) {
      function_requires<ESequenceConcept<Sequence> >();
    }

    void init_iterator() {
      s->init_iterator(); fetch_next();
    }

    int get_next() {
      int result = front;
      while (result == front && !empty) fetch_next();
      return result;
    }

    bool has_next() {
      return !empty;
    }

  };
  
}
#endif
