package server;

import bridge.distrib.BridgeDescriptor;
import common.XmlSpecChConv;
import common.commun.StaticProtocolStrings;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.io.StringReader;
import java.net.Socket;
import java.util.Vector;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

public class SClient implements Runnable, SClientDescriptor {

    private static final int SERVER_INTRODUCTION_MSG = 1;
    private static final int CLIENTS_INFO_MSG = 2;
    private boolean passive;
    private int connectionPort;
    private long shareAmount;
    private SClientManager sclientManager;
    private String name;
    private String description;
    private String id;
    private Socket socket;
    private InputStream is;
    private BufferedWriter messageSender;
    private DocumentBuilder docBuilder;
    private Thread waiterThread = new Thread(this);

    public SClient(Socket s, InputStream is, SClientManager cm, String name, String description, boolean passive, long shareAmount,
            int conPort, String id) throws ParserConfigurationException, IOException {

        this.socket = s;
        this.is = is;
        sclientManager = cm;
        this.name = name;
        this.setDescription(description);
        this.passive = passive;
        this.connectionPort = conPort;
        this.shareAmount = shareAmount;
        this.id = id;

        docBuilder = DocumentBuilderFactory.newInstance().newDocumentBuilder();
        messageSender = new BufferedWriter(new OutputStreamWriter(socket.getOutputStream()));
        waiterThread.start();
        sendMessage(SERVER_INTRODUCTION_MSG);
        sendMessage(CLIENTS_INFO_MSG);
    }

    public boolean isPassive() {
        return passive;
    }

    public int getCommunicationPort() {
        return connectionPort;
    }

    public long getShareAmount() {
        return shareAmount;
    }

    public String getName() {
        return name;
    }

    public String getDescription() {
        return description;
    }

    public void setDescription(String description) {
        this.description = description;
    }

    public String getIPAddress() {
        return socket.getInetAddress().getHostAddress();
    }

    private void sendMessage(int type) {
        sendMessage(getMessage(type));
    }

    private void sendMessage(final String msg) {

        Thread t = new Thread(new Runnable() {

            public void run() {
                sendMessageHlp(msg);
            }
        });
        t.start();
    }

    private String getMessage(int messageType) {
        StringBuilder res = null;

        if (messageType == SERVER_INTRODUCTION_MSG) {
            res = new StringBuilder(120);
            res.append("<msg type=\"serverintroduction\">");
            res.append("<name>" + XmlSpecChConv.convert(SOptionalValueProvider.getServerName()) + "</name>");
            res.append("<id>" + XmlSpecChConv.convert(SOptionalValueProvider.getServerID()) + "</id>");
            res.append("<clientnum>" + sclientManager.getConnectedClientNumber() + "</clientnum>");
            res.append("</msg>");

        } else if (messageType == CLIENTS_INFO_MSG) {

            Vector<SClient> clients = (Vector<SClient>) sclientManager.getClients().clone();
            res = new StringBuilder(clients.size() * 60);

            res.append("<msg type = \"clientsinfo\">");
            for (SClient c : clients) {
                res.append("<client>");
                res.append("<name>" + XmlSpecChConv.convert(c.getName()) + "</name>");
                res.append("<desc>" + XmlSpecChConv.convert(c.getDescription()) + "</desc>");
                res.append("<ispassive>" + c.isPassive() + "</ispassive>");
                res.append("<conport>" + c.getCommunicationPort() + "</conport>");
                res.append("<shareamount>" + shareAmount + "</shareamount>");
                res.append("<id>" + XmlSpecChConv.convert(c.getId()) + "</id>");
                res.append("<ip>" + socket.getInetAddress().getHostAddress() + "</ip>");
                res.append("</client>");
            }

            res.append("</msg>");
        }
        return res.toString();
    }

    /** don't call directly */
    private synchronized void sendMessageHlp(String msg) {
        try {
            messageSender.write(msg);
            messageSender.newLine();
            messageSender.flush();
        } catch (IOException ex) {
            ex.printStackTrace();
        }
    }

    public void requestATPConTo(SClientDescriptor connectTo) {
        String msg = "<msg type = \"" + StaticProtocolStrings.TOACTO_FROM_PASS_SERVER_REQ + "\">" +
                "<ip>" + connectTo.getIPAddress() + "</ip>" +
                "<port>" + connectTo.getCommunicationPort() + "</port>" +
                "</msg>";
        sendMessage(msg);
    }

    private void processMessage(String msg) {
        try {
            Document doc = docBuilder.parse(new InputSource(new StringReader(msg)));
            Element root = doc.getDocumentElement();
            String type = root.getAttribute("type");
            if (type.equals(StaticProtocolStrings.REQ_SERVER_TO_ASK_PASSIVE_CON_TYPE)) {
                String cid = root.getElementsByTagName("id").item(0).getTextContent();
                handleTPRequest(cid, true);
            } else if (type.equals(StaticProtocolStrings.REQ_SERVER_TO_ASK_PASSIVE_TRANS_TYPE)) {
                String cid = root.getElementsByTagName("id").item(0).getTextContent();
                handleTPRequest(cid, false);
            }
        } catch (IOException ex) {
            ex.printStackTrace();
        } catch (SAXException ex) {
            ex.printStackTrace();
        }
    }

    public void sendToBridgeConReq(boolean asRequester, String id, boolean isConnectionPurpose) {
        BridgeDescriptor bd = sclientManager.getBridgeProvider().getBridgeDescriptor();
        if (isConnectionPurpose) {
            if (asRequester) {
                String msg = "<msg type = \"" + StaticProtocolStrings.SERV_TO_REQUESTER_TO_BRIDGE + "\">" +
                        "<ip>" + bd.getHost() + "</ip>" +
                        "<port>" + bd.getPort() + "</port>" +
                        "<id>" + XmlSpecChConv.convert(id) + "</id></msg>";

                sendMessage(msg);
            } else {
                String msg = "<msg type = \"" + StaticProtocolStrings.SERV_TO_REQUESTED_TO_BRIDGE + "\">" +
                        "<ip>" + bd.getHost() + "</ip>" +
                        "<port>" + bd.getPort() + "</port>" +
                        "<id>" + XmlSpecChConv.convert(id) + "</id></msg>";
                sendMessage(msg);
            }
        } else {
            if (asRequester) {
                String msg = "<msg type = \"" + StaticProtocolStrings.SERV_TO_REQUESTER_TO_BRIDGE_TRANS + "\">" +
                        "<ip>" + bd.getHost() + "</ip>" +
                        "<port>" + bd.getPort() + "</port>" +
                        "<id>" + XmlSpecChConv.convert(id) + "</id></msg>";

                sendMessage(msg);
            } else {
                String msg = "<msg type = \"" + StaticProtocolStrings.SERV_TO_REQUESTED_TO_BRIDGE_TRANS + "\">" +
                        "<ip>" + bd.getHost() + "</ip>" +
                        "<port>" + bd.getPort() + "</port>" +
                        "<id>" + XmlSpecChConv.convert(id) + "</id></msg>";
                sendMessage(msg);
            }
        }
    }

    private void handleTPRequest(String id, boolean isConnectPurpose) {
        Vector<SClient> cs = (Vector<SClient>) sclientManager.getClients().clone();
        SClient to = null;
        for (SClient c : cs) {
            if (c.getId().equals(id)) {
                to = c;
                break;
            }
        }

        if (to != null) {
            if (isPassive()) {
                to.sendToBridgeConReq(false, this.id, isConnectPurpose);
                sendToBridgeConReq(true, id, isConnectPurpose);
            } else {
                to.requestATPConTo(this);
            }
        } else {
        //TODO : reply : no such client 
        }
    }

    public void run() {
        BufferedReader br = new BufferedReader(new InputStreamReader(is));
        try {
            while (true) {
                String msg = br.readLine();
                if (msg == null) {
                    break;
                }
                processMessage(msg);
            }
        } catch (IOException ex) {
            ex.printStackTrace();
        } finally {
            sclientManager.fireClientDisconnected(this);
        }
    }

    public void sendClientDisconnectedMsg(String id) {
        String msg = "<msg type = \"" + StaticProtocolStrings.CLIENT_DISCONNECTED +
                "\" ><id>" + XmlSpecChConv.convert(id) + "</id></msg>";
        sendMessage(msg);
    }

    public void sendClientConnectedMessage(SClient c) {
        String msg = "<msg type = \"" + StaticProtocolStrings.CLIENT_CONNECTED + "\">" +
                "<name>" + XmlSpecChConv.convert(c.getName()) + "</name>" +
                "<desc>" + XmlSpecChConv.convert(c.getDescription()) + "</desc>" +
                "<ispassive>" + c.isPassive() + "</ispassive>" +
                "<conport>" + c.getCommunicationPort() + "</conport>" +
                "<shareamount>" + c.getShareAmount() + "</shareamount>" +
                "<id>" + XmlSpecChConv.convert(c.getId()) + "</id>" +
                "<ip>" + c.getIPAddress() + "</ip>" +
                "</msg>";
        sendMessage(msg);
    }

    public String getId() {
        return id;
    }
}
