from functools import wraps
from flask import request, Response, g
from http import HTTPStatus
import random
import time
import shutil
import os

from app import app
from .models import Client, Problem
from .constant import (
    Language,
    DEFAULT_SUBMIT_PRIORITY,
    MAX_SUBMIT_ID_SURROGATE,
    MIN_SUBMIT_ID_SURROGATE
)


def authenticate():
    return Response('Invalid or missing API key\n', HTTPStatus.UNAUTHORIZED)


def api_key_auth(f):
    @wraps(f)
    def decorated(*args, **kwargs):
        api_key = request.headers.get('Authorization')
        client = Client.query.filter_by(api_key=api_key).first()

        if client is None:
            return authenticate()

        g.client = client
        return f(*args, **kwargs)
    return decorated


def generate_submit_id():
    unix_timestamp = int(time.time())
    n = random.randint(MIN_SUBMIT_ID_SURROGATE, MAX_SUBMIT_ID_SURROGATE)
    return f'{unix_timestamp}-{n:05}'


def validate_submit(f):
    @wraps(f)
    def decorated(*args, **kwargs):
        try:
            problem_id = request.form['problem_id']
            language = request.form['language']
            submission_file = request.files['file']
        except KeyError:
            return Response('Missing parameter.\n', HTTPStatus.BAD_REQUEST)

        if not Language.has_value(language):
            return Response('Unsupported language.\n', HTTPStatus.BAD_REQUEST)
        if submission_file.filename == '':
            return Response('Missing submission file.\n', HTTPStatus.BAD_REQUEST)

        problem = Problem.query.filter_by(id=problem_id).first()
        if problem is None:
            return Response('Chosen problem does not exist.\n', HTTPStatus.BAD_REQUEST)

        g.uid = getattr(request.form, 'submit_id', generate_submit_id())
        g.problem = problem
        g.language = Language(language)
        g.submission_file = submission_file
        g.priority = getattr(request.form, 'priority', DEFAULT_SUBMIT_PRIORITY)

        return f(*args, **kwargs)
    return decorated


def submit_filepath(submit):
    return f'{submit.client.name}/{submit.uid}.{submit.language.value}'


def store_submit_file(submit, submit_file_path):
    new_path = os.path.join(app.config['SUBMIT'], submit_filepath(submit))
    shutil.move(submit_file_path, new_path)
