import importlib.machinery
import logging
import glob
import os
import shutil
import subprocess
import types

import isolate
from languages import languages
from utils import read_config, execute_external
from settings import (
    TASK_DIRECTORY,
    TEMP_DIRECTORY,
    SUBMIT_LANGUAGE,
    SUBMIT_FILE_PATH,
)
from constants import (
    COMPILATION_LOG_PATH,
    MAX_COMPILER_OUTPUT_LENGTH,
)


def is_task_custom_compile(task_config):
    return 'custom_compile' in task_config['task']


def copy_python_bytecode(sandbox_path):
    pycache_directory = os.path.join(sandbox_path, '__pycache__')
    try:
        pyc_file = os.listdir(pycache_directory)[0]
        pyc_path_source = os.path.join(pycache_directory, pyc_file)
        pyc_path_destination = os.path.join(sandbox_path, 'submit.pyc')
        shutil.copy(pyc_path_source, pyc_path_destination)
    except FileNotFoundError:
        pass


def create_java_jar(sandbox_path):
    # We require there be only a *single* class in the submit file.
    # However, there can be nested/private classes. When compiled, these get
    # a name in the form "X$Y", where X is the outer class and Y is the inner
    # class (this means class without $ should be the main one).
    class_files = glob.glob(os.path.join(sandbox_path, '*.class'))
    if len(class_files) == 0:
        return
    nested_class_files = glob.glob(os.path.join(sandbox_path, '*$*.class'))
    main_class_name = list(set(class_files) - set(nested_class_files))[0]
    main_class_name = main_class_name[:-6]

    manifest_file_path = os.path.join(TEMP_DIRECTORY, 'Manifest.txt')
    with open(manifest_file_path, 'w') as manifest:
        manifest.write(f'Main-Class: {main_class_name}\n')

    create_jar_command = ['jar', 'cf', 'submit.jar', manifest_file_path]
    create_jar_command += class_files
    subprocess.call(create_jar_command)


def copy_executable(sandbox_path):
    copy_python_bytecode(sandbox_path)
    create_java_jar(sandbox_path)


def custom_compile_module(task_directory):
    custom_compile_module_path = os.path.join(task_directory, 'custom_compile.py')
    loader = importlib.machinery.SourceFileLoader(
        'custom_compile',
        custom_compile_module_path
    )
    custom_compile_module = types.ModuleType(loader.name)
    loader.exec_module(custom_compile_module)
    return custom_compile_module


def trim_compiler_output(output):
    if len(output) > MAX_COMPILER_OUTPUT_LENGTH:
        return output[:MAX_COMPILER_OUTPUT_LENGTH] \
            + '\n\n(Compiler output too long, truncated.)\n'
    return output


def write_compiler_output(output):
    with open(COMPILATION_LOG_PATH, 'w') as compiler_log:
        compiler_log.write(output)


def compile_submit(sandbox_path):
    language = languages[SUBMIT_LANGUAGE]
    if not language.requires_compilation:
        return True

    logging.basicConfig(level=logging.INFO)

    task_config_path = os.path.join(TASK_DIRECTORY, 'task.config')
    task_config = read_config(task_config_path)
    if is_task_custom_compile(task_config):
        return custom_compile_module(TASK_DIRECTORY).run()

    submit_filename = f'submit.{language.extension}'
    sandbox_submit_file_path = os.path.join(sandbox_path, submit_filename)

    shutil.copy(SUBMIT_FILE_PATH, sandbox_submit_file_path)

    # FIXME z metadat zistit co vsetko este nakopirovat do sandboxu a nakopirovat to tam
    # FIXME z metadat zistit ci nemame custom commands na kompilaciu

    command = isolate.compilation_command(language)
    compiler_output, error = execute_external(command)
    copy_executable(sandbox_path)

    compiler_output = str(compiler_output, 'utf-8', errors='replace')
    compiler_output = trim_compiler_output(compiler_output)
    write_compiler_output(compiler_output)

    if error != 0:
        logging.info('Final result: --> \033[1;31mCE\033[0;37m <--')
        return False
    return True
