/*
 * Produce a pdf of a cubic graph with any edge that belongs to a pair of removable edges coloured in green.
 * The label of the edge is the number of removable pairs that contain it.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>

#include "jmgraph.h"


void incorrectParameters() {
    fprintf(stderr, "Incorrect command line parameters, use 'showremovable <g6/ba> <filename> <graph index (starting from 1)> [first red vertex]'\n");
    exit(1);
}


int **computeRemovableCounts(jmgraph *g) {
    int **counts = malloc(sizeof(int *) * g->n);
    for (int v = 0; v < g->n; v++) {
        counts[v] = malloc(sizeof(int) * g->maxDegree);        
        for (int i = 0; i < g->maxDegree; i++)
            counts[v][i] = 0;
    }
    for (int v = 0; v < g->n; v++) {
        printf("%d ", v); fflush(stdout);
        for (int i = 0; i < g->maxDegree; i++) {
            for (int u = v+1; u < g->n; u++) {
                for (int j = 0; j < g->maxDegree; j++) {
                    if ((g->e[v][i] == u) || (g->e[u][j] == v) || (g->e[u][j] == g->e[v][i]))
                        continue;
                    edgePair p;
                    p.v = v; p.i = i;
                    p.u = u; p.j = j;
                    if (isRemovableEdgePair(g, &p)) {
                        counts[v][i]++;
                        counts[u][j]++;
                    }
                }
            }                
        }
    }
    printf("\n");
    return counts;
}


void destroyRemovableCounts(jmgraph *g, int **removableCounts) {
    for (int v = 0; v < g->n; v++)
        free(removableCounts[v]);
    free(removableCounts);
}


char *neatoRepresentation(jmgraph *g, int **removableCounts, int firstRedVertex) {
    char s[1000000]; s[0] = '\0';
    strcpy(s, "graph {\nmaxiter=1000;\nmclimit=100;\noverlap = false;\nsplines = true;\nsep=.3;\nnode[margin=0, fontsize=12, shape=circle, height=.3, width=.3];\n");
    char *t = s + strlen(s);

    if (firstRedVertex < g->n) {
        char *a = "{node [color=red, style=filled]";
        strcpy(t, a); t += strlen(a);
        for (int v = firstRedVertex; v < g->n; v++) {
            char x[100];
            sprintf(x, "%d ", v);
            strcpy(t, x); t += strlen(x);
        }
        a = ";}\n";
        strcpy(t, a); t += strlen(a);
    }

    for (int v = 0; v < g->n; v++) {
        for (int i = 0; i < g->maxDegree && g->e[v][i] != -1; i++) {
            int u = g->e[v][i];
            if (u > v) {
                char a[1000]; a[0] = '\0';
                if (removableCounts[v][i] > 0) {
                    sprintf(a, "%d -- %d [label = \"%d\", fontsize = 10; color = green];\n",
                            v, u, removableCounts[v][i]);
                } else {
                    sprintf(a, "%d -- %d [color = black];\n", v, u);
                }
                strcpy(t, a);
                t += strlen(a);
            }
        }
    }
    strcpy(t, "}\n");
    return strdup(s);
}


int main(int argc, char **argv) {
    if (argc < 4)
        incorrectParameters();

    char *filename = argv[2];
    int graphNumber = strtol(argv[3], NULL, 10) - 1;

    int nGraphs;
    jmgraph **graphs;
    if (!strcasecmp(argv[1], "g6"))
        graphs = readGraphsFromFileG6(filename, &nGraphs);
    else if (!strcasecmp(argv[1], "ba"))
        graphs = readGraphsFromFileG6(filename, &nGraphs);
    else
        incorrectParameters();

    if ((graphNumber < 0) || (graphNumber >= nGraphs)) {
        fprintf(stderr, "There is no such graph in the input file\n");
        exit(1);
    }

    jmgraph *g = graphs[graphNumber];
    int firstRedVertex = (argc >= 5) ? strtol(argv[4], NULL, 10) : g->n + 1;

    int **removableCounts = computeRemovableCounts(g);
    char *r = neatoRepresentation(g, removableCounts, firstRedVertex);
    destroyRemovableCounts(g, removableCounts);
    printf("%s", r);
    char command[1000000];
    sprintf(command, "echo \"%s\" | neato -Tpdf > showremovable_%s_%d.pdf", r, filename, graphNumber);
    free(r);

    system(command);

    return 0;
}
