// Graph.cpp: implementation of the CGraph class.
//
//////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include "VRP.h"
#include "Graph.h"
#include "Globals.h"

#include <sstream>
#include <iostream>
#include <fstream>
#include <string>

using namespace std;

#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

CGraph::CGraph()
{
	// graph nodes
	m_pNodeArray = NULL;
	m_nNodeArrayLength = 0;
	// graph edges
	m_pEdgeArray = NULL;
	m_nEdgeArrayLength = 0;
	// graph routes
	m_pRouteArray = NULL;
	m_nRouteArrayLength = 0;

	m_pSlctNode = NULL;
	m_nCapacity = 100;
}

CGraph::~CGraph()
{
	if (m_pNodeArray != NULL) delete[] m_pNodeArray;
	if (m_pEdgeArray != NULL) delete[] m_pEdgeArray;
	if (m_pRouteArray != NULL) delete[] m_pRouteArray;
}

void CGraph::Serialize(CArchive &ar)
{
	if (ar.IsStoring()) {
		ar << m_nNodeArrayLength << m_nCapacity;	
	}
	else {
		ar >> m_nNodeArrayLength >> m_nCapacity;
		m_pNodeArray = new CNode*[m_nNodeArrayLength];
		for (int i = 0; i < m_nNodeArrayLength;i++) {
			m_pNodeArray[i] = new CNode;
		}
	}
	for (int i = 0; i < m_nNodeArrayLength; i++) {
		m_pNodeArray[i]->Serialize(ar);
	}
}

// add node to node array on the end
void CGraph::addNodeTail(CNode* pNode)
{
	CNode** tempArray; 
	tempArray = new CNode*[m_nNodeArrayLength+1];

	for (int i = 0; i < m_nNodeArrayLength; i++) {
		tempArray[i] = m_pNodeArray[i];	
	}
	m_pNodeArray = tempArray;
	pNode->setId(m_nNodeArrayLength);
	m_pNodeArray[m_nNodeArrayLength] = pNode;
	m_nNodeArrayLength += 1;
	CreateEdgeTable();
}

void CGraph::Show(CDC *pDC)
{
	if (m_nRouteArrayLength > 0) RoutesToEdges();

	for (int i = 0; i < m_nEdgeArrayLength; i++) {
		for (int j = 0; j < m_nEdgeArrayLength; j++) {
			m_pEdgeArray[i][j]->Show(pDC);
		}
	}

	for (i = 0; i < m_nNodeArrayLength; i++) {
		m_pNodeArray[i]->Show(pDC);
	}
	
}

CRect CGraph::getGraphRect()
{
	return m_GraphRect;
}

void CGraph::UpdateGraphRect()
{
	CPoint point;
	if (m_nNodeArrayLength > 0) {
		m_GraphRect.SetRect(m_pNodeArray[0]->getLocation(), 
						    m_pNodeArray[0]->getLocation()); 
	}
	for (int i = 0; i < m_nNodeArrayLength; i++) {
		point = m_pNodeArray[i]->getLocation();
	if (m_GraphRect.TopLeft().x >= point.x) 
		m_GraphRect.TopLeft().x = point.x; 
	if (m_GraphRect.TopLeft().y >= point.y) 
		m_GraphRect.TopLeft().y = point.y; 
	if (m_GraphRect.BottomRight().x <= point.x) 
		m_GraphRect.BottomRight().x = point.x; 
	if (m_GraphRect.BottomRight().y <= point.y) 
		m_GraphRect.BottomRight().y = point.y; 
	}
}

CNode* CGraph::SelectNode(CPoint point)
{
	CNode* node = NULL;
	BOOL selected = false;
	m_pSlctNode = NULL;
	for (int i=0; i < m_nNodeArrayLength; i++) {
		m_pNodeArray[i]->setSelected(false);
		if ((m_pNodeArray[i]->isLocation(point)) && (!selected)){
			m_pNodeArray[i]->setSelected(true);
			node = m_pNodeArray[i]; 
			m_pSlctNode = node;
			selected = true;
		}
	}
	return node;
}

CNode* CGraph::getSlctNode()
{
	return m_pSlctNode;
}

BOOL CGraph::LoadFromFile(CString filename)
{
	
	BOOL Ok = true;
	string line, keyword, value;
		
	int separator0, separator1, state = 0;

	int index, demand;
	CPoint location;


	ifstream file (filename);
	if (file.is_open()) {
		while (! file.eof() ) {
			getline (file,line);
			
			if (line.compare("NODE_COORD_SECTION ") == 0) {
				state = 1;				 
				m_pNodeArray = new CNode*[m_nNodeArrayLength];
				getline (file,line);
			} else if (line.compare("DEMAND_SECTION ") == 0) {
				state = 2;
				getline (file,line);
			} else if (line.compare("DEPOT_SECTION ") == 0) {
				state =3;
				getline (file,line);
			}
			
			switch (state) {
				case 0: 
					separator0 = line.find(":");
					keyword = line.substr(0,separator0+1);
					value = line.substr(separator0+1);	
					if (keyword.compare("NAME :") == 0) m_sName = value.c_str();
					else if (keyword.compare("COMMENT :") == 0) m_sComment = value.c_str(); 
					else if (keyword.compare("TYPE :") == 0) m_sType = value.c_str();
					else if (keyword.compare("DIMENSION :") == 0) m_nNodeArrayLength = g_AppData->StringToInt(value);
					else if (keyword.compare("CAPACITY :") == 0) m_nCapacity = g_AppData->StringToInt(value);
					break;
				case 1: 
					separator0 = line.find(" ",1);
					separator1 = line.find(" ",separator0+1);

					index = g_AppData->StringToInt(line.substr(1,separator0+1));
					location.x = g_AppData->StringToInt(line.substr(separator0+1,separator1+1));
					location.y = g_AppData->StringToInt(line.substr(separator1+1));							
					m_pNodeArray[index-1] = new CNode;
					m_pNodeArray[index-1]->setLocation(location);
					m_pNodeArray[index-1]->setId(index-1);
					break;
				case 2:
					separator0 = line.find(" ",0);
					index = g_AppData->StringToInt(line.substr(0,separator0+1));
					demand = g_AppData->StringToInt(line.substr(separator0+1));
					
					m_pNodeArray[index-1]->setDemand(demand);
					break;
				case 3:
			//		location.x = StringToInt(line);
					getline (file,line);
			//		location.y = StringToInt(line);
			//		m_pNodeArray[0].setLocation(location);
			//		m_pNodeArray[0].setId(0);
					getline (file,line);
					break;

			};

		};
		file.close();
		CreateEdgeTable();
		return true;
	} else return false;
}

CString CGraph::GetName()
{
	return m_sName;
}

int CGraph::GetNodeCount()
{
	return m_nNodeArrayLength;
}

float CGraph::GetEdgeValue(int i ,int j)
{
	if (m_pEdgeArray[i][j] != NULL) {
		return m_pEdgeArray[i][j]->getValue();
	}
	else return 0;
}

// crete selected number of routes, destroy all exist routes
void CGraph::CreateRoutes(int count)
{
	// delete routes if exist
	if (m_nRouteArrayLength > 0) {
		for (int i=0; i < m_nRouteArrayLength; i++) {
			delete m_pRouteArray[i];
		}

	} 
	// crate new routes
	m_pRouteArray = new CRoute*[count];
	m_nRouteArrayLength = count;
	for (int i=0; i < count; i++) {
			m_pRouteArray[i] = new CRoute;
	}
}

void CGraph::SetRouteId(int route, int id)
{
	if ((route >= 0) && (m_nRouteArrayLength > route)) {
		m_pRouteArray[route]->setId(id);
	}
}

void CGraph::SetRouteState(int route, int state)
{
	if ((route >= 0) && (m_nRouteArrayLength > route)) {
		m_pRouteArray[route]->setState(state);
	}
}

void CGraph::SetRouteCost(int route, int cost)
{
	if ((route >= 0) && (m_nRouteArrayLength > route)) {
		m_pRouteArray[route]->setCost(cost);
	}
}

void CGraph::SetRouteDemand(int route, int demand)
{
	if ((route >= 0) && (m_nRouteArrayLength > route)) {
		m_pRouteArray[route]->setDemand(demand);
	}
}

int CGraph::GetNodeDemand(int node)
{
	if ((node >= 0) && (m_nNodeArrayLength > node)) {
		return m_pNodeArray[node]->getDemand();
	}	else return 0;
}

void CGraph::AddNodeToRouteTail(int route,CNode *node)
{
	if ((route >= 0) && (m_nRouteArrayLength > route)) {
		m_pRouteArray[route]->AddNodeTail(node);
		node->setRouteId(route);
	}	
}

CNode* CGraph::GetNode(int index)
{
	if ((index >= 0) && (m_nNodeArrayLength > index)) {
		return m_pNodeArray[index];
	}	else return NULL;
}

void CGraph::RoutesToEdges()
{

	// clear all edges
	for (int i = 0; i < m_nNodeArrayLength; i++) {
		for (int j = 0; j < m_nNodeArrayLength; j++) {
			m_pEdgeArray[i][j]->setState(0);
		}
	}

	// proces routes 
	for (i = 0;i < m_nRouteArrayLength;i++) {
		if (m_pRouteArray[i]->getLength() > 1) {
			for (int j = 0; j < m_pRouteArray[i]->getLength()-1;j++) {
				m_pEdgeArray[m_pRouteArray[i]->getNodeFromPos(j)->getId()][m_pRouteArray[i]->getNodeFromPos(j+1)->getId()]->setState(1);
				m_pEdgeArray[m_pRouteArray[i]->getNodeFromPos(j)->getId()][m_pRouteArray[i]->getNodeFromPos(j+1)->getId()]->SetRouteId(i);
			}
		}
	}
}


void CGraph::CreateEdgeTable()
{
	// delete old edges if exists
	if (m_pEdgeArray != NULL) {
		for (int i = 0; i < m_nEdgeArrayLength; i++) {
			for (int j = 0; j < m_nEdgeArrayLength; j++) {
				delete m_pEdgeArray[i][j];
			}
		}
	}
	m_nEdgeArrayLength = m_nNodeArrayLength;
	m_pEdgeArray = new CEdge**[m_nEdgeArrayLength];
	for (int i = 0; i < m_nEdgeArrayLength; i++) {
		m_pEdgeArray[i] = new CEdge*[m_nEdgeArrayLength];
		for (int j = 0; j < m_nEdgeArrayLength; j++) {
			m_pEdgeArray[i][j] = new CEdge(m_pNodeArray[i],m_pNodeArray[j]);
		}
	}
}

// try to merge routes fr and sr by adding edge (fn,sn)
void CGraph::MergeRoutes(int fr, int sr, int fn, int sn)
{

/*	if ((m_pRouteArray[fr]->GetLastCustId() == fn) && (m_pRouteArray[sr]->GetFirstCustId()) == sn)
		MergeRoutesLF(fn,sn);
	else if ((m_pRouteArray[fr]->GetFirstCustId() == fn) && (m_pRouteArray[sr]->GetLastCustId()) == sn) 
		MergeRoutesFL(fn,sn);
	else if ((m_pRouteArray[fr]->GetFirstCustId() == fn) && (m_pRouteArray[sr]->GetFirstCustId()) == sn) 
		MergeRoutesFF(fn,sn);
	else if ((m_pRouteArray[fr]->GetLastCustId() == fn) && (m_pRouteArray[sr]->GetLastCustId()) == sn) 
		MergeRoutesLL(fn,sn);

*/

}


// merge routes in by edge (fn,sn)
void CGraph::MergeRoutesFS(int fr, int sr, int fn, int sn)
{
	BOOL test = true;
	
	// count new route length
	int fIndex = m_pRouteArray[fr]->GetIndex(fn);
	int sIndex = m_pRouteArray[sr]->GetIndex(sn); 
	
	if (fIndex == -1) fIndex = m_pRouteArray[fr]->getLength()-1;
	if (sIndex == -1) sIndex = 0;

	int fPartLength = fIndex+1;
	int sPartLength = m_pRouteArray[sr]->getLength() - sIndex;
	
	m_pRouteArray[fr]->setLength(fPartLength + sPartLength);
	
	// insert all nodes from second route to enlarged first route
	for (int i = 0; i < sPartLength; i++) {
		m_pRouteArray[fr]->setNodeToPos((fIndex+1)+i,m_pRouteArray[sr]->getNodeFromPos(sIndex+i));
	}
		
	m_pRouteArray[fr]->UpdateDemand();
	m_pRouteArray[sr]->setLength(0);
}

void CGraph::MergeRoutesSF(int fr, int sr, int fn, int sn)
{
	int fIndex = m_pRouteArray[fr]->GetIndex(fn);
	int sIndex = m_pRouteArray[sr]->GetIndex(sn); 
	
	if (fIndex == -1) fIndex = 0;
	if (sIndex == -1) sIndex = m_pRouteArray[fr]->getLength()-1;

	int fPartLength = m_pRouteArray[fr]->getLength() - fIndex;
	int sPartLength = sIndex+1;
	
	CRoute* tempRoute = new CRoute(m_pRouteArray[fr]);
	tempRoute->setLength(fPartLength + sPartLength);
	// copy first part
	for (int i = 0;i < fPartLength;i++) {
		tempRoute->setNodeToPos(i,m_pRouteArray[fr]->getNodeFromPos((m_pRouteArray[fr]->getLength()-1)-i));
	}
	
	// copy second part
	for (i = 0;i < sPartLength;i++) {
		tempRoute->setNodeToPos(fPartLength+i,m_pRouteArray[sr]->getNodeFromPos((sPartLength-1)-i));
	}
		
	m_pRouteArray[sr]->setLength(0);
	delete m_pRouteArray[fr];
	m_pRouteArray[fr] = tempRoute;
	m_pRouteArray[fr]->UpdateDemand();


}

void CGraph::MergeRoutesFF(int fr, int sr, int fn, int sn)
{
	int fIndex = m_pRouteArray[fr]->GetIndex(fn);
	int sIndex = m_pRouteArray[sr]->GetIndex(sn); 
	
	if (fIndex == -1) fIndex = m_pRouteArray[fr]->getLength()-1;
	if (sIndex == -1) sIndex = m_pRouteArray[sr]->getLength()-1;

	int fPartLength = fIndex+1;
	int sPartLength = sIndex+1;
	
	CRoute* tempRoute = new CRoute(m_pRouteArray[fr]);
	tempRoute->setLength(fPartLength + sPartLength);
		
	// copy first part
	for (int i = 0;i < fPartLength;i++) {
		tempRoute->setNodeToPos(i,m_pRouteArray[fr]->getNodeFromPos(i));
	}
	
	// copy second part
	for (i = 0;i < sPartLength;i++) {
		tempRoute->setNodeToPos(fPartLength+i,m_pRouteArray[sr]->getNodeFromPos((sPartLength-1)-i));
	}

	m_pRouteArray[sr]->setLength(0);
	delete m_pRouteArray[fr];
	m_pRouteArray[fr] = tempRoute;
	m_pRouteArray[fr]->UpdateDemand();
}

void CGraph::MergeRoutesSS(int fr, int sr, int fn, int sn)
{
	int fIndex = m_pRouteArray[fr]->GetIndex(fn);
	int sIndex = m_pRouteArray[sr]->GetIndex(sn); 
	
	if (fIndex == -1) fIndex = 0;
	if (sIndex == -1) sIndex = 0;

	int fPartLength = m_pRouteArray[fr]->getLength() - fIndex;
	int sPartLength = m_pRouteArray[sr]->getLength() - sIndex;
	
	CRoute* tempRoute = new CRoute(m_pRouteArray[fr]);
	tempRoute->setLength(fPartLength + sPartLength);
		
	// copy first part
	for (int i = 0;i < fPartLength;i++) {
		tempRoute->setNodeToPos(i,m_pRouteArray[fr]->getNodeFromPos((m_pRouteArray[fr]->getLength()-1)-i));
	}
	
	// copy second part
	for (i = 0;i < sPartLength;i++) {
		tempRoute->setNodeToPos(fPartLength+i,m_pRouteArray[sr]->getNodeFromPos(sIndex+i));
	}

	m_pRouteArray[sr]->setLength(0);
	delete m_pRouteArray[fr];
	m_pRouteArray[fr] = tempRoute;
	m_pRouteArray[fr]->UpdateDemand();
}

CRoute* CGraph::GetRoute(int id)
{
	return m_pRouteArray[id];
}

int CGraph::GetCapacity()
{
	return m_nCapacity;
}

void CGraph::UpdateEdgeValues()
{
	for (int i=0; i < m_nEdgeArrayLength;i++) {
		for (int j=i+1; j < m_nEdgeArrayLength;j++) {
			float value = ComputeEuclidianDistance(
				m_pEdgeArray[i][j]->GetFirstLocation(),
				m_pEdgeArray[i][j]->GetSecondLocation());
			m_pEdgeArray[i][j]->setValue(value);
			m_pEdgeArray[j][i]->setValue(value);
		}
	}
}

float CGraph::ComputeEuclidianDistance(CPoint p1, CPoint p2)
{
	return sqrt(pow(p1.x - p2.x,2) + pow(p1.y - p2.y,2));
}

int CGraph::GetRouteDemand(int id)
{
	return m_pRouteArray[id]->getDemand();
}

void CGraph::IncRouteDemand(int route, int inc)
{
	if ((route >= 0) && (m_nRouteArrayLength > route)) {
		int demand = m_pRouteArray[route]->getDemand();
		m_pRouteArray[route]->setDemand(demand+inc);
	}
}

void CGraph::DecRouteDemand(int route, int dec)
{
	if ((route >= 0) && (m_nRouteArrayLength > route)) {
		int demand = m_pRouteArray[route]->getDemand();
		m_pRouteArray[route]->setDemand(demand-dec);
	}
}

int CGraph::GetRouteCount()
{
	return m_nRouteArrayLength;
}

int CGraph::GetNodeState(int id)
{
	return m_pNodeArray[id]->GetState();
}

void CGraph::AddRoute()
{
	CRoute** tempRouteArray;
	CRoute* newRoute;
	newRoute = new CRoute();
	newRoute->setId(m_nRouteArrayLength);
	tempRouteArray = new CRoute*[m_nRouteArrayLength+1];

	for (int i=0; i < m_nRouteArrayLength; i++) {
		tempRouteArray[i] = new CRoute(m_pRouteArray[i]);	
	}
	
	tempRouteArray[m_nRouteArrayLength] = newRoute;
	m_nRouteArrayLength += 1;
	m_pRouteArray = tempRouteArray;
}

float CGraph::CountRouteCost(int RouteId)
{
	CRoute* route = GetRoute(RouteId);
	float cost = 0;
	for (int i=1;i < route->getLength();i++) {
		cost += GetEdgeValue(route->GetNodeId(i-1),route->GetNodeId(i));
	}
	return cost;
}

void CGraph::SetCapacity(int cap)
{
	m_nCapacity = cap;
}

void CGraph::CreateNodeArray(int length)
{
	m_pNodeArray = new CNode*[length];
	m_nNodeArrayLength = length;
}

void CGraph::AddNodeToPos(int index, CNode *node)
{
	m_pNodeArray[index] = node;
}

void CGraph::ClearEdgesState()
{
	if (m_nEdgeArrayLength > 0) {
		for (int i = 0; i < m_nNodeArrayLength; i++) {
			for (int j = 0; j < m_nNodeArrayLength; j++) {
				m_pEdgeArray[i][j]->setState(0);
			}
		}
	}
}
