namespace("org.dforms.validation");
depends("org.dforms.dom");
with(org.dforms.validation){
	org.dforms.validation.NamespaceURI="http://www.dforms.org/validation/1.0";
	org.dforms.validation._attributes=[
	"type",	"required", 
	"minLength", "maxLength", "lengthUnit", "pattern",
	"minInclusive", "minExclusive", "maxInclusive", "maxExclusive", "fractionDigits", "totalDigits",
	"message",
	"validate",
	"onChildSuccess", "onChildFailure",
	"onSuccess", "onFailure"];
	/**
	 * Event type system.
	 */
	org.dforms.validation.Event=function(ctx){
		if(ctx){
			this.source=ctx.element;
		}
		this.context=ctx;
	};
	Event.fireEvent=function(event, ctx){
		ctx=ctx||event.context
		var handler="on"+event.type;
		if(ctx[handler]){
			ctx[handler](ctx.element, event);
		}else if(ctx.parent){
			Event.fireEvent(event, ctx.parent);
		}
	};
	/**
	 * Child success validation event.
	 * The event is fired each time the validation of child component has succeeded.
	 * @param ctx validation context
	 */
	org.dforms.validation.ChildSuccessEvent=function(ctx){
		Event.call(this, ctx);
		this.type="ChildSuccess";
	};
	ChildSuccessEvent.protype=new Event();
	ChildSuccessEvent.protype.constructor=ChildSuccessEvent;
	/**
	 * Child failure validation event.
	 * The event is fired each time the validation of child component has failed.
	 * @param ctx validation context
	 * @param message validation messages describing failure reasons
	 */
	org.dforms.validation.ChildFailureEvent=function(ctx, messages){
		ChildSuccessEvent.call(this, ctx);
		this.type="ChildFailure";
		this.messages=messages;
	};
	ChildFailureEvent.protype=new ChildSuccessEvent();
	ChildFailureEvent.protype.constructor=ChildFailureEvent;
	/**
	 * Success validation event.
	 * The event is fired each time the validation of the component itself has succeeded.
	 * @param ctx validation context
	 */
	org.dforms.validation.SuccessEvent=function(ctx){
		Event.call(this, ctx);
		this.type="Success";
	};
	SuccessEvent.protype=new Event();
	SuccessEvent.protype.constructor=SuccessEvent;
	/**
	 * Failure validation event.
	 * The event is fired each time the validation of the component itself has failed.
	 * @param ctx validation context
	 * @param events array of child failure events, which caused the failure
	 */
	org.dforms.validation.FailureEvent=function(ctx, events){
		SuccessEvent.call(this, ctx);
		this.events=events;
		this.type="Failure";
	};
	FailureEvent.protype=new SuccessEvent();
	FailureEvent.protype.constructor=FailureEvent;
	/**
	 *
	 */
	org.dforms.validation.Context=function(element, parent){
		this.element=element;
		if(element.nodeName.toLowerCase()==="select"){
			this.value=element.options[element.selectedIndex].value;
		}else{
			this.value=element.value;
		}
		if(this.value==null)this.value="";
		this.parent=parent;
		this.children=[];
//		this.validate=validate;
		if(parent){
			this.validate=parent.validate;
			parent.children.push(this);
		}
		this.ns='v'; //getNamespacePrefixForURI(element, NamespaceURI);
		for(var j=0;j<_attributes.length;j++){
			var name=_attributes[j];
			var value=element.getAttributeNode(this.ns+":"+name);
			if(value){
				value=value.value;
				if(name.indexOf("on")==0 || name=="validate"){
					value=new Function("element", "event",value);
				}
				this[name]=value;
			}
		}
		var idx=0;
		for(var i in this){
			idx++;
		}
	};
	Context.prototype.validate=function(){
		var result=[];
		var children=this.children;
		if(children.length==0 && this.type){
			var event=this.validateImpl();
			if(event){
				Event.fireEvent(event);
				if(event.type=="ChildFailure"){
					result.push(event);
				}
			}else{
				alert(event);
			}
		}else{
			var success=true;
			for(var i=0;i<children.length;i++){
				var childContext=children[i];
				var res=childContext.validate();
				if(res.length != 0){
					result.push(new FailureEvent(childContext, res));
					success=false;
				}
			}
			if(success){
				Event.fireEvent(new SuccessEvent(this));
			}else{
				Event.fireEvent(new FailureEvent(this, result));
			}
		}
		return result;
	};
	Context.prototype.validateImpl=function(){
		var ret;
		switch(this.type){
			case 'string':
				ret=this.validateString();
				break;
			case 'integer':
				ret=this.validateInteger();
				break;
			case 'decimal':
				ret=this.validateDecimal();
				break;
			case 'boolean':
				ret=this.validateBoolean();
				break;
			case 'date':
				ret=this.validateDate();
				break;
			case 'choice':
				ret=this.validateChoice();
				break;
		}
		return ret;
	};
	/**
	 * Validates form element holding string against validation attributes.
	 * @param element form element
	 * @param ctx validation context
	 */
	Context.prototype.validateString=function(){
		try{
			var element=this.element;
			var value=this.value;
			if(value==null)value="";
			var required=this.required?this.required=='true':false;
			var lengthUnit=this.lengthUnit?this.lengthUnit:'character';
			var minLength=this.minLength?this.minLength:0;
			var maxLength=this.maxLength;
			var length=Utils.getStringLength(value, lengthUnit);
			var pattern=this.pattern;
			var unitName=lengthUnit=="character"?"znakov":"bytov";
			var messages=[];
			if(required && value == ''){
				messages.push("Text nemôže byť prázdny.");
			}
			if(length < minLength){
				messages.push("Text musí byť dlhší alebo rovný ako "+minLength+" "+unitName+".");
			}
			if(maxLength && length > maxLength){
				messages.push("Text musí byť kratší alebo rovný ako "+maxLength+" "+unitName+".");
			}
			if(pattern && value && value != ''){
				var p=new RegExp('^'+pattern+'$');
//				alert("value: "+value+"\npattern: "+pattern+"\nmatches: "+p.test(value));
				if(!value.match(p)){
					messages.push("Text musí zodpovedať regulárnemu výrazu '"+pattern+"'.");
				}
			}
		}catch(e){	// ignore
			alert(e.name+': '+e.message);
		}
		var ret;
		if(messages.length==0){
			ret=new ChildSuccessEvent(this);
		}else{
			ret=new ChildFailureEvent(this, this.message?[this.message]:messages);
		}
		return ret;
	};
	/**
	 * Validates form element holding integer representation against validation attributes.
	 * @param ctx map of validation attributes
	 */
	Context.prototype.validateInteger=function(){
		var element=this.element;
		var value=Number(this.value);
		var required=this.required?this.required=='true':false;
		var messages=[];
		if(required && (element.value == '' || element.value == null)){
			var label=Utils.getLabelTextFor(element.getAttribute('id'));
			var labelName=label==null?'':"'"+label+"' ";
			messages.push("Pole nemôže byť prázdne.");
		}
		if(!isNaN(value) && (value - Math.floor(value)) == 0){
			var minInclusive=Number(this.minInclusive);
			var maxInclusive=Number(this.maxInclusive);
			var minExclusive=Number(this.minExclusive);
			var maxExclusive=Number(this.maxExclusive);
			var totalDigits=Number(this.totalDigits);
			var fractionDigits=Number(this.fractionDigits);
			if(!isNaN(totalDigits) && Utils.getMaxPlaceValue(value) > totalDigits){
				messages.push("Hodnota musí mať nanajvyš "+totalDigits+" číslic.");
			}
			if(!isNaN(minInclusive) && (value < minInclusive)){
				messages.push("Hodnota musí byť vačšia alebo rovná ako "+minInclusive+".");
			}
			if(!isNaN(minExclusive) && value <= minExclusive){
				messages.push("Hodnota musí byť vačšia ako "+minExclusive+".");
			}
			if(!isNaN(maxInclusive) && value > maxInclusive){
				messages.push("Hodnota musí byť menšia alebo rovná ako "+maxInclusive+".");
			}
			if(!isNaN(maxExclusive) && value >= maxExclusive){
				messages.push("Hodnota musí byť menšia ako "+maxExclusive+".");
			}
		}else{
			messages.push("Hodnota musí byť celé číslo.");
		}
		var ret;
		if(messages.length==0){
			ret=new ChildSuccessEvent(this);
		}else{
			ret=new ChildFailureEvent(this, this.message?[this.message]:messages);
		}
		return ret;
	};
	/**
	 * Validates form element holding decimal representation against validation attributes.
	 * @param ctx map of validation attributes
	 */
	Context.prototype.validateDecimal=function(){
		var element=this.element;
		var value=Number(this.value);
		var required=this.required?this.required=='true':false;
		var messages=[];
		if(required && (element.value == '' || element.value == null)){
			messages.push("Pole nemôže byť prázdne.");
		}
		if(!isNaN(value)){
			var minInclusive=Number(this.minInclusive);
			var maxInclusive=Number(this.maxInclusive);
			var minExclusive=Number(this.minExclusive);
			var maxExclusive=Number(this.maxExclusive);
			var totalDigits=Number(this.totalDigits);
			var fractionDigits=Number(this.fractionDigits);
			if(!isNaN(fractionDigits) && (Utils.getFractionDigits(value) > fractionDigits)){
				messages.push("Hodnota musí mať počet desatinných miest vačší alebo rovný ako "+fractionDigits+".");
			}
			if(!isNaN(totalDigits) && Utils.getMaxPlaceValue(value)+Utils.getFractionDigits(value) > totalDigits){
				messages.push("Hodnota musí mať nanajvyš "+totalDigits+" číslic.");
			}
			if(!isNaN(minInclusive) && (value < minInclusive)){
				messages.push("Hodnota musí byť vačšia alebo rovná ako "+minInclusive+".");
			}
			if(!isNaN(minExclusive) && value <= minExclusive){
				messages.push("Hodnota musí byť vačšia ako "+minExclusive+".");
			}
			if(!isNaN(maxInclusive) && value > maxInclusive){
				messages.push("Hodnota musí byť menšia alebo rovná ako "+maxInclusive+".");
			}
			if(!isNaN(maxExclusive) && value >= maxExclusive){
				messages.push("Hodnota musí byť menšia ako "+maxExclusive+".");
			}
		}else{
			messages.push("Hodnota musí byť celé číslo.");
		}
		var ret;
		if(messages.length==0){
			ret=new ChildSuccessEvent(this);
		}else{
			ret=new ChildFailureEvent(this, this.message?[this.message]:messages);
		}
		return ret;
	};
	Context.prototype.validateBoolean=function(){
		var element=this.element;
		return [];
	};
	Context.prototype.validateDate=function(){
		return [];
	};
	/**
	 * Validates the form element representing choice (select, radio set,...).
	 */
	Context.prototype.validateChoice=function(){
		try{
			var element=this.element;
			var value=this.value;
			var required=this.required?this.required=='true':false;
			var messages=[];
			if(required && value == ''){
				messages.push("Musíte si vybrať jednu z možností.");
			}
		}catch(e){	// ignore
			alert(e.name+': '+e.message);
		}
		var ret;
		if(messages.length==0){
			ret=new ChildSuccessEvent(this);
		}else{
			ret=new ChildFailureEvent(this, this.message?[this.message]:messages);
		}
		return ret;
	};
	/**
	 * Builds context tree of dforms items and panels.
	 */
	org.dforms.validation.ValidationTreeBuilter={};
	ValidationTreeBuilter.build=function(element, parent){
		var ctx=parent?parent:new Context(element);
		var ext=ValidationTreeBuilter.isExtending(element);
		var leaf=ValidationTreeBuilter.isLeaf(element);
		if(ext || leaf){
			ctx=new Context(element, ctx);
		}
		var childNodes=element.childNodes;
		for(var i=0;i<childNodes.length;i++){
			var node=childNodes[i];
			if(node.nodeType==1 && !node.disabled && !ValidationTreeBuilter.isTemplate(node)){	// element node, enabled, not in -d-multi template
				ValidationTreeBuilter.build(node, ctx);
			}
		}
		return ctx;
	};
	ValidationTreeBuilter.isTemplate=function(element){
		var id=element.getAttribute("id");
		return id!=null && id.indexOf("-template")!=-1
	}
	ValidationTreeBuilter.isExtending=function(element){
		return element.getAttributeNode("v:validate")||
		element.getAttributeNode("v:onChildSuccess")||
		element.getAttributeNode("v:onChildFailure")||
		element.getAttributeNode("v:onSuccess")||
		element.getAttributeNode("v:onFailure");
	};
	ValidationTreeBuilter.isLeaf=function(element){
		return element.getAttributeNode("v:type");
	};
	/**
	 *
	 */
	org.dforms.validation.Utils={};
	Utils.validate=function(element){
		var root=ValidationTreeBuilter.build(element);
		var result=root.validate();
		return result;
	};
	Utils.highlightLabel=function(id){
		var label=Utils.getLabelFor(id);
		if(label!=null){
			label.style.color="red";
		}
	};
	Utils.unhighlightLabel=function(id){
		var label=Utils.getLabelFor(id);
		if(label!=null){
			label.style.color="";
		}
	};
	Utils.showMessages=function(msgs, messages, addLabel){
		for(var i=0;i<messages.length;i++){
			var message=messages[i];
			var span=org.dforms.dom.Document.createElementNS("http://www.w3.org/1999/xhtml", "div");
			span.appendChild(document.createTextNode(message));
			span.className="dforms-m-error";
			msgs.appendChild(span);
		}
	};
	Utils.clearMessages=function(msgs){
		while (msgs.firstChild) {
		  msgs.removeChild(msgs.firstChild);
		}
	};
	Utils.getLabelFor=function(id){
		var labels=document.getElementsByTagName("label");
		for(var i=0; i<labels.length;i++){
			var label=labels[i];
			if(label.getAttribute("class") != "description" && label.htmlFor==id){
				return label;
			}
		}
		return null;
	};
	Utils.getLabelText=function(label){
		var ret='';
		if(label != null){
			var text='';
			var nodes=label.childNodes;
			for(var j=0;j<nodes.length;j++){
				var node=nodes[j];
				if(node.nodeType==3){// text node
					text=node.nodeValue;break;
				}
			}
			ret=text.indexOf("*")==0?text.substr(1):text;
		}
		return ret;
	};
	Utils.getLabelTextFor=function(id){
		var label=Utils.getLabelFor(id);
		return Utils.getLabelText(label);
	};
	Utils.addLabelTextTo=function(event){
		var labelText=Utils.getLabelTextFor(event.source.id);
		for(var i=0;i< event.messages.length;i++){
			event.messages[i]=labelText+": "+event.messages[i];
		}
	};
	Utils.getStringLength=function(string, lengthUnit){
		if(lengthUnit=="character"){
			return string.length;
		}else{	// byte
			var enters=0;
			for(var i=0;i<string.length;i++){
				if(string.charAt(i)=='\n'){
					enters++;
				}
			}
			string=encodeURI(string);
			string=unescape(string);
			return string.length+enters;
		}
	};
	/**
	 * Gets place value of the digit at highest order. 
	 */
	Utils.getMaxPlaceValue=function(number){
		var text=""+Math.abs(number)+".";
		return text.indexOf('.');
	};
	/**
	 * Gets fraction digits count for given number.
	 * @param number
	 */
	Utils.getFractionDigits=function(number){
		var text=""+number;
		var index=text.indexOf('.');
		if(index != -1){
			return text.substring(index).length-1;
		}
		return 0;
	};
}