#include "Process.h"

Process::Process(memory& shared, SymbolTable* functions, std::vector<CodeObject *> &codeObjects, std::istream& input, std::string fName)
	: shared(shared), functions(functions), codeObjects(codeObjects), input(input)
{
	frames.push(new frame(*this, *codeObjects[functions->getIndex(fName)], input));
}

Process::Process( memory& shared, SymbolTable* functions, std::vector<CodeObject *> &codeObjects, std::istream& input, frame* _frame)
	: shared(shared), functions(functions), codeObjects(codeObjects), input(input)
{
	frames.push(_frame);
}

Process::Process(memory& shared, SymbolTable* functions, std::vector<CodeObject *> &codeObjects, std::istream& input)
	: shared(shared), functions(functions), codeObjects(codeObjects), input(input)
{
}

Process::~Process() {
}

memObject* Process::getShared(unsigned int position) {
	return shared.get(position);
}

void Process::setShared(memObject* mObj, unsigned int position) throw(memAccessException) {
	shared.add(mObj, position);
}

void Process::doInstruction() throw(pramException) {
	processState = ps_ok;
	frame* currFrame = frames.top();

	currFrame->doInstruction();

	switch (currFrame->getFrameState()) {
		case fs_ok: break;
		case fs_ret: {
			memObject* retValue = currFrame->exprStackPop();
			frames.pop();
			delete currFrame;
			if (frames.size() == 0) {
				processState = ps_exit;
			} else {
				currFrame = frames.top();
				currFrame->exprStackPush(retValue);
			}
			break;
		}
		case fs_call: {
			int address = currFrame->code.getInstructionAt(currFrame->IP-1).second;
			if (codeObjects[address]->isParallel()) {
				callAddr = address;
				processState = ps_call_par;
				return;
			}
			frame* newFrame = new frame(*this, *codeObjects[address], input);
			for (unsigned int i = 0; i < newFrame->code.getNumArgs(); i++) {
				memObject* mObj = currFrame->exprStackPop();
				newFrame->exprStackPush(mObj);
			}
			frames.push(newFrame);
			break;
		}
		case fs_pardo: {
			int address = currFrame->code.getInstructionAt(currFrame->IP-1).second;
			memInteger* to = dynamic_cast<memInteger *>(currFrame->exprStackPop());
			memInteger* from = dynamic_cast<memInteger *>(currFrame->exprStackPop());
			std::vector<memObject *> args;
			for (int j = 0; j < currFrame->code.getPardoArgCount(); j++) {
				args.push_back(currFrame->exprStackPop());
			}
			if (to == NULL || from == NULL) {
				throw badOpException("Internal error: pardo called with non-integer arguments");
			}
			children.clear();
			for (int i = from->getValue(); i <= to->getValue(); i++) {
				frame* newFrame = new frame(*this, *codeObjects[address], input);
				newFrame->exprStackPush(new memInteger(i));
				for (int j = 0; j < args.size(); j++) {
					newFrame->exprStackPush(args[j]->getObj());
				}
				Process *p = new Process(shared, functions, codeObjects, input, newFrame);
				children.push_back(p);
			}
			for (int i = 0; i < args.size(); i++) {
				delete args[i];
			}
			processState = ps_child;
			break;
		}
		case fs_exit: {
			delete currFrame;
			frames.pop();
			if (frames.size() == 0) {
				processState = ps_exit;
			} else {
				currFrame = frames.top();
			}
			break;
		}
		case fs_if_false: {
			processState = ps_if_false;
			break;
		}
		case fs_if_true: {
			processState = ps_if_true;
			break;
		}
		case fs_sync: {
			processState = ps_sync;
			break;
		}
		case fs_sync_begin: {
			processState = ps_sync_begin;
			break;
		}
		case fs_setp: {
			processState = ps_setp;
			break;
		}
	}
}

void Process::run() {
	while (frames.top()->hasNexInstruction()) {
		doInstruction();
	}
}

SyncObj* Process::getSyncObj() {
	return syncObj;
}

void Process::setSyncObj(SyncObj* newSyncObj) {
	syncObj = newSyncObj;
}

ProcessState Process::getProcessState() {
	return processState;
}

std::vector<Process *>& Process::getChildren() {
	return children;
}

unsigned int Process::getCallAddr() {
	return callAddr;
}

void Process::pushFrame(frame* f) {
	frames.push(f);
}

memObject* Process::exprStackPop() {
	return frames.top()->exprStackPop();
}

