/* $Id: scanner.ll 44 2008-10-23 09:03:19Z tb $ -*- mode: c++ -*- */
/** \file scanner.ll Define the example Flex lexical scanner */

%{ /*** C/C++ Declarations ***/

#include <string>
#include <stdio.h>
#include "scanner.h"

/* import the parser's token type into a local typedef */
typedef example::Parser::token token;
typedef example::Parser::token_type token_type;

/* By default yylex returns int, we use token_type. Unfortunately yyterminate
 * by default returns 0, which is not of token_type. */
#define yyterminate() return token::END

/* This disables inclusion of unistd.h, which is not available under Visual C++
 * on Win32. The C++ scanner uses STL streams instead. */
#define YY_NO_UNISTD_H

%}

%x IN_COMMENT

/*** Flex Declarations and Options ***/

/* enable c++ scanner class generation */
%option c++

/* change the name of the scanner class. results in "ExampleFlexLexer" */
%option prefix="Example"

/* the manual says "somewhat more optimized" */
%option batch

/* enable scanner to generate debug output. disable this for release
 * versions. */
%option debug

/* no support for include files is planned */
%option yywrap nounput 

/* enables the use of start condition stacks */
%option stack

/* The following paragraph suffices to track locations accurately. Each time
 * yylex is invoked, the begin position is moved onto the end position. */
%{
#define YY_USER_ACTION  yylloc->columns(yyleng);
%}

DIGIT [0-9]
ID [A-Za-z][A-Za-z0-9_]*


%% /*** Regular Expressions Part ***/

 /* code to place at the beginning of yylex() */
%{
    // reset location
    yylloc->step();
%}

 /*** BEGIN EXAMPLE - Change the example lexer rules below ***/
 
begin { return token::T_BEGIN; }
end { return token::T_END; }
read { return token::READ; }
write { return token::WRITE; }
if { return token::IF; }
then { return token::THEN; }
else { return token::ELSE; }
while { return token::WHILE; }
do { return token::DO; }
def { return token::DEF; }
local { return token::LOCAL; }
shared { return token::SHARED; }
for { return token::FOR; }
to { return token::TO; }
pardo { return token::PARDO; }
return { return token::RETURN; }
int { return token::INT; }
array { return token::ARRAY; }
new { return token::NEW; }
delete { return token::DELETE; }
parallel { return token::PARALLEL; }
sizeof { return token::SIZEOF; }
setP { return token::SETP; }
:= { return token::ASSIGNOP; }
; { return token::SEMICOLON; }  
"(" { return token::LPAREN; }
")" { return token::RPAREN; }
"<<" { return token::SHL; }
">>" { return token::SHR; }
">=" { return token::GEQ; }
"<=" { return token::LEQ; }
"&" { return token::AND; }
"&&" { return token::LAND; }
"|" { return token::OR; }
"||" { return token::LOR; }
"^" { return token::XOR; }
 
{DIGIT}+ {
    yylval->integerVal = atoi(yytext);
    return token::INTEGER;
}

[0-9]+"."[0-9]* {
    yylval->doubleVal = atof(yytext);
    return token::DOUBLE;
}

".*" {
	yylval->stringVal = new std::string(yytext, yyleng);
	return token::STRING;
}

{ID} {
    yylval->stringVal = new std::string(yytext, yyleng);
    return token::IDENTIFIER;
}

"//"[^*\n]+\n {  yylloc->lines(1); yylloc->step(); }
 
<INITIAL>{
	"/*"              BEGIN(IN_COMMENT);
}
<IN_COMMENT>{
	"*/"      BEGIN(INITIAL);
	[^*\n]+   // eat comment in chunks
	"*"       // eat the lone star
	\n        yylloc->lines(yyleng); yylloc->step();;
}


 /* gobble up white-spaces */
[ \t\r]+ {
    yylloc->step();
}

 /* gobble up end-of-lines */
\n {
    yylloc->lines(yyleng); yylloc->step();
    //return token::EOL;
}

 /* pass all other characters up to bison */
. {
    return static_cast<token_type>(*yytext);
}

 /*** END EXAMPLE - Change the example lexer rules above ***/

%% /*** Additional Code ***/

namespace example {

Scanner::Scanner(std::istream* in,
		 std::ostream* out)
    : ExampleFlexLexer(in, out)
{
}

Scanner::~Scanner()
{
}

void Scanner::set_debug(bool b)
{
    yy_flex_debug = b;
}

}

/* This implementation of ExampleFlexLexer::yylex() is required to fill the
 * vtable of the class ExampleFlexLexer. We define the scanner's main yylex
 * function via YY_DECL to reside in the Scanner class instead. */

#ifdef yylex
#undef yylex
#endif

int ExampleFlexLexer::yylex()
{
    std::cerr << "in ExampleFlexLexer::yylex() !" << std::endl;
    return 0;
}

/* When the scanner receives an end-of-file indication from YY_INPUT, it then
 * checks the yywrap() function. If yywrap() returns false (zero), then it is
 * assumed that the function has gone ahead and set up `yyin' to point to
 * another input file, and scanning continues. If it returns true (non-zero),
 * then the scanner terminates, returning 0 to its caller. */

int ExampleFlexLexer::yywrap()
{
    return 1;
}
