#include "syntaxtree.h"
#include "context.h"

#include <iostream>

//////////  MainNode  //////////

MainNode::MainNode(SyntaxTreeNode *blockList) {
	blocks = blockList;
}

MainNode::~MainNode() {
	delete blocks;
}

void MainNode::generateCode(Context &context) {
	blocks->generateCode(context);
}

//////////  ~MainNode~  //////////

//////////  BlockList  //////////

BlockList::~BlockList() {
	for (unsigned int i = 0; i < blocks.size(); i++) {
		delete blocks[i];
	}
}

void BlockList::addBlock(SyntaxTreeNode *block) {
	blocks.push_back(block);
}


void BlockList::generateCode(Context &context) {
	for (unsigned int i = 0; i < blocks.size(); i++) {
		blocks[i]->generateCode(context);
	}
}

//////////  ~BlockList~  //////////

//////////  StatementList  //////////

StatementList::~StatementList() {
	for (unsigned int i = 0; i < statements.size(); i++) {
		delete statements[i];
	}
}

void StatementList::addStatement(SyntaxTreeNode *statement) {
	statements.push_back(statement);
}

void StatementList::generateCode(Context &context) {
	for (unsigned int i = 0; i < statements.size(); i++) {
		statements[i]->generateCode(context);
	}
}

//////////  ~StatementList~  //////////

//////////  FunctionDefinition  //////////

FunctionDefinition::FunctionDefinition(std::string name, VarType type, ArgList* arglist, bool par)
: name(name), type(type), arglist(arglist), parallel(par) {}

FunctionDefinition::~FunctionDefinition() {
	delete arglist;
}

void FunctionDefinition::generateCode(Context &context) {
	context.addFunction(name, type, arglist->getLength(), parallel);
	if (parallel) {
		context.addInstruction(STORE_LOC, context.addLocal("pid", v_int));
		context.addInstruction(STORE_LOC, context.addLocal("p", v_int));
	}
	arglist->generateCode(context);
}

//////////  ~FunctionDefinition~  //////////

////////// ArgList  //////////

void ArgList::addArg(VarType varType, std::string name) {
	arguments.push_back(std::make_pair(varType, name));
}

void ArgList::generateCode(Context &context) {
	for (unsigned int i = 0; i < arguments.size(); i++) {
		int addr = context.addLocal(arguments[i].second, arguments[i].first);
		context.addInstruction(STORE_LOC, addr);
	}
}

unsigned int ArgList::getLength() {
	return arguments.size();
}

////////// ~ArgList~  //////////

//////////  FunctionNode  //////////

FunctionNode::~FunctionNode() {
	delete definition; delete body;
}

FunctionNode::FunctionNode(SyntaxTreeNode *def, SyntaxTreeNode *statements)
: definition(def), body(statements) {}

void FunctionNode::generateCode(Context &context) {
	definition->generateCode(context);
	body->generateCode(context);
}

//////////  ~FunctionNode~  //////////

//////////  AssignmentNode  //////////

AssignmentNode::AssignmentNode(std::string idName, SyntaxTreeNode *expression)
: left(idName), right(expression) {}

AssignmentNode::~AssignmentNode() {
	delete right;
}

void AssignmentNode::generateCode(Context &context) {
	right->generateCode(context);
	genAssignment(context, left);
}

void AssignmentNode::genAssignment(Context &context, std::string name) {
	addrAndNamespace addr = context.getAddress(name);
	Instructions instr;
	if (addr.second == local) {
		instr = STORE_LOC;
	} else if (addr.second == shared) {
		instr = STORE_SHARE;
	}
	context.addInstruction(instr, addr.first);
}

//////////  ~AssignmentNode~  //////////

//////////  ArrayAssignmentNode  //////////

ArrayAssignmentNode::ArrayAssignmentNode(SyntaxTreeNode *arr, SyntaxTreeNode *index, SyntaxTreeNode *expression):
		arr(arr), index(index), expression(expression) {}

ArrayAssignmentNode::~ArrayAssignmentNode() {
	delete index; delete expression;
}

void ArrayAssignmentNode::generateCode(Context &context) {
	expression->generateCode(context);
	genAssignment(context, index, arr);
}

void ArrayAssignmentNode::genAssignment(Context &context, SyntaxTreeNode *index, SyntaxTreeNode *arr) {
	index->generateCode(context);
	arr->generateCode(context);
	context.addInstruction(STORE_ARR_LOC, noArgument);
}

//////////  ~ArrayAssignmentNode~  //////////

//////////  WhileNode  //////////

WhileNode::WhileNode(SyntaxTreeNode *expression, SyntaxTreeNode *statement)
: expr(expression), stmnt(statement) {}

WhileNode::~WhileNode() {
	delete expr; delete stmnt;
}

void WhileNode::generateCode(Context &context) {
	context.addInstruction(SYNC_BEGIN, noArgument);
	int begin = context.getCurrentAddress();
	expr->generateCode(context);
	int addr = context.reserveInstruction(JMPFALSE);
	stmnt->generateCode(context);
	context.addInstruction(JMP, begin);
	context.backPatch(addr);
	context.addInstruction(SYNC, noArgument);
}

//////////  ~WhileNode~  //////////

//////////  DoWhileNode  //////////

DoWhileNode::DoWhileNode(SyntaxTreeNode *statement, SyntaxTreeNode *expression)
: stmnt(statement), expr(expression) {}

DoWhileNode::~DoWhileNode() {
	delete expr; delete stmnt;
}

void DoWhileNode::generateCode(Context &context) {
	context.addInstruction(SYNC_BEGIN, noArgument);
	int begin = context.getCurrentAddress();
	stmnt->generateCode(context);
	expr->generateCode(context);
	context.addInstruction(JMPTRUE, begin);
	context.addInstruction(SYNC, noArgument);
}

//////////  ~DoWhileNode~  //////////

//////////  ForNode  //////////

ForNode::ForNode(std::string id, SyntaxTreeNode *expressionFrom, SyntaxTreeNode *expressionTo, SyntaxTreeNode *statement):
	identifier(id), exprFrom(expressionFrom), exprTo(expressionTo), statement(statement) { }

ForNode::~ForNode() {
	delete exprFrom, delete exprTo; delete statement;
}

void ForNode::generateCode(Context &context) {
	exprFrom->generateCode(context);
	AssignmentNode::genAssignment(context, identifier);
	context.addInstruction(SYNC_BEGIN, noArgument);
	int begin = context.getCurrentAddress();

	IDexpression::genIDLoad(context, identifier);
	exprTo->generateCode(context);
	context.addInstruction(LEQ, 0);
	int addr = context.reserveInstruction(JMPFALSE);

	statement->generateCode(context);

	context.addInstruction(LOAD_CONST, 1); // increment
	IDexpression::genIDLoad(context, identifier);
	context.addInstruction(ADD, 0);
	AssignmentNode::genAssignment(context, identifier);

	context.addInstruction(JMP, begin);
	context.backPatch(addr);
	context.addInstruction(SYNC, noArgument);
}

//////////  ~ForNode~  //////////

//////////  PardoNode  //////////

PardoNode::PardoNode(std::string id, SyntaxTreeNode *expressionFrom, SyntaxTreeNode *expressionTo, SyntaxTreeNode *statement):
	identifier(id), exprFrom(expressionFrom), exprTo(expressionTo), statement(statement) { }

PardoNode::~PardoNode() {
	delete exprFrom, delete exprTo; delete statement;
}

void PardoNode::generateCode(Context &context) {
	if (context.isInParProcedure()) {
		throw compileException("pardo loop not allowed in parallel procedure " + context.getCurrentFunction());
	}

	std::string lastFunction = context.getCurrentFunction();
	std::vector<std::string> parArgs = context.getPardoArgs();

	// telo novej funkcie - telo pardo
	int pardoFunc = context.addPardoFunction();
	int addr;
	for (int i = 0; i < parArgs.size(); i++) {
		addr = context.addPardoArg(parArgs[i]);
		context.addInstruction(STORE_LOC, addr);
		//IDexpression::genIDLoad(context, parArgs[i]);
	}
	addr = context.addPardoArg(identifier);
	context.addInstruction(STORE_LOC, addr);
	//IDexpression::genIDLoad(context, identifier);
	statement->generateCode(context);
	// koniec novej funkcie

	context.switchToFunction(lastFunction);

	for (int i = 0; i < parArgs.size(); i++) {
		IDexpression::genIDLoad(context, parArgs[i]);
	}
	exprFrom->generateCode(context);
	exprTo->generateCode(context);
	context.addInstruction(PARDO, pardoFunc);
}

//////////  ~PardoNode~  //////////

//////////  IfThenNode  //////////

IfThenNode::IfThenNode(SyntaxTreeNode *expression, SyntaxTreeNode *then)
: expr(expression), thenStmnt(then) { }

IfThenNode::~IfThenNode() {
	delete expr; delete thenStmnt;
}

void IfThenNode::generateCode(Context &context) {
	expr->generateCode(context);
	int addr = context.reserveInstruction(JMPFALSE_SYNC);
	thenStmnt->generateCode(context);
	context.backPatch(addr);
	context.addInstruction(SYNC, noArgument);
}

//////////  ~IfThenNode~  //////////

//////////  IfThenElseNode  //////////

IfThenElseNode::IfThenElseNode(SyntaxTreeNode *expression, SyntaxTreeNode *then, SyntaxTreeNode *Else)
: expr(expression), thenStmnt(then), elseStmnt(Else) { }

IfThenElseNode::~IfThenElseNode() {
	delete expr; delete thenStmnt; delete elseStmnt;
}

void IfThenElseNode::generateCode(Context &context) {
	expr->generateCode(context);
	int addr = context.reserveInstruction(JMPFALSE_SYNC);
	thenStmnt->generateCode(context);

	int addr2 = context.reserveInstruction(JMP);
	context.backPatch(addr);
	elseStmnt->generateCode(context);
	context.backPatch(addr2);
	context.addInstruction(SYNC, noArgument);
}

//////////  ~IfThenElseNode~  //////////

//////////  IDexpression  //////////

IDexpression::IDexpression(std::string name)
: name(name) {}

void IDexpression::generateCode(Context &context) {
	genIDLoad(context, name);
}

void IDexpression::genIDLoad(Context &context, std::string name) {
	addrAndNamespace  addr = context.getAddress(name);
		Instructions instr;
		if (addr.second == local) {
			instr = LOAD_LOC;
		} else if (addr.second == shared) {
			instr = LOAD_SHARE;
		}
	context.addInstruction(instr, addr.first);
}

//////////  ~IDexpression~  //////////

//////////  BinaryExpressionNode  //////////


BinaryExpressionNode::BinaryExpressionNode(SyntaxTreeNode *left, SyntaxTreeNode *right, Instructions instr)
: left(left), right(right), instr(instr) {}

BinaryExpressionNode::~BinaryExpressionNode() {
	delete left; delete right;
}

void BinaryExpressionNode::generateCode(Context &context) {
	left->generateCode(context);
	right->generateCode(context);
	context.addInstruction(instr, noArgument);
}

//////////  ~BinaryExpressionNode~  //////////

//////////  VariableDefinition  //////////

VariableDefinition::VariableDefinition(std::string name, VarType type, IDNamespace ns)
	: name(name), type(type), _namespace(ns) {

}
void VariableDefinition::generateCode(Context &context) {
	int i = 1;
	if (_namespace == local) {
		context.addLocal(name, type);
	} else if (_namespace == shared) {
		context.addShared(name, type);
	}
}

//////////  ~VariableDefinition~  //////////

////////// DereferenceNode  //////////

DereferenceNode::DereferenceNode(SyntaxTreeNode *identifier, SyntaxTreeNode *expression)
	: identifier(identifier), expression(expression) {
}

void DereferenceNode::generateCode(Context &context) {
	expression->generateCode(context);
	identifier->generateCode(context);
	context.addInstruction(DEREF, noArgument);
}

DereferenceNode::~DereferenceNode() {
	delete identifier; delete expression;
}


////////// ~DereferenceNode~  //////////


////////// SysCallNode  //////////

SysCallNode::SysCallNode(std::string type, SyntaxTreeNode *child, std::string name): type(type), child(child), name(name)  {}

void  SysCallNode::generateCode(Context &context) {
	SysCalls sys;
	if (type.substr(0, 4).compare("READ") == 0) {
		sys = READ;
	} else if (type.compare("WRITE") == 0) {
		sys = WRITE_;
		if (child != NULL) {
			child->generateCode(context);
		}
	}

	context.addInstruction(SYS, sys);

	if (type.compare("READ") == 0) {
		AssignmentNode::genAssignment(context, name);
	}
}

////////// ~SysCallNode~  //////////

////////// ReadArrNode  //////////

ReadArrNode::ReadArrNode(SyntaxTreeNode *arr, SyntaxTreeNode *index): arr(arr), index(index) {}

ReadArrNode::~ReadArrNode() {
	delete arr; delete index;
}

void ReadArrNode::generateCode(Context &context) {
	context.addInstruction(SYS, READ);
	ArrayAssignmentNode::genAssignment(context, index, arr);
}

////////// ~ReadArrNode~  //////////

////////// CallNode  //////////

CallNode::CallNode(std::string identifier, SyntaxTreeNode *expressions): identifier(identifier), expressions(expressions) {}

void CallNode::generateCode(Context &context) {
	unsigned int exprCount = 0;
	if (expressions != NULL) {
		expressions->generateCode(context);
		//kontrola poctu parametrov, casov by sa hodilo aj typu:
		ExprList * exprList = dynamic_cast<ExprList *>(expressions);
		if (exprList != NULL) {
			exprCount = exprList->getExprLength();
		}
	}
	if (context.getNumArgs(identifier) != exprCount) {
		throw compileException("Function call with wrong number of arguments. Function: " +
				identifier + " called from: " + context.getCurrentFunction());
	}
	int address = context.getFunction(identifier);
	context.addInstruction(CALL, address);
}

CallNode::~CallNode() {
	if (expressions != NULL) {
		delete expressions;
	}
}

////////// ~CallNode~  //////////

////////// ConstNode  //////////

ConstNode::ConstNode(int value): value(value) {}

void ConstNode::generateCode(Context &context) {
	context.addInstruction(LOAD_CONST, value);
}

////////// ~ConstNode~  //////////

////////// ExprList  //////////

ExprList::~ExprList() {
	for (unsigned int i = 0; i < expressions.size(); i++) {
		delete expressions[i];
	}
}

void ExprList::addExpression(SyntaxTreeNode *expression) {
	expressions.push_back(expression);
}

void ExprList::generateCode(Context &context) {
	for (unsigned int i = 0; i < expressions.size(); i++) {
		expressions[i]->generateCode(context);
	}
}

unsigned int ExprList::getExprLength() {
	return expressions.size(); 
}

////////// ~ExprList~  //////////

////////// ReturnNode  //////////

ReturnNode::ReturnNode(SyntaxTreeNode *expression): expression(expression) {}

void ReturnNode::generateCode(Context &context) {
	if (expression != NULL) {
		expression->generateCode(context);
	} else {
		context.addInstruction(LOAD_CONST, 0);
	}
	context.addInstruction(RET, noArgument);
}

ReturnNode::~ReturnNode() {
	delete expression;
}

////////// ~ConstNode~  //////////

////////// NewNode  //////////

NewNode::NewNode(SyntaxTreeNode *size, VarType varType):  expr(size), varType(varType) {}


void NewNode::generateCode(Context &context) {
	expr->generateCode(context);
	context.addInstruction(NEW, (ArgumentType)varType);
}

NewNode::~NewNode() {
	delete expr;
}

////////// ~NewNode~  //////////

////////// DeleteNode  //////////

DeleteNode::DeleteNode(SyntaxTreeNode* arr): arr(arr) {}

DeleteNode::~DeleteNode() {
	delete arr;
}

void DeleteNode::generateCode(Context &context) {
	arr->generateCode(context);
	context.addInstruction(DELETE, noArgument);
}

////////// ~DeleteNode~  //////////

////////// SizeOfNode  //////////

SizeOfNode::SizeOfNode(SyntaxTreeNode* expr): expression(expr) {}

SizeOfNode::~SizeOfNode() {
	delete expression;
}

void SizeOfNode::generateCode(Context &context) {
	expression->generateCode(context);
	context.addInstruction(SIZEOF, noArgument);
}

////////// ~SizeOfNode~  //////////

////////// SetPNode  //////////

SetPNode::SetPNode(SyntaxTreeNode* expr): expression(expr) {}

SetPNode::~SetPNode() {
	delete expression;
}

void SetPNode::generateCode(Context &context) {
	expression->generateCode(context);
	context.addInstruction(SETP, noArgument);
}

////////// ~SetPNode~  //////////

////////// PopNode  //////////

PopNode::PopNode(SyntaxTreeNode* body): body(body) {}

PopNode::~PopNode() {
	delete body;
}

void PopNode::generateCode(Context &context) {
	body->generateCode(context);
	context.addInstruction(POP, noArgument);
}

////////// ~PopNode~  //////////

////////// NotNode  //////////

NotNode::NotNode(SyntaxTreeNode* expr): expr(expr) {}

NotNode::~NotNode() {
	delete expr;
}

void NotNode::generateCode(Context &context) {
	expr->generateCode(context);
	context.addInstruction(NOT, noArgument);
}

////////// ~NotNode~  //////////
