#include <iostream>
#include <fstream>
#include <vector>

#include "driver.h"
#include "symboltable.h"
#include "context.h"
#include "codeobject.h"
#include "memory.h"
#include "Process.h"
#include "Scheduler.h"

using namespace std;

void printUsage(char *name) {
	cout << "Usage: " << name << " [OPTIONS] FILE" << endl << endl;
	cout << "OPTIONS:" << endl
			<< "-t \t\tPRAM type: erew, crew, ccrcw (common), acrcw (arbitrary)" << endl
			<< "-f file \tInput file. If no file is given, input is read from stdin." << endl
			<< "-h \t\tdisplay this help" << endl
			<< "-d name \tdump function (for debugging)" << endl;
}

/**
** Hlavna funkcia programu. Tu sa nacitava vstup, parsuju parametre, kompiluje program pre PRAM a
** nasledne sa spusta.
** Pri implementacii kompilatora bola pouzita sablona zo stranky
** https://idlebox.net/2007/flex-bison-cpp-example/
** ktora prepaja nastroje Flex a Bison tak aby generovali kod v jazyku C++.
*/
int main(int argc, char *argv[]) {
	vector<CodeObject *> *codeObjects = new vector<CodeObject *>();
	SymbolTable *share = new SymbolTable();
	SymbolTable *functions = new SymbolTable();

	MemType memType = mt_erew;

	Context context(*share, *functions, *codeObjects);

	example::Driver driver(context);

	char* fileName = NULL;
	char* inputFileName = NULL;
	bool argError = false;
	bool dump = false;
	string dumpFunction;

	//parsovanie parametrov
	for(int ai = 1; ai < argc; ++ai) {
		if (argv[ai] == string("-p")) {
			driver.trace_parsing = true;
		} else if (argv[ai] == string("-s")) {
			driver.trace_scanning = true;
		} else if (argv[ai] == string("-t")) {
			if (++ai >= argc) {
				argError = true;
				break;
			}
			if (argv[ai] == string("erew")) {
				memType = mt_erew;
			} else if (argv[ai] == string("crew")) {
				memType = mt_crew;
			} else if (argv[ai] == string("ccrcw")) {
				memType = mt_crcw_comm;
			} else if (argv[ai] == string("acrcw")) {
				memType = mt_crcw_arb;
			} else {
				argError = true;
				break;
			}
		} else if (argv[ai] == string("-h") || argv[ai] == string("--help")) {
			printUsage(argv[0]);
			return 0;
		} else if (argv[ai] == string("-d")) {
			if (++ai >= argc) {
				argError = true;
				break;
			}
			dumpFunction = argv[ai];
			dump = true;
		} else if (argv[ai] == string("-f")) {
			if (++ai >= argc) {
				argError = true;
				break;
			}
			inputFileName = argv[ai];
		} else {
			fileName = argv[ai];
		}
	}

	if (argError || fileName == NULL) {
		printUsage(argv[0]);
		return -1;
	}
	////////////////////

	// kompilacia
	fstream infile(fileName);
	if (!infile.good())	{
		std::cerr << "Could not open file: " << fileName << endl;
		return -1;
	}

	//vstupny subor pre pram
	fstream* inputfile = NULL;
	if (inputFileName != NULL) {
		inputfile = new fstream(inputFileName);
		if (!inputfile->good())	{
			std::cerr << "Could not open input file: " << inputFileName << endl;
			return -1;
		}
	}

        // samotne parsovanie
	bool result = driver.parse_stream(infile, fileName);

	if (!result) {
		cerr << "Parsing failed! Exiting" << endl;
		return -1;
	}

	try {
		driver.context.generateCode();
	} catch (symbolTableException &e) {
		cerr << "Symbol table error: " << e.what() << endl;
		return -1;
	} catch (compileException &e) {
		cerr << "Compilation error: " << e.what() << endl;
		return -1;
	}

	context.freeSyntaxTree();
	cout << "Parsing done" << endl;
	//////////////////////

	// dump funkcie:
	if (dump) {
		int index;
		try {
			index = functions->getIndex(dumpFunction);
		} catch (symbolTableException &e) {
			cerr << "Function " << dumpFunction << " not defined!" << endl;
			return -1;
		}
		cout << "Dumping function " << dumpFunction << endl;
		cout << (*codeObjects)[index]->dump();
		cout << "Exiting." << endl;
		return 0;
	}

	memory* sharedMem = share->getMemory();
	delete share;
	sharedMem->setMemType(memType);

	// kontrola ci existuje funkcia main
	try {
		int index = functions->getIndex("main");
	} catch (symbolTableException &e) {
		cerr << "Function main not found. Exiting." << endl;
		return -1;
	}

	// nasleduje spustenie programu
	istream& istr = (inputfile == NULL) ? cin : *inputfile;
	Scheduler scheduler(*sharedMem, functions, *codeObjects, istr);

	try {
		scheduler.runInit();
		scheduler.runMain();
		scheduler.runFinal();
	} catch (memAccessException &e) {
		cerr << "Memory access error: " << e.what() << endl;
		return -1;
	} catch (badOpException &e) {
		cerr << "Bad op: " << e.what() << endl;
		return -1;
	}

	cout << endl << "Program exited normally" << endl;
	cout << "Total time: " << scheduler.getTime() << endl;
	cout << "Total WT time: " << scheduler.getWTTime() << endl;
	cout << "Total work: " << scheduler.getWork() << endl;

	delete sharedMem;
	delete functions;
	for (int i = 0; i < codeObjects->size(); i++) {
		delete (*codeObjects)[i];
	}
	delete codeObjects;

	return 0;
}
