/* $Id: parser.yy 48 2009-09-05 08:07:10Z tb $ -*- mode: c++ -*- */
/** \file parser.yy Contains the example Bison parser source */

%{ /*** C/C++ Declarations ***/

#include <stdio.h>
#include <string>
#include <vector>

#include "syntaxtree.h"
#include "codeobject.h"

%}

/*** yacc/bison Declarations ***/

/* Require bison 2.3 or later */
%require "2.3"

/* add debug output code to generated parser. disable this for release
 * versions. */
%debug

/* start symbol is named "start" */
%start program

/* write out a header file containing the token defines */
%defines

/* use newer C++ skeleton file */
%skeleton "lalr1.cc"

/* namespace to enclose parser in */
%name-prefix="example"

/* set the parser's class identifier */
%define "parser_class_name" "Parser"

/* keep track of the current position within the input */
%locations
%initial-action
{
    // initialize the initial location object
    @$.begin.filename = @$.end.filename = &driver.streamname;
};

/* The driver is passed by reference to the parser and to the scanner. This
 * provides a simple but effective pure interface, not relying on global
 * variables. */
%parse-param { class Driver& driver }

/* verbose error messages */
%error-verbose

 /*** BEGIN EXAMPLE - Change the example grammar's tokens below ***/

%union {
    int  			integerVal;
    double 			doubleVal;
    std::string*		stringVal;
    class SyntaxTreeNode* node;
    class BlockList* blocks;
    class ArgList* argumentlist;
    class StatementList* statements;
    class ExprList* exprs;
}

%token			END	     0	"end of file"
%token			EOL		"end of line"
%token <integerVal> 	INTEGER		"integer"
%token <doubleVal> 	DOUBLE		"double"
%token <stringVal> 	STRING		"string"
%token <stringVal> 	IDENTIFIER		"identifier"
%token T_BEGIN "begin"
%token T_END "end" 
%token SEMICOLON 
%token ASSIGNOP ":="
%token <lbls> IF
%token  THEN  ELSE
%token  WHILE DO FOR TO PARDO RETURN NEW DELETE
%token READ WRITE DEF CALL 
%token LPAREN "(" 
%token RPAREN ")"
%token LOCAL SHARED INT ARRAY SHL SHR LEQ GEQ PARALLEL SIZEOF SETP

%type <node> program blok functiondef statement assignment expression iteration ifstatement idexpression dereference constant functioncall arr 
%type <blocks> blocks  
%type <exprs> exprlist
%type <statements> statements stmtblock
%type <argumentlist> arglist
%type <stringVal> namespace
%type <integerVal> varType

%left '-' '+'
%left '*' '/'
%left SHL SHR
%left '>' '<' '=' LEQ GEQ LAND LOR AND OR XOR
%left LPAREN RPAREN
%left NEG
%nonassoc REDUCE
%nonassoc ELSE

%destructor { delete $$; } IDENTIFIER STRING  

 /*** END EXAMPLE - Change the example grammar's tokens above ***/

%{

#include "driver.h"
#include "scanner.h"

#include "codeobject.h"

/* this "connects" the bison parser in the driver to the flex scanner class
 * object. it defines the yylex() function call to pull the next token from the
 * current lexer object of the driver context. */
#undef yylex
#define yylex driver.lexer->lex


%}

%% /*** Grammar Rules ***/

 /*** BEGIN EXAMPLE - Change the example grammar rules below ***/

program
	: blocks { $$ = new MainNode($1); driver.context.addSyntaxtTreeRoot($$); }

blocks
	: blocks blok { $$ = $1; $$->addBlock($2); }
	| /* empty */ {$$ = new BlockList(); }

blok
	: functiondef stmtblock { $$ = new FunctionNode($1, $2); }
	
functiondef
	: varType IDENTIFIER LPAREN arglist RPAREN { $$ = new FunctionDefinition(*$2, (VarType)$1, $4); }
	| varType IDENTIFIER LPAREN RPAREN { $$ = new FunctionDefinition(*$2, (VarType)$1, new ArgList()); }	
	| PARALLEL varType IDENTIFIER LPAREN RPAREN { $$ = new FunctionDefinition(*$3, (VarType)$2, new ArgList(), true); }
	
arglist
	: arglist ',' varType IDENTIFIER  { $$->addArg((VarType)$3, *$4); }
	| varType IDENTIFIER  {$$ = new ArgList(); $$->addArg((VarType)$1, *$2); }


statements
	: statements statement  { $$ = $1; $$->addStatement($2); }
	| /* empty */ {$$ = new StatementList(); }

stmtblock
	: T_BEGIN statements T_END {$$ = $2;}

statement
	: stmtblock  {$$ = $1; }
	| assignment {$$ = $1; }
	| iteration  {$$ = $1; }	
	| READ IDENTIFIER SEMICOLON { $$ = new SysCallNode("READ", NULL, *$2); }
	| READ arr '[' expression ']' SEMICOLON { $$ = new ReadArrNode($2, $4); }	
	| WRITE expression SEMICOLON { $$ = new SysCallNode("WRITE", $2, ""); }
	| ifstatement 
	| functioncall SEMICOLON { $$ = new PopNode($1); }
	| namespace varType IDENTIFIER  SEMICOLON {
	  IDNamespace ns;
	  if ($1->compare("local") == 0) { ns = local;}
	  if ($1->compare("shared") == 0) { ns = shared;}
	  $$ = new VariableDefinition(*$3, (VarType)$2, ns); 
	  delete $1; 
	 }
	 | RETURN SEMICOLON {
	 	$$ = new ReturnNode(NULL);
	 } 
	 | RETURN expression SEMICOLON {
		$$ = new ReturnNode($2);	 
	 }
	 | DELETE arr SEMICOLON { $$ = new DeleteNode($2); }
	 | SETP expression SEMICOLON {$$ = new SetPNode($2); }
	 
	
varType
	: INT { $$ = (int)v_int; }
	| ARRAY { $$ = (int)v_arr; }
	
	 
functioncall
	: IDENTIFIER LPAREN RPAREN { $$ = new CallNode(*$1, NULL); }
	| IDENTIFIER LPAREN exprlist RPAREN  { $$ = new CallNode(*$1, $3); }
	
namespace
	: /* empty */ { $$ = new std::string("local"); } // default local
	| LOCAL  { $$ = new std::string("local");  }
	| SHARED { $$ = new std::string("shared"); }	
	
ifstatement
	: IF expression THEN statement %prec REDUCE    { $$ = new IfThenNode($2, $4); }
	| IF expression THEN statement ELSE statement  { $$ = new IfThenElseNode($2, $4, $6); }

iteration
	: WHILE expression statement  { $$ = new WhileNode($2, $3); }
	| DO statements WHILE expression SEMICOLON { $$ = new DoWhileNode($2, $4); }
	| FOR IDENTIFIER ASSIGNOP expression TO expression DO statement { $$ = new ForNode(*$2, $4, $6, $8); }
	| FOR IDENTIFIER ASSIGNOP expression TO expression PARDO statement { $$ = new PardoNode(*$2, $4, $6, $8); }

assignment
	: IDENTIFIER ASSIGNOP expression SEMICOLON { $$ = new AssignmentNode(*$1, $3); }
	| arr '[' expression ']' ASSIGNOP expression SEMICOLON { $$ = new ArrayAssignmentNode($1, $3, $6); }
	
arr 
	: IDENTIFIER { $$ = new IDexpression(*$1); }
	| arr '[' expression ']' { $$ = new DereferenceNode($1, $3); }

expression
	: idexpression { $$ = $1; }
	| functioncall
	| constant { $$ = $1; }
	| '!' expression %prec NEG { $$ = new NotNode($2); }
	| expression '+' expression { $$ = new BinaryExpressionNode($1, $3, ADD); }
	| expression '-' expression { $$ = new BinaryExpressionNode($1, $3, SUB); }
	| expression '*' expression { $$ = new BinaryExpressionNode($1, $3, MUL);  }
	| expression '/' expression { $$ = new BinaryExpressionNode($1, $3, DIV); }
	| expression '>' expression { $$ = new BinaryExpressionNode($1, $3, GRT); }
	| expression '<' expression { $$ = new BinaryExpressionNode($1, $3, LESS); }
	| expression GEQ expression { $$ = new BinaryExpressionNode($1, $3, GEQ); }
	| expression LEQ expression { $$ = new BinaryExpressionNode($1, $3, LEQ); }
	| expression '=' expression { $$ = new BinaryExpressionNode($1, $3, EQ); }
	| expression SHL expression { $$ = new BinaryExpressionNode($1, $3, SHL); }
	| expression SHR expression { $$ = new BinaryExpressionNode($1, $3, SHR); }
	| expression AND expression { $$ = new BinaryExpressionNode($1, $3, AND); }
	| expression OR expression { $$ = new BinaryExpressionNode($1, $3, OR); }
	| expression XOR expression { $$ = new BinaryExpressionNode($1, $3, XOR); }
	| expression LAND expression { $$ = new BinaryExpressionNode($1, $3, LAND); }
	| expression LOR expression { $$ = new BinaryExpressionNode($1, $3, LOR); }
	| LPAREN expression RPAREN  { $$ = $2; }
	| dereference	
	| NEW LPAREN expression ',' varType RPAREN { $$ = new NewNode($3, (VarType)$5); }
	| SIZEOF LPAREN expression RPAREN { $$ = new SizeOfNode($3); }
	
constant
	: INTEGER { $$ = new ConstNode($1); }

exprlist
	: expression { $$ = new ExprList(); $$->addExpression($1); }
	| exprlist ',' expression {  $$->addExpression($3);  }
	
idexpression
	: IDENTIFIER { $$ = new IDexpression(*$1); }	

dereference
	: idexpression '[' expression ']' { $$ = new DereferenceNode($1, $3); }
	| dereference '[' expression ']' { $$ = new DereferenceNode($1, $3); }


 /*** END EXAMPLE - Change the example grammar rules above ***/

%% /*** Additional Code ***/

void example::Parser::error(const Parser::location_type& l,
			    const std::string& m)
{
    driver.error(l, m);
}
