#ifndef SYNTAXTREE_H_
#define SYNTAXTREE_H_


#include <vector>
#include <string>

#include "symboltable.h"
#include "codeobject.h"


enum IDNamespace {local, shared};

class Context; // Forward declaration

class SyntaxTreeNode {
public:
	virtual ~SyntaxTreeNode() {};
	virtual void generateCode(Context &context) = 0;
};

class MainNode: public SyntaxTreeNode {
private:
	SyntaxTreeNode *blocks;
public:
	MainNode(SyntaxTreeNode *blockList);
	virtual ~MainNode();
	void generateCode(Context &context);

};

class BlockList: public SyntaxTreeNode {
private:
	std::vector<SyntaxTreeNode *> blocks;
public:
	virtual ~BlockList();
	void addBlock(SyntaxTreeNode *block);
	void generateCode(Context &context);
};

class StatementList: public SyntaxTreeNode {
private:
	std::vector<SyntaxTreeNode *> statements;
public:
	virtual ~StatementList();
	void addStatement(SyntaxTreeNode *statement);
	void generateCode(Context &context);
};


class ArgList: public SyntaxTreeNode {
private:
	std::vector<std::pair<VarType, std::string> > arguments;
public:
	void addArg(VarType varType, std::string name);
	void generateCode(Context &context);
	unsigned int getLength();
};

class FunctionDefinition: public SyntaxTreeNode {
private:
	std::string name;
	VarType type;
	ArgList* arglist;
	bool parallel;
public:
	FunctionDefinition(std::string name, VarType type, ArgList* arglist, bool par = false);
	virtual ~FunctionDefinition();
	void generateCode(Context &context);
};


class FunctionNode: public SyntaxTreeNode {
private:
	SyntaxTreeNode *definition;
	SyntaxTreeNode *body;
public:
	virtual ~FunctionNode();
	FunctionNode(SyntaxTreeNode *def, SyntaxTreeNode *statements);
	void generateCode(Context &context);
};

class AssignmentNode: public SyntaxTreeNode {
private:
	std::string left;       /* identifier */
	SyntaxTreeNode *right;  /* expression */
public:
	AssignmentNode(std::string idName, SyntaxTreeNode *expression);
	virtual ~AssignmentNode();
	void generateCode(Context &context);
	static void genAssignment(Context &context, std::string name);
};

class ArrayAssignmentNode: public SyntaxTreeNode {
private:
	std::string identifier;  // identifier[index] := expression;
	SyntaxTreeNode *arr;
	SyntaxTreeNode *index;
	SyntaxTreeNode *expression;
public:
	ArrayAssignmentNode(SyntaxTreeNode *arr, SyntaxTreeNode *index, SyntaxTreeNode *expression);
	virtual ~ArrayAssignmentNode();
	void generateCode(Context &context);
	static void genAssignment(Context &context, SyntaxTreeNode *index, SyntaxTreeNode *arr);
};

class WhileNode: public SyntaxTreeNode {
private:
	SyntaxTreeNode *expr;
	SyntaxTreeNode *stmnt;
public:
	WhileNode(SyntaxTreeNode *expression, SyntaxTreeNode *statement);
	virtual ~WhileNode();
	void generateCode(Context &context);
};

class DoWhileNode: public SyntaxTreeNode {
private:
	SyntaxTreeNode *stmnt;
	SyntaxTreeNode *expr;
public:
	DoWhileNode(SyntaxTreeNode *statement, SyntaxTreeNode *expression);
	virtual ~DoWhileNode() ;
	void generateCode(Context &context);
};

class ForNode: public SyntaxTreeNode {
private:
	std::string identifier;
	SyntaxTreeNode *exprFrom;
	SyntaxTreeNode *exprTo;
	SyntaxTreeNode *statement;
public:
	ForNode(std::string id, SyntaxTreeNode *expressionFrom, SyntaxTreeNode *expressionTo, SyntaxTreeNode *statement);
	virtual ~ForNode() ;
	void generateCode(Context &context);
};

class PardoNode: public SyntaxTreeNode {
private:
	std::string identifier;
	SyntaxTreeNode *exprFrom;
	SyntaxTreeNode *exprTo;
	SyntaxTreeNode *statement;
public:
	PardoNode(std::string id, SyntaxTreeNode *expressionFrom, SyntaxTreeNode *expressionTo, SyntaxTreeNode *statement);
	virtual ~PardoNode() ;
	void generateCode(Context &context);
};

class IfThenNode: public SyntaxTreeNode {
private:
	SyntaxTreeNode *expr;
	SyntaxTreeNode *thenStmnt;
public:
	IfThenNode(SyntaxTreeNode *expression, SyntaxTreeNode *then);
	virtual ~IfThenNode();
	void generateCode(Context &context);
};

class IfThenElseNode: public SyntaxTreeNode {
private:
	SyntaxTreeNode *expr;
	SyntaxTreeNode *thenStmnt;
	SyntaxTreeNode *elseStmnt;
public:
	IfThenElseNode(SyntaxTreeNode *expression, SyntaxTreeNode *then, SyntaxTreeNode *Else);
	virtual ~IfThenElseNode();
	void generateCode(Context &context);
};

class IDexpression: public SyntaxTreeNode {
private:
	std::string name;
public:
	IDexpression(std::string name);
	void generateCode(Context &context) ;
	static void genIDLoad(Context &context, std::string name);
};

class BinaryExpressionNode: public SyntaxTreeNode {
private:
	SyntaxTreeNode *left;
	SyntaxTreeNode *right;
	Instructions instr;
public:
	BinaryExpressionNode(SyntaxTreeNode *left, SyntaxTreeNode *right, Instructions instr);
	virtual ~BinaryExpressionNode();
	void generateCode(Context &context);
};

class VariableDefinition: public SyntaxTreeNode {
private:
	std::string name;
	VarType type;
	IDNamespace _namespace;
public:
	VariableDefinition(std::string name, VarType type, IDNamespace ns);
	void generateCode(Context &context);
};

class DereferenceNode: public SyntaxTreeNode {
private:
	SyntaxTreeNode *identifier;
	SyntaxTreeNode *expression;
public:
	DereferenceNode(SyntaxTreeNode *identifier, SyntaxTreeNode *expression);
	void generateCode(Context &context);
	virtual ~DereferenceNode();
};

class SysCallNode: public SyntaxTreeNode {
private:
	std::string type;
	SyntaxTreeNode *child;
	std::string name;
public:
	SysCallNode(std::string type, SyntaxTreeNode *child, std::string name);
	void generateCode(Context &context);
};

class ReadArrNode: public SyntaxTreeNode {
private:
	SyntaxTreeNode *arr;
	SyntaxTreeNode *index;
public:
	ReadArrNode(SyntaxTreeNode *arr, SyntaxTreeNode *index);
	virtual ~ReadArrNode();
	void generateCode(Context &context);
};

class CallNode: public SyntaxTreeNode {
private:
	std::string identifier;
	SyntaxTreeNode *expressions;
public:
	CallNode(std::string identifier, SyntaxTreeNode *expressions);
	void generateCode(Context &context);
	virtual ~CallNode();
};

class ConstNode: public SyntaxTreeNode {
private:
	int value;
public:
	ConstNode(int value);
	void generateCode(Context &context);
};

class ExprList: public SyntaxTreeNode {
private:
	std::vector<SyntaxTreeNode *> expressions;
public:
	virtual ~ExprList();
	void addExpression(SyntaxTreeNode *expression);
	void generateCode(Context &context);
	unsigned int getExprLength();
};

class ReturnNode: public SyntaxTreeNode {
private:
	SyntaxTreeNode *expression;
public:
	ReturnNode(SyntaxTreeNode *expression);
	virtual ~ReturnNode();
	void generateCode(Context &context);
};

class NewNode: public SyntaxTreeNode {
private:
	VarType varType;
	SyntaxTreeNode *expr;
public:
	NewNode(SyntaxTreeNode *size, VarType varType);
	virtual ~NewNode();
	void generateCode(Context &context);
};

class DeleteNode: public SyntaxTreeNode {
private:
	SyntaxTreeNode* arr;
public:
	DeleteNode(SyntaxTreeNode* arr);
	virtual ~DeleteNode();
	void generateCode(Context &context);
};

class SizeOfNode: public SyntaxTreeNode {
private:
	SyntaxTreeNode* expression;
public:
	SizeOfNode(SyntaxTreeNode* expr);
	virtual ~SizeOfNode();
	void generateCode(Context &context);
};

class SetPNode: public SyntaxTreeNode {
private:
	SyntaxTreeNode* expression;
public:
	SetPNode(SyntaxTreeNode* expr);
	virtual ~SetPNode();
	void generateCode(Context &context);
};

class PopNode: public SyntaxTreeNode {
private:
	SyntaxTreeNode* body;
public:
	PopNode(SyntaxTreeNode* body);
	virtual ~PopNode();
	void generateCode(Context &context);
};

class NotNode: public SyntaxTreeNode {
private:
	SyntaxTreeNode* expr;
public:
	NotNode(SyntaxTreeNode* expr);
	virtual ~NotNode();
	void generateCode(Context &context);
};
#endif /* SYNTAXTREE_H_ */
