/// unit obsahujuci implemetacie zoznamu, dvojsmerneho zoznamu, stacku, ...
unit structures;

interface

uses constants;

type
  PPair = ^TPair;
  /// record obsahujuci par (dvojicu celociselnych hodnot x,y)
  TPair = record
    x: Integer;
    y: Integer;
  end;

  PListNode = ^TListNode;
  /// record obsahujuci strukturu prvku v obojsmernom zozname
  TListNode = record
    data: Integer;
    prev: PListNode;
    next: PListNode;
  end;

  /// trieda implementujuca obojsmerny zretazeny zoznam
  TDLinkedList = class
    head: PListNode;
    tail: PListNode;
  private
  public
    constructor create; overload;
    constructor create(x: PListNode); overload;
    procedure clear;
    destructor Destroy; override;
    procedure append(x: PListNode);
    procedure insertBefore(x: PListNode; p: PListNode);
    procedure remove(x: PListNode);
    procedure remandfree(x: PListNode);
    procedure remandfreeAll;
    function getByData(i: Integer): PListNode;
    function next(x: PListNode): PListNode;
    function prev(x: PListNode): PListNode;
    function getHead: PListNode;
    function getTail: PListNode;
  end;

  /// trieda implementujuca frontu/stack
  TQueueStack = class
    currentNode: PListNode;
    FSize: Integer;  //unsafe size due to possible list operations and concatenations (but safe in bounds of TQueueStack manipulation)
  private
  public
    list: TDLinkedList;
    constructor create;
    destructor Destroy; override;
    function clone: TQueueStack;
    function isEmpty: boolean;
    function pop: Integer;
    function dequeue: Integer;
    procedure push(x: Integer); overload;
    procedure push(A: array of Integer; first,last: Integer); overload;
    procedure enqueue(x: Integer);
    procedure addFirst(x: Integer);
    function top: Integer;
    function peek: Integer; overload;
    function peek(x: Integer): Integer; overload;
    procedure peekNextReset;
    function getCurrentNode: PListNode;
    function getLastNode: PListNode;
    procedure removeCurrentNode;
    function peekNext: Integer; overload;
    function peekNext(var I: Integer): Boolean; overload;
    function searchFor(x: Integer): boolean;
    procedure pushEx(x: Integer);
    procedure clear;
    function toString: xString;
    function getSize: Integer;
    procedure addQueueStack(x: TQueueStack);
    procedure moveQueueStack(x: TQueueStack);
  end;

  PSListNode = ^TSListNode;
  /// record obsahujuci strukturu prvku v utriedenom zretazenom zozname (vyuzivane iba pri vyznacovani farieb v ramci riadku)
  TSListNode = record
    key: Integer;
    data: Integer;
    tag1: Integer;
    tag2: Integer;
    tag3: Integer;
    prev: PSListNode;
    next: PSListNode;
  end;

  /// trieda implementujuca utriedeny zretazeny zoznam
  TSortedDLinkedList = class
    head: PSListNode;
    tail: PSListNode;
    currentNode: PSListNode;
  private
  public
    constructor create; overload;
    constructor create(x: PSListNode); overload;
    constructor create(key,data,tag1,tag2,tag3: Integer); overload;
    procedure clear;
    destructor Destroy; override;
    procedure add(x: PSListNode); overload;
    procedure addrev(x: PSListNode);
    procedure add(key,data,tag1,tag2,tag3: Integer); overload;
    procedure getNextReset;
    function getNext(var x: PSListNode): Boolean; overload;
    function getNext(var key,data,tag1,tag2,tag3: Integer): Boolean; overload;
  end;

  PIntNode = ^TIntNode;
  /// record obsahujuci strukturu prvku jednoducheho zretazeneho zoznamu celych cisiel
  TIntNode = record
    i: Integer;
    next: PIntNode;
    prev: PIntNode;
  end;
  PIntList = ^TIntList;
  /// record obsahujuci hlavu a chvost jednoducheho zretazeneho zoznamu celych cisiel
  TIntList = record
    head: PIntNode;
    tail: PIntNode;
//    size: Integer;
  end;
  function newlist: PIntList;
  procedure freelist(var p: PIntList);
  procedure append(var p: PIntList; i: PIntNode);
  procedure appendInteger(var p: PIntList; x: Integer);
  procedure appendtolist(var p: PIntList; var r: PIntList);
  procedure prepend(var p: PIntList; i: PIntNode);
  procedure prependInteger(var p: PIntList; x: Integer);
  procedure prependtolist(var p: PIntList; var r: PIntList);
  function clonelist(p: PIntList): PIntList;
  procedure clearlist(var p: PIntList);
  function listtoString(p: PIntList): xString;

implementation

uses
  SysUtils;

/// funkcia vypisujuca prvky zoznamu do stringu - TIntList
function listtoString(p: PIntList): xString;
var i: PIntNode;
begin
  result := '';
  if p = nil then
    Exit;

  i := p.head;
  while i <> nil do begin
    result := result + ' ' + IntToStr(i.i);
    i := i.next;
  end;
end;

/// funkcia vytvarajuca novy zoznam - TIntList
function newlist: PIntList;
begin
  new(result);
  result.head := nil;
  result.tail := nil;
end;

/// funkcia vytvarajuca kopiu daneho zoznamu (klon) - TIntList
function clonelist(p: PIntList): PIntList;
var i: PIntNode;
begin
  result := nil;
  if p = nil then
    Exit;

  result := newlist;

  i := p.head;
  while i <> nil do begin
(*    j := i.next;
    new(k);
    k^ := i^;
    append(result,k);
    i := j;*)
    appendInteger(result,i.i);
    i := i.next;
  end;
end;

/// procedura uvolnujuca prvky zoznamu z pamate - TIntList
procedure clearlist(var p: PIntList);
var i,j: PIntNode;
begin
  if p = nil then
    Exit;
  i := p.head;
  while i <> nil do begin
    j := i.next;
    dispose(i);
    i := j;
  end;
  p.head := nil;
  p.tail := nil;
end;

/// procedura uvolnujuca zoznam z pamate - TIntList
procedure freelist(var p: PIntList);
var i,j: PIntNode;
begin
  if p = nil then
    Exit;
  i := p.head;
  while i <> nil do begin
    j := i.next;
    dispose(i);
    i := j;
  end;
  dispose(p);
  p := nil;
end;

/// procedura pridavajuca na koniec zoznamu p prvok i - TIntList
procedure append(var p: PIntList; i: PIntNode);
begin
  if i = nil then
    Exit;
  if p = nil then begin
    p := newlist;
    p.head := i;
    p.tail := i;
    i.next := nil;
    i.prev := nil;
  end
  else begin
    if p.tail = nil then begin
      p.head := i;
      p.tail := i;
      i.next := nil;
      i.prev := nil;
    end
    else begin
      p.tail.next := i;
      i.prev := p.tail;
      p.tail := i;
      i.next := nil;
    end;
  end;
end;

/// procedura pridavajuca na koniec zoznamu p prvok vytvoreny z celeho cisla x - TIntList
procedure appendInteger(var p: PIntList; x: Integer);
var i: PIntNode;
begin
  new(i);
  i.i := x;
  i.next := nil;
  i.prev := nil;

  if p = nil then begin
    p := newlist;
    p.head := i;
    p.tail := i;
  end
  else begin
    if p.tail = nil then begin
      p.head := i;
      p.tail := i;
    end
    else begin
      p.tail.next := i;
      i.prev := p.tail;
      p.tail := i;
    end;
  end;
end;

/// procedura pridavajuca na koniec zoznamu p zoznam r - TIntList
procedure appendtolist(var p: PIntList; var r: PIntList);
begin
  if r = nil then
    Exit;
  if p = nil then begin
    p := r;
    r := nil;
  end
  else begin
    if p.tail = nil then begin
      p.head := r.head;
      p.tail := r.tail;
    end
    else begin
      p.tail.next := r.head;
      r.head.prev := p.tail;
      p.tail := r.tail;
    end;
    dispose(r);
    r := nil;
  end;
end;

/// procedura pridavajuca pred zaciatok zoznamu p prvok i - TIntList
procedure prepend(var p: PIntList; i: PIntNode);
begin
  if i = nil then
    Exit;
  if p = nil then begin
    p := newlist;
    p.head := i;
    p.tail := i;
    i.next := nil;
    i.prev := nil;
  end
  else begin
    if p.head = nil then begin
      p.head := i;
      p.tail := i;
      i.next := nil;
      i.prev := nil;
    end
    else begin
      p.head.prev := i;
      i.next := p.head;
      p.head := i;
      i.prev := nil;
    end;
  end;
end;

/// procedura pridavajuca pred zaciatok zoznamu p prvok vytvoreny z celeho cisla x - TIntList
procedure prependInteger(var p: PIntList; x: Integer);
var i: PIntNode;
begin
  new(i);
  i.i := x;
  i.next := nil;
  i.prev := nil;

  if p = nil then begin
    p := newlist;
    p.head := i;
    p.tail := i;
  end
  else begin
    if p.head = nil then begin
      p.head := i;
      p.tail := i;
    end
    else begin
      p.head.prev := i;
      i.next := p.head;
      p.head := i;
    end;
  end;
end;

/// procedura pridavajuca pred zaciatok zoznamu p zoznam r - TIntList
procedure prependtolist(var p: PIntList; var r: PIntList);
begin
  if r = nil then
    Exit;
  if p = nil then begin
    p := r;
    r := nil;
  end
  else begin
    if p.tail = nil then begin
      p.head := r.head;
      p.tail := r.tail;
    end
    else if r.tail <> nil then begin
      r.tail.next := p.head;
      p.head.prev := r.tail;
      p.head := r.head;
    end;
    dispose(r);
    r := nil;
  end;
end;

{ TDLinkedList }

/// konstruktor vytvarajuci obojsmerny zretazeny zoznam
constructor TDLinkedList.create;
begin
  inherited create;
  head := nil;
  tail := nil;
end;

/// konstruktor vytvarajuci obojsmerny zretazeny zoznam s prvkom x
constructor TDLinkedList.create(x: PListNode);
begin
  inherited create;
  head := x;
  tail := x;
end;

/// funkcia vracajuca posledny prvok zoznamu (chvost)
function TDLinkedList.getTail: PListNode;
begin
  result := tail;
end;

/// procedura vkladajuca novy prvok x pred prvok p
procedure TDLinkedList.insertBefore(x, p: PListNode);
begin
  if tail = nil then begin
    tail := x;
    head := x;
  end
  else if p = nil then begin
    tail.next := x;
    x.prev := tail;
    x.next := nil;
    tail := x;
  end
  else if p = head then begin
    head.prev := x;
    x.prev := nil;
    x.next := head;
    head := x;
  end
  else begin
    p.prev := x;
    x.prev := p.prev;
    x.next := p;
  end;
end;

/// procedura odstranujuca prvok x zo zoznamu
procedure TDLinkedList.remove(x: PListNode);
begin
  if x = nil then
    exit;
  if tail <> x then begin
    x.next.prev := x.prev;
  end
  else
    tail := x.prev;
  if head <> x then begin
    x.prev.next := x.next;
  end
  else
    head := x.next;
end;

/// funkcia vracajuca prvok nalavo od prvku x
function TDLinkedList.prev(x: PListNode): PListNode;
begin
  if (x <> nil) then
    result := x.prev
  else
    result := nil;
end;


/// funkcia vkladajuca novy prvok x na koniec zoznamu
procedure TDLinkedList.append(x: PListNode);
begin
  insertBefore(x,nil);
end;

/// funkcia vracajuca prvy prvok zoznamu (hlavu)
function TDLinkedList.getHead: PListNode;
begin
  result := head;
end;

/// procedura odstranujuca prvok x zo zoznamu a uvolnujuca ho z pamate
procedure TDLinkedList.remandfree(x: PListNode);
begin
  if x = nil then
    exit;
  remove(x);
  dispose(x);
end;

/// procedura, ktora odstani vsetky prvky zo zoznamu aj z pamate
procedure TDLinkedList.remandfreeAll;
var p,s,temp: PListNode;
begin
  p := getHead;
  s := getTail;
  if p <> nil then begin
    while p <> s do begin
      temp := p.next;
      remandfree(p);
      p := temp;
    end;
    if p <> nil then
      remandfree(p);
  end;
end;

/// funkcia vracajuca prvok, ktoreho datova cast struktury je rovna i (linearnym prechod celeho zoznamu)
function TDLinkedList.getByData(i: Integer): PListNode;
var temp: PListNode;
begin
  result := nil;
  temp := getHead;
  while temp <> nil do begin
    if temp.data = i then begin
      result := temp;
      exit;
    end;
    temp := temp.next;
  end;
end;

/// funkcia vracajuca prvok napravo od prvku x
function TDLinkedList.next(x: PListNode): PListNode;
begin
  if (x <> nil) then
    result := x.next
  else
    result := nil;
end;

/// procedura odstranujuca vsetky prvky zo zoznamu
procedure TDLinkedList.clear;
var p,s: PListNode;
begin
  p := head;
  while (p <> nil) do begin
    s := next(p);
    dispose(p);
    p := s;
  end;
  head := nil;
  tail := nil;
end;

/// destruktor odstranujuci zoznam z pamate
destructor TDLinkedList.Destroy;
begin
  clear;
  inherited;
end;

{ TQueueStack }

/// konstruktor vytvarajuci frontu/zasobnik
constructor TQueueStack.create;
begin
  inherited;
  list := TDLinkedList.create;
  currentNode := nil;
  FSize := 0;
end;

/// procedura pridavajuca prvok x do fronty zpredu
procedure TQueueStack.enqueue(x: Integer);
var P: PListNode;
begin
  new(P);
  P.data := x;
  P.next := nil;
  P.prev := nil;
  list.append(P);
  FSize := FSize + 1;
end;

/// procedura presuvajuca prvky z tejto fronty do druhej
procedure TQueueStack.moveQueueStack(x: TQueueStack);  //destructive append
begin
  while not x.isEmpty do begin
    push(x.dequeue);
  end;
end;

/// procedura pridavajuca kopie prvky z druhej fronty do tejto fronty
procedure TQueueStack.addQueueStack(x: TQueueStack);
var i: Integer;
begin
  x.peekNextReset;
  while x.peekNext(i) do begin
    push(i);
  end;
end;

/// procedura vkladajuca prvok x na koniec fronty
procedure TQueueStack.push(x: Integer);
begin
  enqueue(x);
end;

/// funkcia zistujuca, je prvok x vo fronte (linearne prehladavanie)
function TQueueStack.searchFor(x: Integer): boolean;
var P: PListNode;
begin
  P := list.head;
  while P <> nil do begin
    if P.data = x then begin
      result := true;
      exit;
    end;
    P := P.next;
  end;
  result := false;
end;

/// procedura vkladajuca prvok x do fronty ak tam este taky prvok nie je (linearne prehladavanie)
procedure TQueueStack.pushEx(x: Integer);
begin
  if not searchFor(x) then
    enqueue(x);
end;


/// procedura vkladajuca prvok x na zaciatok fronty
procedure TQueueStack.addFirst(x: Integer);
var P: PListNode;
begin
  new(P);
  P.data := x;
  P.next := nil;
  P.prev := nil;
  list.insertBefore(P,list.head);
  FSize := FSize + 1;
end;

/// funkcia vyberajuca prvok zo zaciatku fronty
function TQueueStack.dequeue: Integer;
begin
  if isEmpty then begin
    result := -1;
    exit;
  end;
  result := list.getHead.data;
  list.remandfree(list.getHead);
  FSize := FSize - 1;
end;

/// funkcia vyberajuca prvok z konca fronty
function TQueueStack.pop: Integer;
begin
  if isEmpty then begin
    result := -1;
    exit;
  end;
  result := list.getTail.data;
  list.remandfree(list.getTail);
  FSize := FSize - 1;
end;

/// funkcia vracajuca prvy prvok vo fronte
function TQueueStack.peek: Integer;
begin
  if isEmpty then begin
    result := -1;
    exit;
  end;
  result := list.getHead.data;
end;

/// funkcia vracajuca x-ty prvok zpredu vo fronte (v pripade zapornej hodnoty vracia -x-ty prvok zozadu)
function TQueueStack.peek(x: Integer): Integer;
var i: Integer;
    tempNode: PListNode;
begin
  tempNode := nil;
  if (isEmpty) or (x = 0) then begin
    result := -1;
    exit;
  end;
  if x > 0 then begin
    tempNode := list.getHead;
    for i := 1 to Pred(x) do begin
      if tempNode = nil then begin
        result := -1;
        exit;
      end;
      tempNode := list.next(tempNode);
    end;
  end
  else if x < 0 then begin
    tempNode := list.getTail;
    for i := 1 to Pred(Abs(x)) do begin
      if tempNode = nil then begin
        result := -1;
        exit;
      end;
      tempNode := list.prev(tempNode);
    end;
  end;
  if tempNode <> nil then
    result := tempNode.data
  else
    result := -1;
end;

/// procedura, ktora restartuje iteraciu cez prvky fronty (nasledujuci prvok je prvy prvok fronty)
procedure TQueueStack.peekNextReset;
begin
  currentNode := nil;
end;

/// funkcia vracajuca momentalny vrchol v iteracii cez vrcholy fronty
function TQueueStack.getCurrentNode: PListNode;
begin
  result := currentNode;
end;

/// funkcia odstranujuca momentalny vrchol v iteracii cez vrcholy fronty
procedure TQueueStack.removeCurrentNode;
var P: PListNode;
begin
  if currentNode = nil then exit;
  P := currentNode.prev;
  list.remandfree(currentNode);
  currentNode := P;
  FSize := FSize - 1;
end;

/// funkcia vracajuca posledny vrchol pri iteracii cez vrcholy fronty
function TQueueStack.getLastNode: PListNode;
begin
  result := list.getTail;
end;

/// funkcia vracajuca nasledujuci vrchol pri iteracii cez vrcholy fronty
function TQueueStack.peekNext: Integer;
begin
  if isEmpty then begin
    result := -1;
    exit;
  end;

  if currentNode = nil then
    currentNode := list.getHead
  else
    currentNode := list.next(currentNode);

  if currentNode = nil then
    result := -1
  else
    result := currentNode.data;
end;

/// funkcia vracajuca nasledujuci vrchol pri iteracii cez vrcholy fronty v premennej I,
/// vysledkom je true, ak este nasleduje za momentalnym vrchlom vo fronte nejaky dalsi vrchol,
/// inak vrati false
function TQueueStack.peekNext(var I: Integer): Boolean;
begin
  result := false;
  if isEmpty then begin
    I := -1;
    Exit;
  end;

  if currentNode = nil then
    currentNode := list.getHead
  else
    currentNode := list.next(currentNode);

  if currentNode = nil then begin
    I := -1;
    Exit;
  end
  else begin
    I := currentNode.data;
    result := true;
  end;
end;

/// funkcia vracajuca posledny prvok vo fronte (prvy od zadu)
function TQueueStack.top: Integer;
begin
  if isEmpty then begin
    result := -1;
    exit;
  end;
  result := list.getTail.data;
end;

/// funkcia zistujuca, ci je fronta prazdna
function TQueueStack.isEmpty: boolean;
begin
  result := (list.getTail = nil);
end;

/// destruktor odstranujuci frontu z pamate
destructor TQueueStack.Destroy;
begin
  list.Free;
  inherited;
end;

/// funkcia vypisujuca frontu do vystupneho stringu
function TQueueStack.toString: xString;
var p: PListNode;
begin
  result := '';
  p := list.head;
  while not (p = nil) do begin
    result := result + IntToStr(p.data) + ' ';
    p := list.next(p);
  end;
end;

/// funkcia vyprazdnujuca frontu
procedure TQueueStack.clear;
begin
  list.clear;
  currentNode := nil;
  FSize := 0;
end;

/// funkcia vytvarajuca novu frontu, ktora je kopiu tejto fronty (klon)
function TQueueStack.clone: TQueueStack;
var P: PListNode;
begin
  result := TQueueStack.create;
  P := list.head;
  while P <> nil do begin
    result.enqueue(P.data);
    P := P.next;
  end;
end;

/// funkcia vkladajuca pole A od indexu first po index last do fronty
procedure TQueueStack.push(A: array of Integer; first, last: Integer);
var I: Integer;
    len: Integer;
begin
  len := Length(A);
  if (first < 0) or (first > last) or (first >= len) then
    Exit;
  if last >= len then
    last := Pred(len);

  for I := first to last do begin
    push(A[I]);
  end;
end;

/// funkcia vracajuca pocet prvkov vo fronte
function TQueueStack.getSize: Integer;
begin
  result := FSize;
end;

{ TSortedDLinkedList }

/// konstruktor vytvarajuci utriedeny zretazeny zoznam s prvkom s hodnotami (key,data,tag1,tag2,tag3)
constructor TSortedDLinkedList.create(key, data, tag1, tag2, tag3: Integer);
var p: PSListNode;
begin
  new(p);
  p.key := key;
  p.data := data;
  p.tag1 := tag1;
  p.tag2 := tag2;
  p.tag3 := tag3;
  create(p);
end;

/// konstruktor vytvarajuci utriedeny zretazeny zoznam s prvkom x
constructor TSortedDLinkedList.create(x: PSListNode);
begin
  inherited create;
  add(x);
  currentNode := nil;
end;

/// konstruktor vytvarajuci utriedeny zretazeny zoznam
constructor TSortedDLinkedList.create;
begin
  inherited create;
  head := nil;
  tail := nil;
  currentNode := nil;
end;

/// procedura pridavajuca prvok (key,data,tag1,tag2,tag3) do zoznamu
procedure TSortedDLinkedList.add(key, data, tag1, tag2, tag3: Integer);
var p: PSListNode;
begin
  new(p);
  p.key := key;
  p.data := data;
  p.tag1 := tag1;
  p.tag2 := tag2;
  p.tag3 := tag3;
  p.prev := nil;
  p.next := nil;
  addrev(p);
end;

/// procedura pridavajuca prvok x do zoznamu
procedure TSortedDLinkedList.add(x: PSListNode);
var p: PSListNode;
begin
  if x = nil then
    Exit;

  p := head;
  if p = nil then begin
    x.prev := nil;
    x.next := nil;
    head := x;
    tail := x;
  end
  else if p.key < x.key then begin
    while p.next <> nil do begin
      if p.next.key >= x.key then
        break;
      p := p.next;
    end;
    if p.next = nil then begin
      x.next := nil;
      x.prev := p;
      p.next := x;
      tail := x;
    end
    else begin
      x.next := p.next;
      x.prev := p;
      p.next.prev := x;
      p.next := x;
    end;
  end
  else begin
    x.prev := nil;
    x.next := p;
    p.prev := x;
    head := x;
  end;
end;

/// procedura pridavajuca prvok x do zoznamu (vklada sa od konca, co pri vkladani monotone rastucich prvkov ide v konstantnom case)
procedure TSortedDLinkedList.addrev(x: PSListNode);
var p: PSListNode;
begin
  if x = nil then
    Exit;

  p := tail;
  if p = nil then begin
    x.prev := nil;
    x.next := nil;
    head := x;
    tail := x;
  end
  else if p.key > x.key then begin
    while p.prev <> nil do begin
      if p.prev.key <= x.key then
        break;
      p := p.prev;
    end;
    if p.prev = nil then begin
      x.prev := nil;
      x.next := p;
      p.prev := x;
      head := x;
    end
    else begin
      x.prev := p.prev;
      x.next := p;
      p.prev.next := x;
      p.prev := x;
    end;
  end
  else begin
    x.next := nil;
    x.prev := p;
    p.next := x;
    tail := x;
  end;
end;

/// funkcia vracajuca nasledujuci prvok v zozname v podobe patice (key,data,tag1,tag2,tag3)
function TSortedDLinkedList.getNext(var key, data, tag1, tag2, tag3: Integer): Boolean;
var p: PSListNode;
begin
  result := getNext(p);
  key := p.key;
  data := p.data;
  tag1 := p.tag1;
  tag2 := p.tag2;
  tag3 := p.tag3;
end;

/// funkcia vracajuca nasledujuci prvok v podobe pointra x na record obsahujuci strukturu prvku tohto zoznamu
function TSortedDLinkedList.getNext(var x: PSListNode): Boolean;
begin
  result := false;
  if currentNode = nil then begin
    x := head;
    if x <> nil then begin
      currentNode := x;
      result := true;
    end;
  end
  else begin
    x := currentNode.next;
    if x <> nil then begin
      currentNode := x;
      result := true;
    end;
  end;
end;

/// procedura na vyprazdnenie zoznamu
procedure TSortedDLinkedList.clear;
var p,s: PSListNode;
begin
  p := head;
  while (p <> nil) do begin
    s := p.next;
    dispose(p);
    p := s;
  end;
  head := nil;
  tail := nil;
end;

/// destruktor odstranujuci tento zoznam z pamate
destructor TSortedDLinkedList.Destroy;
begin
  clear;
  inherited;
end;

/// procedura na resetovanie iteracie cez vsetky prvky tohto zoznamu (nasledujucim prvkom bude prvy prvok zoznamu)
procedure TSortedDLinkedList.getNextReset;
begin
  currentNode := nil;
end;

end.
