#if !defined(BLOCK_LIST_MAX_SIZE)
	#define BLOCK_LIST_MAX_SIZE 65532
#endif

#define BLOCK_LIST_MAX_ENTRIES ( (BLOCK_LIST_MAX_SIZE - ( 2*sizeof(void *) + sizeof(unsigned int) )) / sizeof(BLOCK_LIST_ITEM_DATATYPE) )


#if defined(BLOCK_LIST_H) || defined(BLOCK_LIST_C)

struct _BLOCK_LIST_ {
	struct _BLOCK_LIST_ *Next;
	struct _BLOCK_LIST_ *Prev;
	unsigned int NumEntries;
	BLOCK_LIST_ITEM_DATATYPE Entries[BLOCK_LIST_MAX_ENTRIES];
};

typedef struct _BLOCK_LIST_ BLOCK_LIST;

#endif

#if defined(BLOCK_LIST_H)

#if defined(BLOCK_LIST_FIND_ENTRY_EQUAL)
	extern BLOCK_LIST_ITEM_DATATYPE *BLOCK_LIST_FIND_ENTRY_EQUAL(BLOCK_LIST **BlocksPtr, BLOCK_LIST_KEY_DATATYPE Key);
#endif
#if defined(BLOCK_LIST_FIND_ENTRY_EQUAL_OR_HIGHER)
	extern BLOCK_LIST_ITEM_DATATYPE *BLOCK_LIST_FIND_ENTRY_EQUAL_OR_HIGHER(BLOCK_LIST **BlocksPtr, BLOCK_LIST_KEY_DATATYPE Key);
#endif
#if defined(BLOCK_LIST_FIND_ENTRY_EQUAL_OR_LOWER)
	extern BLOCK_LIST_ITEM_DATATYPE *BLOCK_LIST_FIND_ENTRY_EQUAL_OR_LOWER(BLOCK_LIST **BlocksPtr, BLOCK_LIST_KEY_DATATYPE Key);
#endif
#if defined(BLOCK_LIST_FIND_ENTRY_FIRST)
	extern BLOCK_LIST_ITEM_DATATYPE *BLOCK_LIST_FIND_ENTRY_FIRST(BLOCK_LIST **BlocksPtr);
#endif
#if defined(BLOCK_LIST_FIND_ENTRY_HIGHER)
	extern BLOCK_LIST_ITEM_DATATYPE *BLOCK_LIST_FIND_ENTRY_HIGHER(BLOCK_LIST **BlocksPtr, BLOCK_LIST_KEY_DATATYPE Key);
#endif
#if defined(BLOCK_LIST_FIND_ENTRY_LAST)
	extern BLOCK_LIST_ITEM_DATATYPE *BLOCK_LIST_FIND_ENTRY_LAST(BLOCK_LIST **BlocksPtr);
#endif
#if defined(BLOCK_LIST_FIND_INDEX)
	extern unsigned int BLOCK_LIST_FIND_INDEX(BLOCK_LIST **BlocksPtr, BLOCK_LIST_KEY_DATATYPE Key);
#endif
#if defined(BLOCK_LIST_FOREACH)
	extern BLOCK_LIST_ITEM_DATATYPE *BLOCK_LIST_FOREACH(BLOCK_LIST **BlocksPtr, int (*proc)(BLOCK_LIST_ITEM_DATATYPE *, void *), void *data);
#endif
extern BLOCK_LIST_ITEM_DATATYPE *BLOCK_LIST_INSERT(BLOCK_LIST **BlocksPtr, BLOCK_LIST_KEY_DATATYPE Key, int IgnoreExisting);

extern int BLOCK_LIST_DELETE(BLOCK_LIST **BlocksPtr, BLOCK_LIST_KEY_DATATYPE Key);
extern void BLOCK_LIST_DELETE_ALL(BLOCK_LIST **BlocksPtr);

#endif

#if defined(BLOCK_LIST_C)

#if defined(BLOCK_LIST_PRINT_ERROR_MSG)
	#include <stdio.h>
#endif
#include <string.h>
#include <malloc.h>

#if defined(BLOCK_LIST_CREATE_CALLBACK)
	extern void BLOCK_LIST_CREATE_CALLBACK(BLOCK_LIST_ITEM_DATATYPE *Item);
#endif

#if defined(BLOCK_LIST_DELETE_CALLBACK)
	extern void BLOCK_LIST_DELETE_CALLBACK(BLOCK_LIST_ITEM_DATATYPE *Item);
#endif


/* find block where key exists or should exist */
BLOCK_LIST *BLOCK_LIST_FIND_BLOCK(BLOCK_LIST *Blocks, BLOCK_LIST_KEY_DATATYPE Key, int FindHigher, unsigned int *NumEntries)
{
	BLOCK_LIST *CurCodeBlock;

	if (NumEntries != NULL) *NumEntries = 0;

	if (Blocks == NULL) return NULL;

	if (Blocks->NumEntries == 0) return Blocks;

	if (BLOCK_LIST_KEY(*Blocks, 0) >= Key) return Blocks;

	CurCodeBlock = Blocks;

	while (BLOCK_LIST_KEY(*CurCodeBlock, CurCodeBlock->NumEntries - 1) < Key &&
			CurCodeBlock->Next != Blocks)
	{
		if (NumEntries != NULL) *NumEntries += CurCodeBlock->NumEntries;
		CurCodeBlock = CurCodeBlock->Next;
	}

	if (BLOCK_LIST_KEY(*CurCodeBlock, 0) > Key && FindHigher == 0)
	{
		if (NumEntries != NULL) *NumEntries -= CurCodeBlock->Prev->NumEntries;
		return CurCodeBlock->Prev;
	}
	else
	{
		return CurCodeBlock;
	}
}

/* find key in block */
unsigned int BLOCK_LIST_FIND_ENTRY_IN_BLOCK(BLOCK_LIST *Block, BLOCK_LIST_KEY_DATATYPE Key)
{
	BLOCK_LIST_KEY_DATATYPE Min;
	BLOCK_LIST_KEY_DATATYPE Max;
	unsigned int MinIndex, MaxIndex;

	if ( Block->NumEntries == 0 ) return 0;

	MinIndex = 0;
	Min = BLOCK_LIST_KEY(*Block, 0);
	if (Min >= Key) return 0;

	MaxIndex = Block->NumEntries - 1;
	Max = BLOCK_LIST_KEY(*Block, MaxIndex);
	if (Max < Key) return Block->NumEntries;

	if (Max == Key) return MaxIndex;

	while (MinIndex < MaxIndex)
	{
		BLOCK_LIST_KEY_DATATYPE Mid;
		unsigned int MidIndex;

		MidIndex = (MaxIndex + MinIndex) / 2;
		Mid = BLOCK_LIST_KEY(*Block, MidIndex);

		if (Mid < Key)
		{
			MinIndex = MidIndex + 1;
			Min = BLOCK_LIST_KEY(*Block, MidIndex + 1);
		}
		else
		{
			MaxIndex = MidIndex;
			Max = Mid;
		}
	}

	return MinIndex;
}

/* find key */
#if defined(BLOCK_LIST_FIND_ENTRY_EQUAL)
BLOCK_LIST_ITEM_DATATYPE *BLOCK_LIST_FIND_ENTRY_EQUAL(BLOCK_LIST **BlocksPtr, BLOCK_LIST_KEY_DATATYPE Key)
{
	BLOCK_LIST *Blocks;
	BLOCK_LIST *Block;
	unsigned int EntryNum;

	if (BlocksPtr == NULL) return NULL;

	Blocks = *BlocksPtr;

	if (Blocks == NULL) return NULL;

	Block = BLOCK_LIST_FIND_BLOCK(Blocks, Key, 0, NULL);
	if (Block == NULL) return NULL;

	EntryNum = BLOCK_LIST_FIND_ENTRY_IN_BLOCK(Block, Key);

	return (EntryNum < Block->NumEntries)?
			( (BLOCK_LIST_KEY(*Block, EntryNum) == Key)?(&(Block->Entries[EntryNum])):(NULL) ):
			(NULL);
}
#endif

/* find key or higher */
#if defined(BLOCK_LIST_FIND_ENTRY_EQUAL_OR_HIGHER)
BLOCK_LIST_ITEM_DATATYPE *BLOCK_LIST_FIND_ENTRY_EQUAL_OR_HIGHER(BLOCK_LIST **BlocksPtr, BLOCK_LIST_KEY_DATATYPE Key)
{
	BLOCK_LIST *Blocks;
	BLOCK_LIST *Block;
	unsigned int EntryNum;

	if (BlocksPtr == NULL) return NULL;

	Blocks = *BlocksPtr;

	if (Blocks == NULL) return NULL;

	Block = BLOCK_LIST_FIND_BLOCK(Blocks, Key, 1, NULL);
	if (Block == NULL) return NULL;

	EntryNum = BLOCK_LIST_FIND_ENTRY_IN_BLOCK(Block, Key);

	return (EntryNum < Block->NumEntries)?(&(Block->Entries[EntryNum])):(NULL);
}
#endif

/* find key or lower */
#if defined(BLOCK_LIST_FIND_ENTRY_EQUAL_OR_LOWER)
BLOCK_LIST_ITEM_DATATYPE *BLOCK_LIST_FIND_ENTRY_EQUAL_OR_LOWER(BLOCK_LIST **BlocksPtr, BLOCK_LIST_KEY_DATATYPE Key)
{
	BLOCK_LIST *Blocks;
	BLOCK_LIST *Block;
	unsigned int EntryNum;

	if (BlocksPtr == NULL) return NULL;

	Blocks = *BlocksPtr;

	if (Blocks == NULL) return NULL;

	Block = BLOCK_LIST_FIND_BLOCK(Blocks, Key, 0, NULL);
	if (Block == NULL) return NULL;

	EntryNum = BLOCK_LIST_FIND_ENTRY_IN_BLOCK(Block, Key);

	if ( EntryNum < Block->NumEntries )
	{
		if ( BLOCK_LIST_KEY(*Block, EntryNum) == Key )
		{
			return &(Block->Entries[EntryNum]);
		}
		else
		{
			if ( EntryNum == 0 )
			{
				if ( Block == Blocks )
				{
					return NULL;
				}
				else
				{
					return &(Block->Prev->Entries[Block->Prev->NumEntries - 1]);
				}
			}
			else
			{
				return &(Block->Entries[EntryNum - 1]);
			}
		}
	}
	else
	{
		if ( Block->NumEntries == 0 )
		{
			return NULL;
		}
		else
		{
			return &(Block->Entries[Block->NumEntries - 1]);
		}
	}
}
#endif

/* find first */
#if defined(BLOCK_LIST_FIND_ENTRY_FIRST)
BLOCK_LIST_ITEM_DATATYPE *BLOCK_LIST_FIND_ENTRY_FIRST(BLOCK_LIST **BlocksPtr)
{
	BLOCK_LIST *Blocks;

	if (BlocksPtr == NULL) return NULL;

	Blocks = *BlocksPtr;

	if (Blocks == NULL) return NULL;

	if (Blocks->NumEntries == 0) return NULL;

	return &(Blocks->Entries[0]);
}
#endif

/* find higher */
#if defined(BLOCK_LIST_FIND_ENTRY_HIGHER)
BLOCK_LIST_ITEM_DATATYPE *BLOCK_LIST_FIND_ENTRY_HIGHER(BLOCK_LIST **BlocksPtr, BLOCK_LIST_KEY_DATATYPE Key)
{
	BLOCK_LIST *Blocks;
	BLOCK_LIST *Block;
	unsigned int EntryNum;

	if (BlocksPtr == NULL) return NULL;

	Blocks = *BlocksPtr;

	if (Blocks == NULL) return NULL;

	Block = BLOCK_LIST_FIND_BLOCK(Blocks, Key, 1, NULL);
	if (Block == NULL) return NULL;

	EntryNum = BLOCK_LIST_FIND_ENTRY_IN_BLOCK(Block, Key);

	if (EntryNum >= Block->NumEntries)
	{
		return NULL;
	}
	else
	{
		if (BLOCK_LIST_KEY(*Block, EntryNum) == Key)
		{
			if (EntryNum == Block->NumEntries - 1)
			{
				if (Block->Next == Blocks)
				{
					return NULL;
				}
				else
				{
					return &(Block->Next->Entries[0]);
				}
			}
			else
			{
				return &(Block->Entries[EntryNum + 1]);
			}
		}
		else
		{
			return &(Block->Entries[EntryNum]);
		}
	}
}
#endif

/* find last */
#if defined(BLOCK_LIST_FIND_ENTRY_LAST)
BLOCK_LIST_ITEM_DATATYPE *BLOCK_LIST_FIND_ENTRY_LAST(BLOCK_LIST **BlocksPtr)
{
	BLOCK_LIST *Blocks;
	BLOCK_LIST *Block;

	if (BlocksPtr == NULL) return NULL;

	Blocks = *BlocksPtr;

	if (Blocks == NULL) return NULL;

	Block = Blocks->Prev;

	if (Block->NumEntries == 0) return NULL;

	return &(Block->Entries[Block->NumEntries - 1]);
}
#endif

/* find index */
#if defined(BLOCK_LIST_FIND_INDEX)
unsigned int BLOCK_LIST_FIND_INDEX(BLOCK_LIST **BlocksPtr, BLOCK_LIST_KEY_DATATYPE Key)
{
	BLOCK_LIST *Blocks;
	BLOCK_LIST *Block;
	unsigned int EntryNum, NumEntries;

	if (BlocksPtr == NULL) return 0;

	Blocks = *BlocksPtr;

	if (Blocks == NULL) return 0;

	Block = BLOCK_LIST_FIND_BLOCK(Blocks, Key, 0, &NumEntries);
	if (Block == NULL) return NumEntries;

	EntryNum = BLOCK_LIST_FIND_ENTRY_IN_BLOCK(Block, Key);

	return NumEntries + (EntryNum < Block->NumEntries)?(EntryNum):(Block->NumEntries);
}
#endif

#if defined(BLOCK_LIST_FOREACH)
BLOCK_LIST_ITEM_DATATYPE *BLOCK_LIST_FOREACH(BLOCK_LIST **BlocksPtr, int (*proc)(BLOCK_LIST_ITEM_DATATYPE *, void *), void *data)
{
	BLOCK_LIST *Blocks;
	BLOCK_LIST *CurCodeBlock;
	unsigned int iEntry;


	if (BlocksPtr == NULL) return NULL;
	Blocks = *BlocksPtr;
	if (Blocks == NULL) return NULL;

	CurCodeBlock = Blocks;
	do
	{
		for (iEntry = 0; iEntry < CurCodeBlock->NumEntries; iEntry++)
		{
			if ( proc(&(CurCodeBlock->Entries[iEntry]), data) )
			{
				return &(CurCodeBlock->Entries[iEntry]);
			}
		}

		CurCodeBlock = CurCodeBlock->Next;
	} while (CurCodeBlock != Blocks);

	return NULL;
}
#endif

/* insert key into block */
BLOCK_LIST_ITEM_DATATYPE *BLOCK_LIST_INSERT_ENTRY_INTO_BLOCK(BLOCK_LIST *Blocks, BLOCK_LIST *Block, BLOCK_LIST_KEY_DATATYPE Key, int IgnoreExisting)
{
	unsigned int EntryNum, NumToMove, NumToMovePrev;

	EntryNum = BLOCK_LIST_FIND_ENTRY_IN_BLOCK(Block, Key);

	if (EntryNum < Block->NumEntries)
	{
		if (BLOCK_LIST_KEY(*Block, EntryNum) == Key)
		{
			if (IgnoreExisting)
			{
				return &(Block->Entries[EntryNum]);
			}
			else
			{
#if defined(BLOCK_LIST_PRINT_ERROR_MSG)
				fprintf(stderr, "Error: Code block entry already exists\n");
#endif
				return NULL;
			}
		}
	}

	NumToMove = Block->NumEntries - EntryNum;

	memmove(&(Block->Entries[EntryNum + 1]), &(Block->Entries[EntryNum]), NumToMove * sizeof(BLOCK_LIST_ITEM_DATATYPE));

#if defined(BLOCK_LIST_CREATE_CALLBACK)
	BLOCK_LIST_CREATE_CALLBACK(&(Block->Entries[EntryNum]));
#else
	memset(&(Block->Entries[EntryNum]), 0, sizeof(BLOCK_LIST_ITEM_DATATYPE));
#endif
	BLOCK_LIST_KEY(*Block, EntryNum) = Key;

	Block->NumEntries++;

	if (Block->NumEntries == BLOCK_LIST_MAX_ENTRIES) /* Block full */
	{
		if (Block->Next == Blocks || Block->Next->NumEntries < BLOCK_LIST_MAX_ENTRIES - 10)
		{
			NumToMove = 0;
		}
		else
		{
			NumToMove = (BLOCK_LIST_MAX_ENTRIES - Block->Next->NumEntries) / 2;
		}

		if (Block->Prev == Blocks || Block->Prev->NumEntries < BLOCK_LIST_MAX_ENTRIES - 10)
		{
			NumToMovePrev = 0;
		}
		else
		{
			NumToMovePrev = (BLOCK_LIST_MAX_ENTRIES - Block->Prev->NumEntries) / 2;
		}

		if (NumToMove == 0 || NumToMovePrev == 0) /* must split */
		{
			BLOCK_LIST *NewBlock;

			NewBlock = (BLOCK_LIST *) malloc(sizeof(BLOCK_LIST));

			if (NewBlock == NULL)
			{
#if defined(BLOCK_LIST_PRINT_ERROR_MSG)
				fprintf(stderr, "Error: Not enough memory\n");
#endif
				return NULL;
			}

			NewBlock->Next = Block->Next;
			NewBlock->Prev = Block;
			NewBlock->Next->Prev = NewBlock;
			Block->Next = NewBlock;

			NewBlock->NumEntries = BLOCK_LIST_MAX_ENTRIES / 2;
			Block->NumEntries -= BLOCK_LIST_MAX_ENTRIES / 2;

			memcpy(&(NewBlock->Entries[0]), &(Block->Entries[Block->NumEntries]), (BLOCK_LIST_MAX_ENTRIES >> 1) * sizeof(BLOCK_LIST_ITEM_DATATYPE));

			if (EntryNum < Block->NumEntries)
			{
				return &(Block->Entries[EntryNum]);
			}
			else
			{
				return &(NewBlock->Entries[EntryNum - Block->NumEntries]);
			}
		}
		else /* move entries */
		{
			BLOCK_LIST *NextBlock;
			BLOCK_LIST *PrevBlock;

			NextBlock = Block->Next;
			PrevBlock = Block->Prev;

			if (NumToMove != 0)
			{
				NextBlock->NumEntries += NumToMove;
				Block->NumEntries -= NumToMove;
				memmove(&(NextBlock->Entries[NumToMove]), &(NextBlock->Entries[0]), NumToMove * sizeof(BLOCK_LIST_ITEM_DATATYPE));
				memcpy(&(NextBlock->Entries[0]), &(Block->Entries[Block->NumEntries]), NumToMove * sizeof(BLOCK_LIST_ITEM_DATATYPE));
			}

			if (NumToMovePrev != 0)
			{
				memcpy(&(PrevBlock->Entries[PrevBlock->NumEntries]), &(Block->Entries[0]), NumToMovePrev * sizeof(BLOCK_LIST_ITEM_DATATYPE));
				PrevBlock->NumEntries += NumToMovePrev;
				Block->NumEntries -= NumToMovePrev;
				memmove(&(Block->Entries[0]), &(Block->Entries[NumToMovePrev]), NumToMovePrev * sizeof(BLOCK_LIST_ITEM_DATATYPE));
			}

			if (EntryNum < NumToMovePrev)
			{
				return &(PrevBlock->Entries[PrevBlock->NumEntries - NumToMovePrev + EntryNum]);
			} else if (EntryNum - NumToMovePrev < Block->NumEntries)
			{
				return &(Block->Entries[EntryNum - NumToMovePrev]);
			}
			else
			{
				return &(NextBlock->Entries[(EntryNum - NumToMovePrev) - Block->NumEntries]);
			}
		}
	}
	else
	{
		return &(Block->Entries[EntryNum]);
	}
}

/* insert key into list */
BLOCK_LIST_ITEM_DATATYPE *BLOCK_LIST_INSERT(BLOCK_LIST **BlocksPtr, BLOCK_LIST_KEY_DATATYPE Key, int IgnoreExisting)
{
	BLOCK_LIST *Blocks;
	BLOCK_LIST *Block;

	if (BlocksPtr == NULL) {
#if defined(BLOCK_LIST_PRINT_ERROR_MSG)
		fprintf(stderr, "Error: No code block specified\n");
#endif
		return NULL;
	}

	Blocks = *BlocksPtr;

	if (Blocks == NULL || Blocks->NumEntries == 0) /* if no block defined */
	{
		if (Blocks == NULL)
		{
			Blocks = (BLOCK_LIST *) malloc(sizeof(BLOCK_LIST));

			if (Blocks == NULL)
			{
#if defined(BLOCK_LIST_PRINT_ERROR_MSG)
				fprintf(stderr, "Error: Not enough memory\n");
#endif
				return NULL;
			}
			*BlocksPtr = Blocks;

			Blocks->Next = Blocks;
			Blocks->Prev = Blocks;
		}

		Blocks->NumEntries = 1;
#if defined(BLOCK_LIST_CREATE_CALLBACK)
		BLOCK_LIST_CREATE_CALLBACK(&(Blocks->Entries[0]));
#else
		memset(&(Blocks->Entries[0]), 0, sizeof(BLOCK_LIST_ITEM_DATATYPE));
#endif
		BLOCK_LIST_KEY(*Blocks, 0) = Key;

		return &(Blocks->Entries[0]);
	}
	else /*  if block already defined */
	{
		Block = BLOCK_LIST_FIND_BLOCK(Blocks, Key, 0, NULL);

		if (Block == NULL)
		{
			return NULL;
		}

		return BLOCK_LIST_INSERT_ENTRY_INTO_BLOCK(Blocks, Block, Key, IgnoreExisting);
	}
}

/* delete key from list */
int BLOCK_LIST_DELETE(BLOCK_LIST **BlocksPtr, BLOCK_LIST_KEY_DATATYPE Key)
{
	BLOCK_LIST *Blocks;
	BLOCK_LIST *Block;
	unsigned int EntryNum, NumToMove;

	if (BlocksPtr == NULL) return -1;

	Blocks = *BlocksPtr;

	if (Blocks == NULL) return -1;

	Block = BLOCK_LIST_FIND_BLOCK(Blocks, Key, 0, NULL);
	if (Block == NULL) return -1;

	if (Block->NumEntries == 1) /* if last entry in block */
	{
		if (BLOCK_LIST_KEY(*Block, 0) == Key)
		{
#if defined(BLOCK_LIST_DELETE_CALLBACK)
			BLOCK_LIST_DELETE_CALLBACK(&(Block->Entries[0]));
#endif
			if (Blocks->Next == Blocks)
			{
/*				free(Blocks);
				*BlocksPtr = NULL;*/
				Block->NumEntries = 0;
				return 0;
			}
			else
			{
				Block->Prev->Next = Block->Next;
				Block->Next->Prev = Block->Prev;
				if (Blocks == Block)
				{
					*BlocksPtr = Block->Next;
					free(Block);
					return 0;
				}
				else
				{
					free(Block);
					return 0;
				}
			}
		}
		else
		{
			return -1;
		}
	}
	else /* if not last entry in block */
	{
		EntryNum = BLOCK_LIST_FIND_ENTRY_IN_BLOCK(Block, Key);

		if (EntryNum >= Block->NumEntries || BLOCK_LIST_KEY(*Block, EntryNum) != Key)
		{
			return -1;
		}
		else
		{
#if defined(BLOCK_LIST_DELETE_CALLBACK)
			BLOCK_LIST_DELETE_CALLBACK(&(Block->Entries[EntryNum]));
#endif

			Block->NumEntries--;

			NumToMove = Block->NumEntries - EntryNum;

			if (NumToMove != 0)
			{
				memmove(&(Block->Entries[EntryNum]), &(Block->Entries[EntryNum + 1]), NumToMove * sizeof(BLOCK_LIST_ITEM_DATATYPE));
			}

			return 0;
		}
	}
}

/* delete all keys */
void BLOCK_LIST_DELETE_ALL(BLOCK_LIST **BlocksPtr)
{
	BLOCK_LIST *Blocks;
	BLOCK_LIST *CurCodeBlock;
#if defined(BLOCK_LIST_DELETE_CALLBACK)
	unsigned int iEntry;
#endif


	if (BlocksPtr == NULL) return;
	Blocks = *BlocksPtr;
	if (Blocks == NULL) return;

	CurCodeBlock = Blocks;
	do
	{
		BLOCK_LIST *TempCodeBlock;

		TempCodeBlock = CurCodeBlock;
		CurCodeBlock = CurCodeBlock->Next;

#if defined(BLOCK_LIST_DELETE_CALLBACK)
		for (iEntry = 0; iEntry < TempCodeBlock->NumEntries; iEntry++)
		{
			BLOCK_LIST_DELETE_CALLBACK(&(TempCodeBlock->Entries[iEntry]));
		}
#endif

		free(TempCodeBlock);
	} while (CurCodeBlock != Blocks);

	*BlocksPtr = NULL;
}

#endif

#undef BLOCK_LIST_MAX_ENTRIES
#undef BLOCK_LIST_MAX_SIZE
