#define MODULE_NAME "pts"
#define PERMISSIVE_MODE 0
#define PRINT_MODE 1

#include "pts.h"

int system_users[] = SYSTEM_USERS;

int c_appr = DEF_C_APPR;
int c_shar = DEF_C_SHAR;
int i_shar = DEF_I_SHAR;


int def_c_o = DEF_C_O;
int def_i_o = DEF_I_O;
char *def_l_o = DEF_L_O;
uid_t def_u_o = DEF_U_O;


int def_cr_s = DEF_CR_S;
int def_cw_s = DEF_CW_S;
int def_crl_s = DEF_CRL_S;
int def_cwl_s = DEF_CWL_S;

int def_ir_s = DEF_IR_S;
int def_iw_s = DEF_IW_S;
int def_irl_s = DEF_IRL_S;
int def_iwl_s = DEF_IWL_S;

int def_cn_s = DEF_CN_S;
int def_in_s = DEF_IN_S;
uid_t def_u_s = DEF_U_S;
char *def_ln_s = DEF_LN_S;

/* test ci uid patri medzi deklarovanych system userov */
int is_system_user(int uid) {
	int i = 0;
	int len=sizeof(system_users)/sizeof(int);

	for (i = 0; i < len; i++) {
		if (uid == system_users[i])
			return 1;
	}
	return 0;
}



/* test ci binarka na danom dentry ma defaultne spustacie atributy mimo standardnych hranic*/
int binary_locked(struct dentry *dentry) {
	struct dentry *d;
	int rc, len;
	char *s;
	
	if (dentry->d_inode->i_op->getxattr == NULL) {
			return 0;
		}
	d = dget(dentry);
	rc = d->d_inode->i_op->getxattr(dentry, XATTR_NAME_BIN_PTS , NULL, 0);
	if (rc <= 0) {
		dput(dentry);
		return 0;
	}
	len = rc;
	s = kzalloc(len+1, GFP_KERNEL);
	s[len] = '\0';
	rc = d->d_inode->i_op->getxattr(dentry,XATTR_NAME_BIN_PTS,s, len);
	dput(dentry);
	if (rc<0) {			
		kfree(s);
		return 0;
	}
	if ( bounded_subject_pts_string(s) ) {
		kfree(s);
		return 0;
	}
	else {
		kfree(s);
		return 1;
	}
}




/* alokacia inode objektu*/
int my_inode_alloc_security(struct inode *inode) {
	struct object_pts *ob = new_object_pts_by_subject((struct subject_pts *) current_security());
	if (ob == NULL)
		return -ENOMEM;

	ob->u_o = inode->i_uid;
    inode->i_security = ob;
	if (inode->i_security == NULL)
    	return -ENOMEM;
    return 0;
}

/*uvolnenie inode objektu*/
void my_inode_free_security(struct inode *inode) {
        kfree(inode->i_security);
        inode->i_security = NULL;
}


/* mazanie inode na adrese dentry */
static int my_inode_unlink(struct inode *dir, struct dentry *dentry) {
	struct object_pts *unl_ob = (struct object_pts *) dentry->d_inode->i_security;
	struct object_pts *dir_ob = (struct object_pts *) dir->i_security;
	struct subject_pts *sub = (struct subject_pts *) current_security();
	int rc,len;

	if (capable(CAP_MAC_ADMIN))
		return 0;

	sub->u_s = current_euid();
	unl_ob->u_o = dentry->d_inode->i_uid;
	dir_ob->u_o = dir->i_uid;
	
	
	rc = pts_access(sub,unl_ob,MAY_WRITE_PTS);
	if (rc != 0) {
		if (PRINT_MODE) {
			printk(KERN_INFO "PRISTUP BY BOL ZAMIETNUTY PROCESU %s PRI MAZANI SUBORU %s.",current->comm,dentry->d_iname);
			printk(KERN_INFO "atrs: %s   %s,majitel procesu: %d, majitel suboru: %d.",subject_pts_to_string(sub,&len),object_pts_to_string(unl_ob,&len),current_euid(),dentry->d_inode->i_uid);
			printDentryPath(dentry);
		}
		if (PERMISSIVE_MODE) {
			return 0;
		}
		return rc;
	}
	rc = pts_access(sub,dir_ob,MAY_WRITE_PTS);

	if (rc != 0) {
		if (PRINT_MODE) {
			printk(KERN_INFO "PRISTUP BY BOL ZAMIETNUTY PROCESU %s PRI MAZANI SUBORU %s, RODICOVSKY ADRESAR WRITE DENIED.",current->comm,dentry->d_iname);
			printk(KERN_INFO "atrs: %s   %s,majitel procesu: %d, majitel suboru: %d.",subject_pts_to_string(sub,&len),object_pts_to_string(unl_ob,&len),current_euid(),dentry->d_inode->i_uid);
			printDentryPath(dentry);
		}
		if (PERMISSIVE_MODE) {
			return 0;
		}
		return rc;
	}
	
	return 0;
}


/* premenovanie inodu */
static int my_inode_rename(struct inode *old_inode,
			      struct dentry *old_dentry,
			      struct inode *new_inode,
			      struct dentry *new_dentry)
{
	int rc,len;	

	struct subject_pts *sub = (struct subject_pts *) current_security();
	struct object_pts *ob = (struct object_pts *) old_dentry->d_inode->i_security;
	
	sub->u_s = current_euid();	
	ob->u_o = old_dentry->d_inode->i_uid;

	if (capable(CAP_MAC_ADMIN))
		return 0;

	rc = pts_access(sub,ob,MAY_WRITE_PTS | MAY_READ_PTS);

	if (rc != 0) {
		if (PRINT_MODE) {
			printk(KERN_INFO "ZAKAZ: PREMENOVAVAM SUBOR %s s atrs: %s ,majitel: %d ",old_dentry->d_iname, object_pts_to_string((struct object_pts *) old_dentry->d_inode->i_security,&len), ob->u_o);
			printk(KERN_INFO "ZAKAZ: PREMENOVAVAJUCI PROCES: %s , atrs: %s ,majitel: %d", current->comm, subject_pts_to_string((struct subject_pts *) current_security(),&len), sub->u_s);
		}
		if (PERMISSIVE_MODE) {
			return 0;
		}
		return rc;
	}
	
	

	if ( (new_dentry != NULL) && (new_dentry->d_inode != NULL) && (new_dentry->d_inode->i_security != NULL) ) {
		ob = (struct object_pts *) new_dentry->d_inode->i_security;
		ob->u_o = new_dentry->d_inode->i_uid;

		rc = pts_access(sub,ob,MAY_WRITE_PTS);

		if (rc != 0) {
			if (PRINT_MODE) {
				printk(KERN_INFO "ZAKAZ: PREMENOVAVANY SUBOR %s s atrs: %s ,majitel: %d.",new_dentry->d_iname, object_pts_to_string((struct object_pts *) new_dentry->d_inode->i_security,&len), ob->u_o);	
				printk(KERN_INFO "PREMENOVAVAJUCI PROCES: %s , atrs: %s ,majitel: %d", current->comm, subject_pts_to_string((struct subject_pts *) current_security(),&len), sub->u_s);
			}
			if (PERMISSIVE_MODE) {
				return 0;
			}
			return rc;
		}
	}
	
	return 0;
}


/* povolenie pri vytvarani specialneho objektu */
static int my_inode_mknod(struct inode *dir, struct dentry *dentry, int mode, dev_t dev) {
	int rc = 0,len;
	struct object_pts *ob = (struct object_pts *) dir->i_security;
	struct subject_pts *sub = (struct subject_pts *) current_security();

	if (capable(CAP_MAC_ADMIN))
		return 0;

	ob->u_o = dir->i_uid;
	sub->u_s = current_euid();

	rc = pts_access(sub,ob,MAY_READ_PTS);
	if (rc != 0) {
		if (PRINT_MODE) {
			printk(KERN_INFO "PRISTUP BY BOL ZAMIETNUTY PROCESU %s PRI VYTVARANI SUBORU %s.",current->comm,dentry->d_iname);
			printk(KERN_INFO "atrs: %s   %s,majitel procesu: %d, majitel adresara: %d.",subject_pts_to_string(sub,&len),object_pts_to_string(ob,&len),current_euid(),dir->i_uid);
			printDentryPath(dentry);
		}
		if (PERMISSIVE_MODE) {
			return 0;
		}
		return rc;
	}

	rc = pts_access(sub,ob,MAY_WRITE_PTS);
	if (rc != 0) {
		if (PRINT_MODE) {
			printk(KERN_INFO "PRISTUP BY BOL ZAMIETNUTY PROCESU %s PRI VYTVARANI SUBORU %s.",current->comm,dentry->d_iname);
			printk(KERN_INFO "atrs: %s   %s,majitel procesu: %d, majitel adresara: %d.",subject_pts_to_string(sub,&len),object_pts_to_string(ob,&len),current_euid(),dir->i_uid);
			printDentryPath(dentry);
		}
		if (PERMISSIVE_MODE) {
			return 0;
		}
		return rc;
	}

	return 0;
}

/* povolenie pri vytvarani standardneho suboru */
static int my_inode_create(struct inode *dir, struct dentry *dentry, int mode) {
	int rc = 0,len;
	struct object_pts *ob = (struct object_pts *) dir->i_security;
	struct subject_pts *sub = (struct subject_pts *) current_security();

	if (capable(CAP_MAC_OVERRIDE))
		return 0;

	ob->u_o = dir->i_uid;
	sub->u_s = current_euid();

	rc = pts_access(sub,ob,MAY_READ_PTS);
	if (rc != 0) {
		if (PRINT_MODE) {
			printk(KERN_INFO "PRISTUP BY BOL ZAMIETNUTY PROCESU %s PRI VYTVARANI SUBORU %s.",current->comm,dentry->d_iname);
			printk(KERN_INFO "atrs: %s   %s,majitel procesu: %d, majitel adresara: %d.",subject_pts_to_string(sub,&len),object_pts_to_string(ob,&len),current_euid(),dir->i_uid);
			printDentryPath(dentry);
		}
		if (PERMISSIVE_MODE) {
			return 0;
		}
		return rc;
	}

	rc = pts_access(sub,ob,MAY_WRITE_PTS);
	if (rc != 0) {
		if (PRINT_MODE) {
			printk(KERN_INFO "PRISTUP BY BOL ZAMIETNUTY PROCESU %s PRI VYTVARANI SUBORU %s.",current->comm,dentry->d_iname);
			printk(KERN_INFO "atrs: %s   %s,majitel procesu: %d, majitel adresara: %d.",subject_pts_to_string(sub,&len),object_pts_to_string(ob,&len),current_euid(),dir->i_uid);
			printDentryPath(dentry);
		}
		if (PERMISSIVE_MODE) {
			return 0;
		}
		return rc;
	}

	return 0;
}


/* inicializacia xaatr pri vzniku inodu */
int my_inode_init_security(struct inode *inode, struct inode *dir,char **name, void **value, size_t *len) {
	int l = 0;
	char *objs = NULL;
	struct object_pts *ob;

	if (inode == NULL) {
		return -EINVAL;
	}
	ob = new_object_pts_by_subject((struct subject_pts *) current_security());
	inode->i_security = ob;

	objs = object_pts_to_string(ob,&l);
	if (l<0)
		return l;
	

	if (name) {
		*name = kstrdup(XATTR_PTS_SUFFIX, GFP_KERNEL);
		if (*name == NULL)
			goto faultAndFree;
	}

	if (value && len) {
		*value = kstrdup(objs, GFP_KERNEL);
		if (*value == NULL)
			goto faultAndFree;
		*len = l;
	}
	kfree(objs);
	return 0;

	faultAndFree:
		kfree(objs);
		return -ENOMEM;
}


/* kontrola pristupu k inode */
static int my_inode_permission(struct inode *inode, int mask) {
	int rc,len;
	struct object_pts *ob = (struct object_pts *) inode->i_security;
	struct subject_pts *sub = (struct subject_pts *) current_security();

	if ( (ob == NULL) || (sub == NULL) )	
		return 0;
	if (capable(CAP_MAC_ADMIN)) {
		return 0;
	}

	ob->u_o = inode->i_uid;
	sub->u_s = current_euid();

	if (S_ISDIR(inode->i_mode)) {
		mask = ((~1) & mask);
	}

	rc = pts_access(sub,ob,mask);

	if (rc != 0) {
		if (PRINT_MODE) {
			printk(KERN_INFO "PRISTUP BY BOL ZAMIETNUTY PROCESU %s PRI INODE,atrs: %s   %s,majitel procesu: %d,majitel inodu:%d, operacia: %d.",current->comm,subject_pts_to_string(sub,&len),object_pts_to_string(ob,&len),current_euid(),inode->i_uid,mask);
			printInodeType(inode);
			printk(KERN_INFO "SB MAGIC: %ld, INODE NUMBER: %ld", inode->i_sb->s_magic,inode->i_ino);
			if (inode->i_sb->s_root != NULL)
				printDentryPath(inode->i_sb->s_root);
		}
		if (PERMISSIVE_MODE) {		
			return 0;
		}
	}	
	
	return rc;
}


/* kontrola nastavovania atributov inodu */
static int my_inode_setattr(struct dentry *dentry, struct iattr *attr) {
	int l;
	struct subject_pts *sub = (struct subject_pts *) current_security();

	if (capable(CAP_MAC_ADMIN))
		return 0;
	
	if ((attr->ia_valid & ATTR_UID) && (dentry->d_inode->i_uid != attr->ia_uid)) {
		if (sub->chown) {
			goto nextCheck;
		}
		if (is_system_user(dentry->d_inode->i_uid) && is_system_user(attr->ia_uid)) {
			goto nextCheck;
		}
		if (PRINT_MODE) {
			printk(KERN_INFO "CHOWN PROCES: %s , atrs:%s..",current->comm, subject_pts_to_string((struct subject_pts *) current_security(),&l) );		
			printk(KERN_INFO "CHOWN OLD UID: %d, NEW UID: %d, atrs: %s",dentry->d_inode->i_uid, attr->ia_uid, object_pts_to_string((struct object_pts *) dentry->d_inode->i_security, &l));
			printk(KERN_INFO "CHOWN FILE: %s.",dentry->d_iname);			
			printInodeType(dentry->d_inode);
			printk(KERN_INFO "CHOWN SB MAGIC: %ld, INODE NUMBER: %ld", dentry->d_inode->i_sb->s_magic,dentry->d_inode->i_ino);
			printDentryPath(dentry);
		}
		if (PERMISSIVE_MODE) {
			goto nextCheck;
		}
		return -EPERM;
		
	}
	nextCheck:
		return 0;
}

/* interpretacia xattr v security namespace politikou */
int my_inode_getsecurity(const struct inode *inode, const char *name, void **buffer, bool alloc)
{
		char *s;
		int ilen;
		struct object_pts *ob = (struct object_pts *) inode->i_security;

		if (strncmp(name,XATTR_PTS_SUFFIX,strlen(XATTR_PTS_SUFFIX)) == 0) {
				if (ob == NULL) {
					return -EOPNOTSUPP;
				}
                s = object_pts_to_string((struct object_pts *) inode->i_security, &ilen);
			 	if (ilen < 0)
					return ilen;
                *buffer = s;
                return ilen;
		}

        return -EOPNOTSUPP;
}



/* nastavenie security kontextu pomocou xattr */
int my_inode_setsecurity(struct inode *inode, const char *name,const void *value, size_t size, int flags) {

	int rc;
	char *s;
	struct object_pts *ob;
	if (strncmp(name,XATTR_PTS_SUFFIX,strlen(XATTR_PTS_SUFFIX)) != 0)
		return -EOPNOTSUPP;
	ob = (struct object_pts *) inode->i_security;
	if (ob == NULL) {
		return -EOPNOTSUPP;
	}
	s = kzalloc(sizeof(char)*(size+1), GFP_KERNEL);
	s[size] = '\0';
	memcpy(s, value, size);
	rc = valid_object_pts_string(s);
	if ( rc != 0) {
		kfree(s);
		return rc;
	}
	rc = string_to_object_pts(s, ob);
	kfree(s);
	if (rc < 0)
		return rc;


	return 0;
}


/* kontrola citania hodnoty rozsiereneho atributu */
int my_inode_getxattr(struct dentry *dentry, const char *name) {
	return 0;
}

/* pomocna funkcia, kontrola file capability pri danom dentry */
int dentry_has_capability(struct dentry *fdentry, int capability) {
	struct dentry *dentry;
	int rc = 0;
	struct cpu_vfs_cap_data vcaps;


	if (!file_caps_enabled)
		return 0;


	dentry = dget(fdentry);

	rc = get_vfs_caps_from_disk(dentry, &vcaps);
	
	if (rc < 0) {		
		if (rc == -ENODATA)
			rc = 0;
		goto out;
	}
	
    if (cap_raised(vcaps.permitted, capability) )
		rc = 1;
out:
	dput(dentry);
	return rc;
}


/* kontrola odstranenia xattr */
int my_inode_removexattr(struct dentry *dentry, const char *name) {
	struct object_pts *ob;

	
	if (strncmp(name,XATTR_NAME_PTS,strlen(XATTR_NAME_PTS)) == 0) {
		if (capable(CAP_MAC_ADMIN))
			return 0;
		if (current_euid() != dentry->d_inode->i_uid)
			return -EPERM;
		if (!capable(CAP_MAC_OVERRIDE))
	    	return -EPERM;
		ob = (struct object_pts *) dentry->d_inode->i_security;
		/* pokus o zmenu mimo hranic */
		if (!bounded_object_pts_struct(ob))  {
			return -EPERM;
		}
		return 0;
	}
	if (!strcmp(name, XATTR_NAME_BIN_PTS)) {
		if (capable(CAP_MAC_ADMIN))
			return 0;
		if (current_euid() != dentry->d_inode->i_uid)
			return -EPERM;
		if  (!capable(CAP_MAC_OVERRIDE))
	    	return -EPERM;
		if (binary_locked(dentry) ) {
			return -EPERM;
		}
		return 0;
	}
	if (!strcmp(name, XATTR_NAME_CAPS)) {
		if ( (dentry_has_capability(dentry,CAP_MAC_ADMIN) || dentry_has_capability(dentry,CAP_MAC_OVERRIDE) || dentry_has_capability(dentry,CAP_CHOWN) || dentry_has_capability(dentry,CAP_SETUID) )  ) {			
			if (capable(CAP_MAC_ADMIN))
				return cap_inode_removexattr(dentry, name);
			return -EPERM;
		}				
	}
	return cap_inode_removexattr(dentry, name);
}

/* kontrola nastavovania xattr */
int my_inode_setxattr(struct dentry *dentry, const char *name,
				const void *value, size_t size, int flags) {
	int rc = 0;
	char *s;
	struct inode *inode;
	struct object_pts *ob;
	struct vfs_cap_data *vcaps;
	kernel_cap_t kcap;


	/* nastavovanie security.capability */	
					
	if (!strcmp(name, XATTR_NAME_CAPS)) {
		vcaps = (struct vfs_cap_data *) value;        
		kcap.cap[0] = vcaps->data[0].permitted;
		kcap.cap[1] = vcaps->data[1].permitted;
		if (capable(CAP_MAC_ADMIN))
			return cap_inode_setxattr(dentry,name,value,size,flags);
		if (cap_raised(kcap,CAP_MAC_ADMIN)) {			
			return -EPERM;
		}
		if (cap_raised(kcap,CAP_MAC_OVERRIDE)) {
			return -EPERM;
		}
		if (cap_raised(kcap,CAP_CHOWN)) {
			return -EPERM;
		}
		if (cap_raised(kcap,CAP_SETUID)) {
			return -EPERM;
		}
		return cap_inode_setxattr(dentry,name,value,size,flags);
	}

	/* nastavovanie security.BIN_PTS */
					
	if (!strcmp(name, XATTR_NAME_BIN_PTS)) {
		if ( (current_euid() != dentry->d_inode->i_uid) && (!capable(CAP_MAC_ADMIN)) )
			return -EPERM;
		if ( (!capable(CAP_MAC_ADMIN)) && (!capable(CAP_MAC_OVERRIDE)) )
	    	return -EPERM;
		if (binary_locked(dentry) && (!capable(CAP_MAC_ADMIN)) ) {
			return -EPERM;
		}		
		s = kzalloc(sizeof(char)*(size+1),GFP_KERNEL);
		if (s == NULL)
			return -ENOMEM;

		s[size] = '\0';
		memcpy(s, value, size);
		rc = valid_subject_pts_string(s);
		if (rc != 0) {
			// zly tvar
			kfree(s);
			return rc;
		}
		/* nastavovanie mimo hranic */
		if ((!bounded_subject_pts_string(s)) && (!capable(CAP_MAC_ADMIN))) {
			kfree(s);
			return -EPERM;
		}			
		kfree(s);
		return 0;
	}
					
	/* nastavovanie atributov, ktore nas nezaujimaju */
	if (strncmp(name,XATTR_NAME_PTS,strlen(XATTR_NAME_PTS)) != 0) {
		return cap_inode_setxattr(dentry, name, value, size, flags);
	}

	/* nastavovanie security.PTS */
	if ( (!capable(CAP_MAC_ADMIN)) && (!capable(CAP_MAC_OVERRIDE)) )
	    return -EPERM;

	inode = dentry->d_inode;
	ob = (struct object_pts *) inode->i_security;

	if ( (! bounded_object_pts_struct(ob))  && (! capable(CAP_MAC_ADMIN)) ) {
		return -EPERM;
	}

	s = kzalloc(sizeof(char)*(size+1), GFP_KERNEL);
	s[size] = '\0';
	memcpy(s, value, size);

	rc = valid_object_pts_string(s);
	if ( rc != 0) {
		// zly tvar
		kfree(s);
		return rc;
	}
	/* nastavovanie mimo hranic */
	if  ( (!bounded_object_pts_string(s)) && (!capable(CAP_MAC_ADMIN)) ) {
		kfree(s);
		return -EPERM;
	}
	kfree(s);
	
	return 0;
}

/* samotne zmeny pri nastavovani xattr v pripade povolenia */
void my_inode_post_setxattr(struct dentry *dentry, const char *name, const void *value, size_t size, int flags) {
	int rc = 0,l;
	char *s;
	struct inode *inode;
	struct object_pts *ob;

	/* zaujima nas len nastavovanie security.PTS */
	if (strncmp(name,XATTR_NAME_PTS,strlen(XATTR_NAME_PTS)) != 0) {
		return;
	}
	
	inode = dentry->d_inode;
	ob = (struct object_pts *) inode->i_security;

	s= kzalloc(sizeof(char)*(size+1), GFP_KERNEL);
	s[size] = '\0';
	memcpy(s, value, size);
	rc = valid_object_pts_string(s);
	if ( rc != 0) {
		kfree(s);
		return;
	}
	rc = string_to_object_pts(s, ob);
	kfree(s);
	if (rc < 0)
		return;
	
	s = object_pts_to_string(ob, &l);
	if (s == NULL) {
		return;
	}
	//spatne nastavenie permanentnej hodnoty xatrr celym kontextom
	inode->i_op->setxattr(dentry, XATTR_NAME_PTS, s, l,XATTR_REPLACE);
	kfree(s);

}

/* zoznam politikou pouzitych xattr pri inode */
int my_inode_listsecurity(struct inode *inode, char *buffer,
				    size_t buffer_size) {
	int len = strlen(XATTR_NAME_PTS);

	if (buffer != NULL && len <= buffer_size) {
		memcpy(buffer, XATTR_NAME_PTS, len);
		return len;
	}
	return -EINVAL;
}


/* instancovanie inode do dentry */
void my_d_instantiate(struct dentry *opt_dentry, struct inode *inode) {
	int rc,len;
	char *s = NULL;
	struct dentry *dentry;
	struct object_pts *ob;

	if (inode == NULL) {
		return;
	}

	ob = (struct object_pts *) inode->i_security;

	if (opt_dentry->d_parent == opt_dentry) {
		return;
	}
	
	if (inode->i_sb->s_magic == PROC_SUPER_MAGIC) {
		rc = string_to_object_pts("i_o=2;", ob);
		return;
	}

	/* pseudoterminaly */
	if (inode->i_sb->s_magic == DEVPTS_SUPER_MAGIC) {
		rc = string_to_object_pts("c_o=-1;i_o=-1;", ob);
		return;
	}
	
	if (inode->i_op->getxattr == NULL) {
		// dentry nepozna xattributy
		return;
	}
	dentry = dget(opt_dentry);
	rc = inode->i_op->getxattr(dentry, XATTR_NAME_PTS , NULL, 0);
	if (rc <= 0) {
		// dentry nema nasatveny PTS xattr
		dput(dentry);
		return;
	}
	len = rc;
	s = kzalloc(len+1, GFP_KERNEL);
	s[len] = '\0';
	rc = inode->i_op->getxattr(dentry,XATTR_NAME_PTS,s, len);
	dput(dentry);
	if (rc<0) {
		kfree(s);
		return;
	}
	rc = valid_object_pts_string(s);
	if ( rc != 0) {
		// zly tvar PTS xattr
		kfree(s);
		return;
	}

	rc = string_to_object_pts(s, ob);
	kfree(s);
	if (rc < 0)
		return;

}


/* ziskanie kontextu procesu */
int my_getprocattr(struct task_struct *p, char *name, char **value) {
	char *cp;
	int slen;
	struct subject_pts *sub;

	if (p == NULL)
		return -EINVAL;
	if (task_security(p) == NULL)
		return 0;
	sub = (struct subject_pts *) task_security(p);

	cp = subject_pts_to_string(sub,&slen);
	if (cp == NULL)
		return slen;

	*value = cp;
	return slen;
}


/* nastavenie kontextu procesu */
int my_setprocattr(struct task_struct *p, char *name, void *value, size_t size) {
	struct subject_pts *sub;
	char *s;
	int rc;


	if (p == NULL) {
		return -EINVAL;
	}

	if (capable(CAP_MAC_ADMIN)) {
		goto evaluate;
	}

	if (p != current) {
		return -EPERM;
	}

	if (!capable(CAP_MAC_OVERRIDE)) {
		return -EPERM;
	}

	evaluate:
	
	if (strcmp(name, "current") != 0) {
		// len cez current file sa nastavuje
	    return -EPERM;
	}

	if (task_security(p) == NULL) {
		return -EINVAL;
	}
	else {
		sub = (struct subject_pts *) task_security(p);
	}

	s = kzalloc(sizeof(char)*(size+1),GFP_KERNEL);
	if (s == NULL)
		return -ENOMEM;

	s[size] = '\0';
	memcpy(s, value, size);

	rc = valid_subject_pts_string(s);

	if (rc != 0) {
		// zly tvar
		kfree(s);
		return rc;
	}

	if ( (!bounded_subject_pts_string(s)) && (!capable(CAP_MAC_ADMIN)) ) {
		// nepovolene nastavovanie mimo hranic
		kfree(s);
		return -EPERM;
	}
	rc = string_to_subject_pts(s,sub,1);
	
	if (rc < 0) {
		kfree(s);
		return rc;
	}

	kfree(s);
	return (size);

}

/* alokovanie security credentials */
int my_cred_alloc_blank(struct cred *cred, gfp_t gfp) {

	struct subject_pts *sub;

	if (cred == NULL) {
		return -EINVAL;
	}
	sub  = new_subject_pts(gfp);
	if (sub == NULL) {
		return -ENOMEM;
	}
	sub->u_s = cred->euid;
	cred->security = sub;
	return 0;
}



/* uvolnenie security credentials */
void my_cred_free(struct cred *cred)
{
	struct subject_pts *sub;
	if (cred == NULL) {
		return;
	}
	if (cred->security != NULL) {
		sub = (struct subject_pts *) cred->security;
		del_subject_pts(sub);
		cred->security = NULL;
	}
}


/* priprava credentials */
int my_cred_prepare(struct cred *new, const struct cred *old, gfp_t gfp) {
	
	struct subject_pts *sub;
	if ( (old == NULL) || (new == NULL) ) {
		return -EINVAL;
	}

	
	if (old->security == NULL) {
		sub = new_subject_pts(gfp);
		if (sub == NULL) {
			return -ENOMEM;
		}
		new->security = sub;
		return 0;
	}

	sub = duplicate_subject_pts((struct subject_pts *)old->security,gfp);
	if (sub == NULL)
		return -ENOMEM;
	if (new->security != NULL)
			del_subject_pts((struct subject_pts *) new->security);
	new->security = sub;
	return 0;

}



/* prenos credentials */
void my_cred_transfer(struct cred *new, const struct cred *old) {

	struct subject_pts *sub;

	if ( (old == NULL) || (new == NULL) ) {
		return;
	}

	if (old->security == NULL) {
		sub = new_subject_pts(GFP_KERNEL);
		new->security = sub;
		return;
	}

	sub = duplicate_subject_pts((struct subject_pts *)old->security,GFP_KERNEL);
	if (sub == NULL)
		return;
	if (new->security != NULL)
			del_subject_pts((struct subject_pts *) new->security);
	new->security = sub;
}


/* vycistenie capabilities */
inline void bprm_clear_caps(struct linux_binprm *bprm) {
	cap_clear(bprm->cred->cap_permitted);
	bprm->cap_effective = false;
}


/* test ci binarka obsahuje danu file capability */
int bprm_has_capability(struct linux_binprm *bprm, bool *effective,int capability) {
	
	struct dentry *dentry;
	int rc = 0;
	struct cpu_vfs_cap_data vcaps;
	
	if (!file_caps_enabled)
		return 0;

	if (bprm->file->f_vfsmnt->mnt_flags & MNT_NOSUID)
		return 0;

	dentry = dget(bprm->file->f_dentry);

	rc = get_vfs_caps_from_disk(dentry, &vcaps);
	if (rc < 0) {	
		if (rc == -ENODATA)
			rc = 0;
		goto out;
	}

    if (vcaps.magic_etc & VFS_CAP_FLAGS_EFFECTIVE)
		*effective = true;
	rc = 0;
    if (cap_raised(vcaps.permitted, capability) )
		rc = 1;

out:
	dput(dentry);
	return rc;
}


/* spustanie binarky */
int my_bprm_set_creds(struct linux_binprm *bprm) {
    int rc = 0;
    struct cred *new = bprm->cred;
	struct subject_pts *oldSub = ( (struct subject_pts *) current_security() );
	struct subject_pts *newSub;
	struct inode *bin = bprm->file->f_dentry->d_inode;	// inode binarky
	struct dentry *dentry;
	int fileMacOverride = 0;		// binarka ma mac_override
	int parent_mac_admin = 0;	// rodic ma mac_admin
	int set_mac_admin = 0;		// na konci hooku hodnota CAP_MAC_ADMIN noveho procesu
	int set_chown = 0;			// na konci hooku moznost CHOWN bez obmedzenia userov
	int set_setuid = 0;			// na konci hooku moznost SETUID bez obmedzenia userov
	int len;
	char *s;
    bool eff;

		
	/* dedin po rodicovi */
	if (oldSub != NULL) {
		parent_mac_admin = oldSub->mac_admin;
	}

	/* z file capability chown cap beriem */
	if (bprm_has_capability(bprm, &eff,CAP_CHOWN)) {
		set_chown = 1;
	}

	/* z file capability setuid cap beriem */
	if (bprm_has_capability(bprm, &eff,CAP_SETUID)) {
		set_setuid = 1;
	}
	
	
    fileMacOverride = bprm_has_capability(bprm, &eff,CAP_MAC_OVERRIDE);	
    if  (fileMacOverride < 0)
        return -EINVAL;


    rc = cap_bprm_set_creds(bprm); // standardne nastavenie capabilities

	/* ak neboli CAP_MAC_OVERRIDE vo file capabilities binarky tak ich zrusim */
    if (!fileMacOverride) {
        cap_lower(new->cap_permitted, CAP_MAC_OVERRIDE);
        cap_lower(new->cap_effective, CAP_MAC_OVERRIDE);
    }

	/* vypocet CAP_MAC_ADMIN */
	if (bprm_has_capability(bprm, &eff,CAP_MAC_ADMIN) || (cap_raised(current->cred->cap_effective, CAP_MAC_ADMIN) && (current->pid != 1)) || parent_mac_admin ) {
		cap_raise(new->cap_permitted, CAP_MAC_ADMIN);
		cap_raise(new->cap_effective, CAP_MAC_ADMIN);
		set_mac_admin = 1;
	}
	else {
		cap_lower(new->cap_permitted, CAP_MAC_ADMIN);
        cap_lower(new->cap_effective, CAP_MAC_ADMIN);
	}


	/* nemam rodica */
	if (oldSub == NULL) {
		newSub = new_subject_pts(GFP_KERNEL);
		new->security = newSub;
		goto ok;
	}

	/* defaultne atributy binarky */
	if ( (bin->i_uid == current_euid()) || (S_ISUID & bin->i_mode) ) {

		/* bootovacia faza, este nepoznam vobec xattr, vynimka s neobmedzenymi pravami */
		if (bin->i_op->getxattr == NULL) {
			newSub = new_subject_pts(GFP_KERNEL);		
			newSub->heritable = 0;
			rc = string_to_subject_pts("cr_s=-1;cw_s=-1;ir_s=-1;iw_s=-1;",newSub,0);
			new->security = newSub;
			goto ok;
			
		}
		
		dentry = dget(bprm->file->f_dentry);
		rc = bin->i_op->getxattr(dentry, XATTR_NAME_BIN_PTS , NULL, 0);
		if (rc <= 0) {
			// nemam nastavene defaultne spustacie atributy
			dput(dentry);
			goto derive;
		}
		len = rc;
		s = kzalloc(len+1, GFP_KERNEL);
		s[len] = '\0';
		rc = bin->i_op->getxattr(dentry,XATTR_NAME_BIN_PTS,s, len);
		dput(dentry);
		if (rc<0) {			
			kfree(s);
			goto derive;
		}
		rc = valid_subject_pts_string(s);
		if ( rc != 0) {
			// zly tvar
			kfree(s);
			goto derive;
		}

		/* nechcem zachovat dedicnost ak nie je explicitne nastavena */
		newSub = new_subject_pts(GFP_KERNEL);	
		newSub->heritable = 0;

		rc = string_to_subject_pts(s,newSub,0);		
		
		kfree(s);
		new->security = newSub;
		goto ok;

	}

	derive:	
		/* dedim od rodica standardne */
		if ( (oldSub->heritable > 0) || (oldSub->heritable == -1) ) {
			if (oldSub->heritable > 0) {
				(oldSub->heritable)--;
			}
			newSub = duplicate_subject_pts(oldSub, GFP_KERNEL);
			new->security = newSub;
			goto ok;
		}
		/*uz sa nededi od rodica */
		else {
			newSub = new_subject_pts(GFP_KERNEL);
			new->security = newSub;
			goto ok;
		}	

    return rc;


	ok:
		/* nastavenie pomocnych flagov */
		((struct subject_pts *) new->security)->mac_admin = set_mac_admin;
		((struct subject_pts *) new->security)->chown = set_chown;
		((struct subject_pts *) new->security)->setuid = set_setuid;
		return 0;
	
}


/* zmena uid procesu */
static int my_task_setuid(uid_t id0, uid_t id1, uid_t id2, int flags) {
	int l;
	struct subject_pts *sub = (struct subject_pts *) current_security();
	
	if ((flags == LSM_SETID_RE) || (flags == LSM_SETID_RES)) {
		if (id1 != current_euid()) {
			if (sub->setuid) {
				return 0;
			}
			if (is_system_user(id1) && is_system_user(current_euid())) {
				return 0;
			}
			if (PRINT_MODE) {
				printk(KERN_INFO "SETUID ZAMIETNUTE, PROCES: %s , atrs:%s.",current->comm, subject_pts_to_string((struct subject_pts *) current_security(),&l) );
				printk(KERN_INFO "SETUID VALUES old:%d new:%d", id1, current_euid());
			}
			if (PERMISSIVE_MODE)
				return 0;
			return -EPERM;
		}
	}
	if (flags == LSM_SETID_ID) {
		if (id0 != current_euid()) {
			if (sub->setuid) {
				return 0;
			}
			if (is_system_user(id0) && is_system_user(current_euid())) {
				return 0;
			}
			
			if (PRINT_MODE) {
				printk(KERN_INFO "SETUID ZAMIETNUTE, PROCES: %s , atrs:%s.",current->comm, subject_pts_to_string((struct subject_pts *) current_security(),&l) );
				printk(KERN_INFO "SETUID VALUES old:%d new:%d", id1, current_euid());
			}
			if (PERMISSIVE_MODE)
				return 0;
			return -EPERM;
		}
	}
	return 0;
}




struct security_operations my_ops = {
  .name = MODULE_NAME,

  .bprm_set_creds = my_bprm_set_creds,
  .inode_alloc_security = my_inode_alloc_security,
  .inode_free_security =  my_inode_free_security,
  .inode_init_security =  my_inode_init_security,
  .inode_create = my_inode_create,
  .inode_mknod = my_inode_mknod,
  .inode_unlink = my_inode_unlink,
  .inode_rename = my_inode_rename,	
  .inode_setattr = my_inode_setattr,	
  .inode_permission = my_inode_permission,
  .inode_listsecurity = my_inode_listsecurity,
  .inode_getsecurity =  my_inode_getsecurity,
  .inode_setsecurity =  my_inode_setsecurity,
  .inode_getxattr =  my_inode_getxattr,
  .inode_setxattr =  my_inode_setxattr,
  .inode_removexattr = my_inode_removexattr,
  .inode_post_setxattr = my_inode_post_setxattr,

  .getprocattr = my_getprocattr,
  .setprocattr = my_setprocattr,
  .d_instantiate = my_d_instantiate,

  .cred_alloc_blank =		my_cred_alloc_blank,
  .cred_free =			my_cred_free,
  .cred_prepare =			my_cred_prepare,
  .cred_transfer =		my_cred_transfer,

  .task_setuid = my_task_setuid,
  

};

static __init int my_init(void) {
	struct cred *cred;
	struct subject_pts *sub;
	if (!security_module_enable(&my_ops))
		return 0;
	printk(KERN_INFO "PTS: spustanie politiky.\n");
	sub = new_subject_pts(GFP_KERNEL);
	if (sub == NULL)
		return -ENOMEM;
	cred = (struct cred *) current->cred;
	cred->security = sub;
	cap_lower(cred->cap_permitted, CAP_MAC_ADMIN);
    cap_lower(cred->cap_effective, CAP_MAC_ADMIN);
	if (register_security(&my_ops))
		panic("PTS: nepodarila sa registracia.\n");
	return 0;
}

security_initcall(my_init);


