// Copyright 2017 <Jozef Brandys>
#ifndef PINTOOL_CLUSTER_H_
#define PINTOOL_CLUSTER_H_

#include <algorithm>
#include <string>
#include <vector>

#include "Trace_binary.h"
#include "Cluster_clasifier.h"

class Cluster {
 private:
  std::vector<Trace_binary> traces_;
  std::shared_ptr<ClusterClasifier> clusterClasifier;

 public:
  int size;
  CoMatrix clusterMatrix;
  CoMatrix clusterMatrixMinimal;

  Cluster() : size(0) {}

  explicit Cluster(const Trace_binary &trace)
      : size(1), clusterMatrix(trace.GetBitmap()),
        clusterMatrixMinimal(trace.GetBitmap()) {}

  Cluster(const std::shared_ptr<Cluster> &a,
        const std::shared_ptr<Cluster> &b) {
    clusterMatrix = a->clusterMatrix | b->clusterMatrix;
    clusterMatrixMinimal = a->clusterMatrixMinimal & b->clusterMatrixMinimal;
    size = a->size + b->size;
  }

  double distance(const Cluster &other) const {
    return (other.clusterMatrix ^ clusterMatrix).count() /
      static_cast<double> (std::min(other.clusterMatrix.count(),
        clusterMatrix.count()));
  }

  double distance(const Trace_binary &trace) {
    return (trace.GetBitmap() ^ clusterMatrix).count() /
      static_cast<double>(std::min(trace.GetBitmap().count(),
        clusterMatrix.count()));
  }

  static double distance(const Cluster &a, const Cluster &b) {
    return a.distance(b);
  }

  double penalty(const Cluster &other) const {
    return log(std::max(size, other.size));
  }

  static double penalty(const Cluster &a, const Cluster &b) {
    return a.penalty(b);
  }

  bool AcceptsTrace(const Trace_binary &trace) {
    if (IsInCluster(trace)) {
      if (clusterClasifier->AcceptsTrace(trace)) {
        return true;
      }
      std::cerr << "clusterClasifier DECLINED" << std::endl;
    } else {
    }
    return false;
  }

  bool IsInCluster(const Trace_binary &trace) {
    return (trace.GetBitmap() | clusterMatrix) == clusterMatrix &&
        (trace.GetBitmap() & clusterMatrixMinimal) == clusterMatrixMinimal;
  }

  void ClearTraces() {
    traces_.clear();
    if (clusterClasifier) clusterClasifier->Clear();
  }

  int NumTraces() {
    return traces_.size();
  }

  void addTrace(Trace_binary trace) {
    traces_.push_back(trace);
  }

  void TrainCluster(std::string clasifier) {
    clusterClasifier.reset(ClusterClasifier::makeNewClasifier(clasifier));
    clusterClasifier->Train(traces_);
  }

  void Print(std::ostream &out) {
    out << clusterMatrix << std::endl;
    out << clusterMatrixMinimal << std::endl;
    clusterClasifier->Print(out);
  }

  void Read(std::istream &in, const std::string &clasifier) {
    clusterClasifier.reset(ClusterClasifier::makeNewClasifier(clasifier));
    in >> clusterMatrix;
    in >> clusterMatrixMinimal;
    clusterClasifier->Read(in);
  }

  void ChangeClasifier(const std::string &clasifier) {
    clusterClasifier.reset(ClusterClasifier::makeNewClasifier(clasifier));
  }
};
typedef std::weak_ptr<Cluster> wpCluster;

#endif  // PINTOOL_CLUSTER_H_
