// Prot386 Descriptor manipulation
// (C) 2006-2007 Peter Ambroz
// This code is free software and is distributed under the terms of GNU GPL

#include "prototypes.h"

char *desc_s0 = "Segmentation\nGlobal Descriptor Table\nLocal Descriptor Table\nInterrupt Desc. Table\nSet Active LDT\nBack";

char *sys_types[16] = { "Reserved","Available TSS 286","LDT","Busy TSS 286",
		 "Call Gate 286","Task Gate","Interrupt Gate 286","Trap Gate 286",
		 "Reserved","Available TSS","Reserved","Busy TSS",
		 "Call Gate","Reserved","Interrupt Gate","Trap Gate" };

// Order of sys_types items in menu
int sys_biject[16] = {12, 5, 14, 15, 9, 11, 2, 4, 6, 7, 1, 3, 0, 8, 10, 13};

char *user_types[2] = { "Data segment","Code segment" };
char *user_flags[2][4] = { { "Expand-up, Read-only","Expand-up, Read/Write",
			   "Expand-down, Read-only","Expand-down, Read/Write" },
			   { "Non-Conforming, Exec-only","Non-Conforming, Readable",
			   "Conforming, Exec-only","Conforming, Readable" } };
char *bit_sizes[2] = { " (16-bit)"," (32-bit)" };
char *presence[2] = { "Not Present","Present" };

char *dt0[3] = { "GDT","LDT","IDT" };
char *dd0 = " Descriptor no. ";
char *suffix[4] = { ""," KB)"," MB)"," GB)" };
char *nochange = "Warning\nThis is kernel descriptor.\nYou are not allowed to change it.";
char *nomem = "Warning\nKernel memory (0-2 MB)\ncan't be used in descriptors.\nChanges will be dropped.";

char *dlg0 = "\nEnter new value\n";
char *dlg1 = "\nEnter new value\n(hexadecimal)\n";
char *dlg2 = "\nEnter new index\n";
char *dlg3 = "\nEnter GDT index\n";
char *dlg5 = "Edit binary field\n|63  56|     48|     40|     32|\n";
char *dlg6 = "Edit binary field\n|31  24|     16|      8|      0|\n";
char *dlg7 = "\nGo to selector:\n(hexadecimal)\n";

// Coordinates for the "Edit descriptor" menu
int ed_x1 = 10;
int ed_x2 = 46;
int ed_y = 6;
int ed_raw = 9;
int ed_hl = 0x7F;

// idt.s
extern char *ex_names[];

char mnu0[224] = "\0";
char mnu1[16] = "\nGDT\nLDT";
char mnu2[192] = "\0";
char mnu3[240] = "\0";
char info[384], temp[40];
char dname[10];

#define LDT_MAX 15
int g2l_biject[LDT_MAX+1];

int desc_old2, desc_old3, desc_old4;
int desc_old5[2] = {0,0};

// Soft protection from several damage to the computer
// Forbids changing of important descriptors and using a memory occupied by kernel
int gdt_protect = 6;	// Last GDT entry, that will be protected against change
unsigned int mem_protect = 0x1FFFFF;	// Last linear memory address that can't be used
int enforce_protect = 1;  // switch the soft protection

DESCRIPTOR dtmp, dtmp2;
MENU_STATE mtmp;

void empty_cb(void) { }

// Check if 2 address ranges are overlapping
// We assume that physical address range is mapped back to the address 0 at the 2^32 boundary
int range_overlap(DESCRIPTOR *r1, int st2, int e2) {
  int st1, e1;

  if (is_expand_down(r1)) { // Expand-down data segments need to consider BASE and LIMIT differently
    st1 = r1->base + r1->limit + 1;
    e1 = r1->base - 1;
  } else { // Expand-up
    st1 = r1->base;
    e1 = r1->base + r1->limit;
  }

  if ((e2 >= st2) && (e1 >= st1))
    if ((st2 > e1) || (st1 > e2)) return 0;
  if ((st2 > e1) && (st1 > e2))
    if ((e2 >= st2) || (e1 >= st1)) return 0;
  return 1;
}

// Check if descriptor has base&limit fields
int has_base_limit(DESCRIPTOR *d1) {
  if ((d1->s) || ((d1->type>=1) && (d1->type<=3)) || (d1->type==9) || (d1->type==11))
    return 1;
  else
    return 0;
}

// Check for expand-down, data segment
int is_expand_down(DESCRIPTOR *d1) {
  if ((d1->s) && (((d1->type >> 2)&0x03) == 1)) return 1;
  else return 0;
}

/* Parse binary descriptor data to comprehensive structure */
void parse_desc_from_raw(DESCRIPTOR *d1) {
  unsigned int desc[2];

  if (!d1) return;
  desc[0] = d1->desc[0];
  desc[1] = d1->desc[1];

  if (desc[0]==0 && desc[1]==0) {
    d1->s = 0;
    d1->type = 0;
    return;
  }

  d1->p = (desc[1] >> 15) & 0x01;
  d1->dpl = (desc[1] >> 13) & 0x03;
  d1->s = (desc[1] >> 12) & 0x01;
  d1->type = (desc[1] >> 8) & 0x0F;

  d1->base = (desc[1] & 0xFF000000) + ((desc[1] & 0xFF) << 16) + (desc[0] >> 16);
  d1->limit = (desc[1] & 0x000F0000) + (desc[0] & 0xFFFF);
  d1->g = (desc[1] >> 23) & 0x01;
  if (d1->g) d1->limit = (d1->limit << 12) + 4095;

  d1->seg = desc[0] >> 16;
  d1->ofs = (desc[1] & 0xFFFF0000) + (desc[0] & 0xFFFF);
  d1->pc = desc[1] & 0x1F;

  if (d1->s) {  // Code or Data segment
    d1->d = (desc[1] >> 22) & 0x01;
  }
}

/* Read descriptor information */
DESCRIPTOR *load_descriptor(int selector, TABLE tab, DESCRIPTOR *d1) {
  unsigned int desc[2];

  if (!d1) return (DESCRIPTOR *)0;
  xdt_read(selector, tab, desc);
  d1->desc[0] = desc[0];
  d1->desc[1] = desc[1];
  d1->index=selector;
  d1->dti=tab;
  if (tab == TB_IDT) d1->sel_min = selector;
  else {
    d1->sel_min = selector * 8 + tab * 4;
    d1->sel_max = d1->sel_min + 3;
  }
  parse_desc_from_raw(d1);

  return d1;
}

#define PRINT_BASE_LIMIT strcat(info, "\nBase: ", hextostr(d1->base, tmps), 0); \
			 strcat(info, "\nLimit: ", hextostr(d1->limit, tmps), 0); \
			 if (suf) strcat(info, " (", inttostr(lim, tmps), suffix[suf], 0);
//			 if (d1->g) strcat(info, " (4kB granular)", 0);

#define PRINT_SELECTORS strcat(info, "\nSelectors: ", hextostr(d1->sel_min, tmps), 0); \
			strcat(info, " - ", hextostr(d1->sel_max, tmps), 0);

#define PRINT_SELECTOR strcat(info, "\nSegment: ", descriptor_name(d1->seg), " (", 0); \
		       strcat(info, inttostr(d1->seg, tmps), ")", 0);
#define PRINT_OFFSET strcat(info, "\nOffset: ", hextostr(d1->ofs, tmps), 0);
#define PRINT_PARAMS strcat(info, "\nParameters: ", inttostr(d1->pc, tmps), 0);
#define PRINT_RAW strcat(info, "\n|63 Raw data 48||47          32|", 0); \
		  strcat(info, "\n", pad_right(32, bintostr(d1->desc[1],tmps), '0'), 0); \
		  strcat(info, "\n", pad_right(32, bintostr(d1->desc[0],tmps), '0'), 0); \
		  strcat(info, "\n|31          16||15           0|", 0);

#define NEW_LINE strcat(info, "\n", 0);

/* Parse descriptor information */
char *parse_descriptor(DESCRIPTOR *d1, char *info) {
  char tmps[33];
  unsigned int lim, suf=0;

  info[0]='\0';

  strcat(info, dt0[d1->dti], dd0, inttostr(d1->index,tmps), 0);
  if ((!d1->s) && (!d1->type)) {
    strcat(info, "\nType: Invalid Descriptor", 0);
    NEW_LINE NEW_LINE NEW_LINE NEW_LINE NEW_LINE
    PRINT_SELECTORS
    PRINT_RAW
    return info;
  }

  if (is_expand_down(d1)) {
    lim = 0xFFFFFFFFu - d1->limit +1;
  } else lim = d1->limit+1;

  if (lim == 0) { lim = 4; suf = 3; }
  if (lim >= 0x40000000u) { lim >>= 30; suf=3; }
  if (lim >= 0x100000) { lim >>= 20; suf=2; }
  if (lim >= 0x400) { lim >>= 10; suf=1; }

  strcat(info, "\nType: ", 0);
  if (d1->s) {  // Code or Data segment
    strcat(info, user_types[d1->type<8?0:1], bit_sizes[d1->d], 0);
    strcat(info, "\nFlags: ", user_flags[d1->type<8?0:1][(d1->type>>1)&0x03], 0);
    PRINT_BASE_LIMIT
  } else {
    strcat(info, sys_types[d1->type], 0);
    switch(d1->type) {
      case 1:  // Available TSS 286
      case 2:  // LDT
      case 3:  // Busy TSS 286
      case 9:  // Available TSS
      case 11: NEW_LINE PRINT_BASE_LIMIT break; // Busy TSS
      case 5: NEW_LINE PRINT_SELECTOR NEW_LINE break; // Task Gate
      case 4:  // Call Gate 286
      case 12: PRINT_PARAMS PRINT_SELECTOR PRINT_OFFSET break; // Call Gate
      case 6:  // Int Gate 286
      case 7:  // Trap Gate 286
      case 14: // Int Gate
      case 15: NEW_LINE PRINT_SELECTOR PRINT_OFFSET break; // Trap Gate
      case 0:
      case 8:
      case 10:
      case 13: NEW_LINE NEW_LINE NEW_LINE break; // Reserved
    }
  }
  strcat(info, "\nDPL: ", inttostr(d1->dpl, tmps), 0);
  strcat(info, "\n", presence[d1->p], 0);
  if (d1->dti == 2) {
    strcat(info, "\nInterrupt: ", hextostr(d1->sel_min, tmps), 0);
  } else {
    PRINT_SELECTORS
  }
  PRINT_RAW
  return info;
}

#define SET_BASE_LIMIT desc[1] |= (d1->base & 0xFF000000) + ((d1->base & 0xFF0000) >> 16); \
		       desc[0] |= (d1->base & 0xFFFF) << 16; \
		       desc[1] |= (limit & 0x000F0000) + ((d1->g & 0x01) << 23); \
		       desc[0] |= (limit & 0xFFFF);

#define SET_SELECTOR desc[0] |= (d1->seg << 16);
#define SET_OFFSET desc[1] |= (d1->ofs & 0xFFFF0000); \
		   desc[0] |= (d1->ofs & 0xFFFF);

#define SET_PARAMS desc[1] |= (d1->pc & 0x1F);

void parse_desc_to_raw(DESCRIPTOR *d1) {
  unsigned int desc[2] = {0,0};
  unsigned int limit;

  if (!d1) return;
  if ((d1->s) || (d1->type)) {  // valid descriptor

  desc[1] |= (d1->p & 0x01) << 15;
  desc[1] |= (d1->dpl & 0x03) << 13;
  desc[1] |= (d1->s & 0x01) << 12;
  desc[1] |= (d1->type & 0x0F) << 8;

  limit = d1->limit;
  if (d1->g) limit = d1->limit >> 12;
  else if (d1->limit > 0xFFFFF) {
    limit = d1->limit >> 12;
    d1->g = 1;
  }

  if (d1->s) {
    desc[1] |= (d1->d & 0x01) << 22;
    SET_BASE_LIMIT
  } else {
    switch(d1->type) {
      case 1:  // Available TSS 286
      case 2:  // LDT
      case 3:  // Busy TSS 286
      case 9:  // Available TSS
      case 11: SET_BASE_LIMIT break; // Busy TSS
      case 5: SET_SELECTOR break; // Task Gate
      case 4:  // Call Gate 286
      case 12: SET_SELECTOR SET_OFFSET SET_PARAMS break; // Call Gate
      case 6:  // Int Gate 286
      case 7:  // Trap Gate 286
      case 14: // Int Gate
      case 15: SET_SELECTOR SET_OFFSET break; // Trap Gate
    }
  }

  }  // valid descriptor

  d1->desc[0] = desc[0];
  d1->desc[1] = desc[1];
}

void store_descriptor(DESCRIPTOR *d1) {
  if (!d1) return;
  // Kernel memory protection:
  if ((d1->dti==TB_GDT) && (d1->index <= gdt_protect) && enforce_protect)
  { // SW protection of essential descriptors.
    okno(13, 49, 6, 2, 116, 0, nochange);
    _read_enter();
    return;
  }
  if (has_base_limit(d1) && enforce_protect) { // Only those descriptors with BASE and LIMIT fields
    if (range_overlap(d1, 0, mem_protect)) { // Kernel protected range
      // If the base is OK, but fails on limit, adjust it, so we can continue in editing
      if (!is_expand_down(d1) && (d1->base > mem_protect)) {
	d1->limit = 0;
	d1->g = 0;
	goto success;
      } else
      if (is_expand_down(d1) && (d1->base > (mem_protect+0x1000))) {
	d1->limit = 0xFFFFEFFFu;
	d1->g = 1;
	goto success;
      }
      okno(13, 45, 6, 4, 116, 0, nomem);
      _read_enter();
      return;
    }
  }
success:
  parse_desc_to_raw(d1);
  xdt_write(d1->index, d1->dti, d1->desc);
}

char *descriptor_name(int no) {
  char tmps[11];
  dname[0]='\0';
  strcat(dname, dt0[(no >> 2) & 1], "[", inttostr(no >> 3, tmps), "]", 0);
  return dname;
}

char *summary(int sel, char *data);
char *lsummary(int sel, char *data);

// helper function for highlighting portions of raw descriptor
void raw_hilight(int from, int to, int clear_other) {
  int x1, x2, y, h_x1, h_x2;

  x1 = ed_x1+2;
  x2 = ed_x1+33;
  y = ed_y+ed_raw;

  if (clear_other) {
    highlight(x1, x2, y, 0x70);
    highlight(x1, x2, y+1, 0x70);
  }
  if (from == -1) return;

  h_x1 = max(from, to);
  h_x2 = min(from, to);

  if ((h_x1 >= 32) && (h_x2 >= 32))
    highlight(x2-(h_x1-32), x2-(h_x2-32), y, ed_hl);
  else if ((h_x1 < 32) && (h_x2 < 32))
    highlight(x2-h_x1, x2-h_x2, y+1, ed_hl);
  else /* if ((h_x1 >= 32) && (h_x2 < 32)) */ {
    highlight(x2-(h_x1-32), x2, y, ed_hl);
    highlight(x1, x2-h_x2, y+1, ed_hl);
  }
}

// functions for changing properties of the descriptor
// and for highlighting the proper parts of it
int nop(DESCRIPTOR *d1, int action) {
  raw_hilight(-1,-1,1); // only clear all highlights
  return NO_OP;
}

int ed_flags(DESCRIPTOR *d1, int action) {
  int typ, index;
  if (action == HLIGHT) {
    raw_hilight(41,43,1);
    return NO_OP;
  }
  typ = d1->type<8?0:1;
  index = (d1->type >> 1) + 2 + typ;
  index = menu(10, 40, 6, 10, 48, index, 0, AL_LEFT, mnu0, 0);
  if (index == -1) return NO_OP;
  if (index > 5) { typ = 8; index -= 5; }
  else typ = 0;
  if (index == 1) index++;
  index-=2;
  d1->type = typ + (index << 1);
  return TO_RAW;
}

int ed_tseg(DESCRIPTOR *d1, int action) {
  int val;
  if (action == HLIGHT) {
    raw_hilight(16,31,1);
    return NO_OP;
  }
  val = menu_ex(32, 69, 7, 8, 48, &mtmp, 256, dt0[0], summary, 0);
  if (val == -1) return NO_OP;
  d1->seg = val << 3;
  return TO_RAW;
}

int ed_seg(DESCRIPTOR *d1, int action) {
  int val, dt;
  if (action == HLIGHT) {
    raw_hilight(16,31,1);
    return NO_OP;
  }
  dt = ((d1->seg >> 2) & 0x01) + 1;
  dt = menu(22, 31, 7, 2, 48, dt, 0, AL_LEFT, mnu1, 0);
  switch (dt) {
    case -1: val = -1; break;
    case 1: val = menu_ex(32, 69, 7, 8, 48, &mtmp, 256, dt0[0], summary, 0); break;
    case 2: val = menu_ex(32, 69, 7, 8, 48, &mtmp, 256, dt0[1], lsummary, 0); break;
  }
  if (val == -1) return NO_OP;
  d1->seg = (val << 3) + ((dt-1) << 2);
  return TO_RAW;
}

int ed_ofs(DESCRIPTOR *d1, int action) {
  unsigned int val;
  if (action == HLIGHT) {
    raw_hilight(0,15,1);
    raw_hilight(48,63,0);
    return NO_OP;
  }
  val = HEXDIALOG(3, dlg1);
  if (val == -1) return NO_OP;
  d1->ofs = val;
  return TO_RAW;
}

int ed_base(DESCRIPTOR *d1, int action) {
  unsigned int val;
  if (action == HLIGHT) {
    raw_hilight(16,39,1);
    raw_hilight(56,63,0);
    return NO_OP;
  }
  val = HEXDIALOG(3, dlg1);
//  if (val == -1) return NO_OP;
  d1->base = val;
  return TO_RAW;
}

int ed_limit(DESCRIPTOR *d1, int action) {
  unsigned int val;
  if (action == HLIGHT) {
    raw_hilight(0,15,1);
    raw_hilight(48,51,0);
    raw_hilight(55,55,0);
    return NO_OP;
  }
  val = HEXDIALOG(3, dlg1);
  d1->limit = val;
  if (val > 0xFFFFF) {
    d1->g = 1;
    d1->limit = ((val >> 12) << 12) + 4095;
  } else d1->g = 0;
  return TO_RAW;
}

int ed_pc(DESCRIPTOR *d1, int action) {
  int val;
  if (action == HLIGHT) {
    raw_hilight(32,36,1);
    return NO_OP;
  }
  val = DIALOG(2, dlg0);
  if ((val < 0) || (val > 31)) return NO_OP;
  d1->pc = val;
  return TO_RAW;
}

int ed_dpl(DESCRIPTOR *d1, int action) {
  int val;
  if (action == HLIGHT) {
    raw_hilight(45,46,1);
    return NO_OP;
  }
  val = DIALOG(2, dlg0);
  if ((val < 0) || (val > 3)) return NO_OP;
  d1->dpl = val;
  return TO_RAW;
}

int ed_p(DESCRIPTOR *d1, int action) {
  if (action == HLIGHT) {
    raw_hilight(47,47,1);
    return NO_OP;
  }
  d1->p ^= 1;
  return TO_RAW;
}

int ed_type(DESCRIPTOR *d1, int action) {
  int old;
  if (action == HLIGHT) {
    raw_hilight(40,44,1);
//    raw_hilight(54,54,0);
    return NO_OP;
  }
  old=menu(17,39,4,15,48,1,0,AL_LEFT,mnu2, 0);
  switch(old) {
    case -1: return NO_OP;
    case 0: break;
    case 1: d1->s = 0; d1->type = 0; break;
    case 2: d1->s = 1; d1->type = 2; break;
    case 3: d1->s = 1; d1->type = 10; break;
    default: d1->s = 0; d1->type = sys_biject[old-4]; break;
  }
  d1->d = 1; d1->dpl = 3; d1->p = 1;
  d1->base = 0x200000; d1->limit = 0x1FFFFF; d1->g = 1;
  d1->seg = 56; d1->ofs = 0; d1->pc = 0;
  if (sys_biject[old-4] == 5) d1->seg = 32; // Task gate
  if (sys_biject[old-4] == 2) {	// LDT - allocate extra space for each LDT
    d1->base = 0x500000 + (d1->index-4)*4096; d1->limit = 0xFFF; d1->g = 0; d1->dpl = 0;
  }
  return TO_RAW;
}

int ed_sel(DESCRIPTOR *d1, int action) {
  int val;
  if (action == HLIGHT) {
    raw_hilight(-1,-1,1);
    return NO_OP;
  }
  val = HEXDIALOG(3,dlg7);
  if ((val < 0x0) || (val > 0x7FF)) return NO_OP;
  val>>=3;
  switch(d1->dti) {
    case TB_GDT: desc_old5[0] = val; break;
    case TB_LDT: desc_old5[1] = val; break;
    default: break;
  }
  return NO_OP;
}

int ed_raw0(DESCRIPTOR *d1, int action) {
  if (action == HLIGHT) return NO_OP;
  d1->desc[0] = BINDIALOG(2, dlg6, d1->desc[0]);
  return FROM_RAW;
}

int ed_raw1(DESCRIPTOR *d1, int action) {
  if (action == HLIGHT) return NO_OP;
  d1->desc[1] = BINDIALOG(2, dlg5, d1->desc[1]);
  return FROM_RAW;
}

int (*fx[12])(DESCRIPTOR *, int);

void hilight_callback(int num) {
  fx[num](0, HLIGHT);
}

void init_fx(DESCRIPTOR *d1) {
  if (!d1) return;
  // Initial functions
  fx[0] = nop;     fx[1] = ed_type;  fx[2] = nop;
  fx[3] = ed_base; fx[4] = ed_limit; fx[5] = ed_dpl;
  fx[6] = ed_p;    fx[7] = ed_sel;   fx[8] = nop;
  fx[9] = ed_raw1; fx[10] = ed_raw0; fx[11] = nop;

  if (d1->s) {
    fx[2] = ed_flags;
  } else {
    switch(d1->type) {
      case  0: fx[3] = nop; fx[4] = nop; fx[5] = nop; fx[6] = nop; break;
      case  5: fx[3] = ed_tseg; fx[4] = nop; break; // Task Gate
      case  4: // Call Gate 286
      case 12: fx[2] = ed_pc; fx[3] = ed_seg; fx[4] = ed_ofs; break; // Call Gate
      case  6: // Int Gate 286
      case  7: // Trap Gate 286
      case 14: // Int Gate
      case 15: fx[3] = ed_seg; fx[4] = ed_ofs; break; // Trap Gate
    }
  }
}

char *summary(int sel, char *data) {
  sel%=256;
  inttostr(sel, data);
  strcat(data, ": ", 0);
  load_descriptor(sel, TB_GDT, &dtmp2);

  if ((!dtmp2.s) && (!dtmp2.type)) {
    strcat(data, "Invalid Descriptor (Empty)", 0);
    return data;
  }
  if (dtmp2.s) strcat(data, user_types[dtmp2.type<8?0:1], bit_sizes[dtmp2.d], 0);
  else strcat(data, sys_types[dtmp2.type], 0);

  return data;
}

char *lsummary(int sel, char *data) {
  sel%=256;
  inttostr(sel, data);
  strcat(data, ": ", 0);
  load_descriptor(sel, TB_LDT, &dtmp2);

  if ((!dtmp2.s) && (!dtmp2.type)) {
    strcat(data, "Invalid Descriptor (Empty)", 0);
    return data;
  }
  if (dtmp2.s) strcat(data, user_types[dtmp2.type<8?0:1], bit_sizes[dtmp2.d], 0);
  else strcat(data, sys_types[dtmp2.type], 0);

  return data;
}

char *isummary(int sel, char *data) {
  sel%=32;
  hextostr(sel, data);
  if (sel > 19) sel = 15;
  strcat(data, ": ", 0);
  strcat(data, ex_names[sel], 0);

  return data;
}

void init_menus(void) {
  int i,j;

  if (!mnu0[0]) {   // Initialize flags menu
    strcat(mnu0, "\n", 0);
    for (i=0; i<2; i++) {
      strcat(mnu0, user_types[i], "\n", 0);
      for (j=0; j<4; j++) strcat(mnu0, " ", user_flags[i][j], "\n", 0);
    }
    mnu0[strlen(mnu0)-1] = '\0';
  }

  if (!mnu2[0]) {
    strcat(mnu2, "\nEmpty\n", 0);
    for (i=0; i<2; i++) strcat(mnu2, user_types[i], "\n", 0);
    for (i=0; i<12; i++) strcat(mnu2, sys_types[sys_biject[i]], "\n", 0);
    mnu2[strlen(mnu2)-1] = '\0';
  }
}

int list_ldt(int active) {
  int i, cnt=0;
  char tmps[11];

  mnu3[0] = '\0';
  load_descriptor(active, TB_GDT, &dtmp);
  if ((dtmp.s == 0) && (dtmp.type == 2)) {
    strcat(mnu3, "Active: ", inttostr(active, tmps), 0);
    strcat(mnu3, ", Base=", hextostr(dtmp.base, tmps), "\n", 0);
  } else strcat(mnu3, "Last Active: ", inttostr(active, tmps), ", REMOVED\n", 0);
  for (i=1; i<256; i++) {
    load_descriptor(i, TB_GDT, &dtmp);
    if ((dtmp.s == 0) && (dtmp.type == 2)) {
      strcat(mnu3, inttostr(i, tmps), ": ", 0);
      strcat(mnu3, "Base=", hextostr(dtmp.base, tmps), "\n", 0);
      cnt++;
      g2l_biject[cnt] = i << 3;
    }
    if (cnt == LDT_MAX) break;
  }
  if (!cnt) return -1;
  mnu3[strlen(mnu3)-1] = '\0';
  return 0;
}

void descriptors(TABLE tab)
{
  menu_left_right = 1;
  desc_old4 = 1;
  do {
    poz(0,0);
    load_descriptor(desc_old5[tab], tab, &dtmp);
    parse_descriptor(&dtmp, info);
    init_fx(&dtmp);
    desc_old4 = menu(ed_x1,ed_x2,ed_y,11,0x70,desc_old4,0,AL_LEFT,info,hilight_callback);
    if (desc_old4 == -1) break;
    switch(menu_lr_dir) {
      case 0:
        switch(fx[desc_old4](&dtmp, EDIT)) {
          case NO_OP: break;
          case TO_RAW: store_descriptor(&dtmp); break;
          case FROM_RAW: parse_desc_from_raw(&dtmp); store_descriptor(&dtmp); break;
        }
        break;
      case K_LEFT: if (desc_old5[tab] > 0) desc_old5[tab]--; else desc_old5[tab] = 255; break;
      case K_RIGHT: if (desc_old5[tab] < 255) desc_old5[tab]++; else desc_old5[tab] = 0; break;
    }
  } while (1);
  menu_left_right = 0;
  return;
}

void descriptor_menu(void) {
  desc_old3 = 1;
  do {
    save_screen();
    switch(desc_old3=menu(menu_x,61,menu_y,0,112,desc_old3,0,AL_LEFT,desc_s0, 0)) {
      case 1: descriptors(TB_GDT); break;
      case 2: descriptors(TB_LDT); break;
      case 3: descriptors(TB_IDT); break;
      case 4:
	poz(0,0);
	if (list_ldt(load_ldtr() >> 3)) {	// No LDT descriptors in GDT
	  okno(21, 48, 7, 4, 116, "Warning", "No LDT descriptor found.\nCreate at least 1 LDT\ndescriptor in GDT");
	  read_enter();
	  load_screen();
	  return;
	}
	desc_old2 = menu(21, 54, 7, 0, 112, 1, 0, AL_LEFT, mnu3, 0);
	if (desc_old2 > 0) store_ldtr(g2l_biject[desc_old2]);
	break;
      case 5:
      case -1: load_screen(); return;
    }
    load_screen();
  } while (1);
  return;
}

void idescriptors(void)
{
  save_screen();
  mtmp.ht = 1; mtmp.it = 0;
  do {
    poz(0,0);
    desc_old2 = menu_ex(ed_x1, ed_x2, ed_y, 8, 112, &mtmp, 32, dt0[2], isummary, 0);
    if (desc_old2 == -1) break;
    sw_int(desc_old2);
  } while (1);
  load_screen();
  return;
}

// Calculate linear address from the logical address
int log2lin(int seg, int ofs, unsigned int *addr) {
  DESCRIPTOR d1;
  load_descriptor(seg>>3, (seg>>2)&1, &d1);
  if (!has_base_limit(&d1)) {
    *addr = 0;
    return ADDR_INVALID;
  }
  *addr = d1.base + ofs;
  if (!range_overlap(&d1, *addr, *addr)) {
    *addr = 0;
    return ADDR_OUTOFLIMIT;
  }

  return ADDR_OK;
}

// Try to find logical address corresponding to given linear address
// Lookup GDT then LDT for first matching segment
int lin2log(unsigned int addr, int *seg, unsigned int *ofs) {
  DESCRIPTOR d1;
  int i,t, found=0;
  for (t=0; t<=1; t++) {
    for (i=0; i<256; i++) {
      load_descriptor(i, t, &d1);
      if (!d1.s) continue;
      if (range_overlap(&d1, addr, addr)) {
        found=1;
        break;
      }
    }
    if (found) break;
  }
  if (found) {
    *seg = d1.sel_min + d1.dpl;
    *ofs = addr - d1.base;
    return ADDR_OK;
  }
  return ADDR_NOTFOUND;
}
