# Prot386 Interrupt handlers
# (C) 2006-2007 Peter Ambroz
# This code is free software and is distributed under the terms of GNU GPL


	.text
	.globl idt_setup
	.globl ex_names
	.globl cpu_ex_0C

# Fill the first 20 entries of IDT with CPU exception handlers
# Fill the next 12 entries with generic exception handlers
# Then fill remaining 224 entries with dummy handler

.macro prep_setup_int pointer number
	lea \pointer, %edx
	movl \number, %eax
	shll $3, %eax
	movl _idt_base, %edi
	addl %eax, %edi
	movl $0x00080000, %eax	# hi(%eax) = 8 (code selector)
	movw %dx, %ax		# lo(%eax) = lo(handler_offset)
	movw $0xEE00, %dx	# lo(%edx) = interrupt gate, dpl=3, present
.endm

.macro setup_int pointer number
# Setup exactly 1 handler
	prep_setup_int \pointer \number
	movl %eax, (%edi)
	movl %edx, 4(%edi)	
.endm

.macro multi_int pointer start count
# setup more handlers - each with the same handler procedure
# start = first int no.
# count = number of ints to store
	prep_setup_int \pointer \start
	movl \count, %ecx
_rp_sidt\start:
	movl %eax, (%edi)
	movl %edx, 4(%edi)	
	addl $8, %edi
	dec %ecx
	jne _rp_sidt\start
.endm

idt_setup:
	# Setup first 20 exception handlers. One after another.
	setup_int cpu_ex_00 $0x00
	setup_int cpu_ex_01 $0x01
	setup_int cpu_ex_02 $0x02
	setup_int cpu_ex_03 $0x03
	setup_int cpu_ex_04 $0x04
	setup_int cpu_ex_05 $0x05
	setup_int cpu_ex_06 $0x06
	setup_int cpu_ex_07 $0x07
	setup_int cpu_ex_08 $0x08
	setup_int cpu_ex_09 $0x09
	setup_int cpu_ex_0A $0x0A
	setup_int cpu_ex_0B $0x0B
#	setup_int cpu_ex_0C $0x0C  # TSS handler
	setup_int cpu_ex_0D $0x0D
	setup_int cpu_ex_0E $0x0E
	setup_int cpu_ex_0F $0x0F
	setup_int cpu_ex_10 $0x10
	setup_int cpu_ex_11 $0x11
	setup_int cpu_ex_12 $0x12
	setup_int cpu_ex_13 $0x13

	# 12 generic interrupts
	multi_int generic_int $0x14 $0x0C

	setup_int cpu_ex_1F $0x1F
	
	# 224 dummy interrupts
	multi_int ignore_int $0x20 $0xE0

	# Keyboard int handler
	setup_int keyb_int $0x21
	
	# INT 0C (stack fault) Task-gate entry
	movl $0x0C, %eax
	shll $3, %eax
	movl _idt_base, %edi
	addl %eax, %edi
	movl $255, %eax		# hi(%eax) = 255 (TSS selector)
	shll $19, %eax
	movl $0x8500, %edx	# lo(%edx) = task gate, dpl=0, present
	movl %eax, (%edi)
	movl %edx, 4(%edi)	

	ret

# Handler procedures section:

generic_int:
	subl $28, %esp
	movl $GST1, 24(%esp)
	movl $GST0, 20(%esp)
	movl $78, 16(%esp)
	movl $3, 12(%esp)
	movl $9, 8(%esp)
	movl $45, 4(%esp)
	movl $3, (%esp)
	call okno
	addl $28, %esp
	
	call read_enter
	jmp reboot
	
	# iret	Never reached though..

ignore_int:
	iret

# Int 08: Double fault (DF)
# Action: Halt
# This is the most beautiful exception - it does not leave return address on the stack
# Only zero is pushed there, so we know the CPU loves us.
# This is very fatal exception and there is no need to know return address.
# Machine reset is the best solution anyway, so: no problem.
cpu_ex_08:
	movw %gs, %ax
	movw %ax, %es
	movl $EX08, %esi
	movl $0x286, %edi	# screen, line 4
	movl $EX08l, %ecx
	movb $79, %ah
_for_1:
	movb (%esi), %al
	incl %esi
	cmpb $0, %al
	je _end_ex08
	stosw
	jmp _for_1
_end_ex08:
	cli
	hlt		# Halt the execution right here
	jmp _end_ex08	# NMI can wake us from the eternal halt, so we play safe
	# iret ... there's no return addres. Sit down and admit your fate..

# Int 00: Divide Error
# Action: Notify & jump to start
# Should not appear. It is fault, we must either change the CS:EIP on the stack
# or modify the faulty instruction and restart execution (implies code analysis).
# I've chosen the first alternative. Let's just modify the return pointer to point
# into some safe location. Program entry point _start is very good point.
# Segment registers are reinitialized, also the stack pointer and some other things.

cpu_ex_00:
	movl $EX00, -4(%esp)

cpu_fault_nonfatal:
	subl $4, %esp
	call okno_ex
	addl $4, %esp
	
	call read_enter
end_fault_nonfatal:
	# Check the TSS we're in
	call load_tr
	cmpw _kern_tss, %ax
	jz _kern_fault		# We're in kernel TSS.
     # we're in user TSS (maybe executing some module)
	pushf
	orl $0x4000, (%esp)	# Set the NT flag, so that iret will bring us back into kernel TSS
	popf

	iret

	# Now modify return address
_kern_fault:
	movl _flags_save, %eax
	movl %eax, 8(%esp)
	movl $8, 4(%esp)	#  CS = 8 (code selector, see GDT)
	movl $_start, (%esp)	# EIP = code entry point

	iret

# Int 01: Debug (DB)
# Int 03: Breakpoint (BP)
# Int 04: Overflow (OF)
# Action: Continue
# Should not appear in this program, let it just pass along

cpu_ex_01:
cpu_ex_03:
cpu_ex_04:
cpu_trap:
	iret

# Int 02: Non-maskable interrupt (NMI)
# Action: Notify & reboot
# This generaly means catastrophic failure in HW (i.e. RAM parity error)
# On newer systems this is generated by I/O APIC but in our program, we don't use APIC

cpu_ex_02:
	movl $EX02, -4(%esp)

cpu_fault_fatal:
	subl $4, %esp
	call okno_ex
	addl $4, %esp
	
	call read_enter
	# Check the TSS we're in
	call load_tr
	cmpw _kern_tss, %ax
	jz _kern_fault2		# We're in kernel TSS.
     # we're in user TSS (maybe executing some module)
	pushf
	orl $0x4000, (%esp)	# Set the NT flag, so that iret will bring us back into kernel TSS
	popf

	iret

_kern_fault2:
	jmp reboot
	# iret	Never reached though..

# Int 05: BOUND Range Exceeded (BR)
# Action: Notify & jump to start
# Who on this world uses BOUND instruction?
# Range checking can be done using two CMP instructions, and no need to hook the int 5.

cpu_ex_05:
	movl $EX05, -4(%esp)
	jmp cpu_fault_nonfatal

# Int 06: Invalid Opcode (UD)
# Action: Notify & reboot
# Processor could not decode next instruction. There may be serious problem in the code.
# Something strange happened and the code segment was overwritten with random values.
# We'd rather reboot the machine to play safe.

cpu_ex_06:
	call okno_st    # takes last 3 ints on the stack
	movl $EX06, -4(%esp)
	jmp cpu_fault_fatal

# Int 07: Device not Available (NM)
# Action: Notify & jump to start
# Numeric coprocessor emulation code sits here with logarithmic ruler in the hands
# ready for calculations. Also generated on EVERY FP instruction after task switch.
# Hope we have no x87 FP / MMX / SSE,SSE2,SSE3 instructions in our code (who wants them??)

cpu_ex_07:
	movl $EX07, -4(%esp)
	jmp cpu_fault_nonfatal

# Int 09: Coprocessor Segment Overrun / Reserved
# Action: Notify & jump to start
# 80387 FPU has done something dirty. It should be reset and everything is all right.
# 486 and higher doesn't generate this exception and it is reserved.

cpu_ex_09:
	movl $EX09, -4(%esp)
	jmp cpu_fault_nonfatal

# Int 0A: Invalid TSS (TS)
# Action: Notify & jump to start
# Every problem related to task switching is dumped here.
# We have an error code on the stack which contains guilty descriptor's selector

cpu_ex_0A:
#	movw $0x10, %ax
#	mov %ax, %ds
#	mov %ax, %es
	call okno_st    # takes last 3 ints on the stack
	pop %esi	# error code
	movl $EX0A, -4(%esp)
	jmp cpu_fault_nonfatal

# Int 0B: Segment Not Present
# Action: Notify & make present
# Segment-level memory virtualization aid. Application can provide lots of memory
# through segments that are not physically located in RAM. When it comes to usage
# of such segment selector, this exception is thrown. Selector is pushed as error code.

cpu_ex_0B:
	push %ds
	push %es
#	movw $0x10, %ax
#	mov %ax, %ds
#	mov %ax, %es
	call okno_st    # takes last 3 ints on the stack
	xchgl %eax, (%esp)
	pushal
	push %eax
	
	subl $4, %esp
	movl $EX0B, (%esp)
	call okno_ex
	addl $4, %esp
	call read_enter
	
	pop %eax	# error code
	testl $6, %eax	# Check only GDT selectors
	jnz end_fault_nonfatal	# Not a GDT selector -> iret to start
	
	andl $0x0000FFF8, %eax	# Clear additional flags
	movl _gdt_base, %edi	# GDT offset
	addl %eax, %edi		# descriptor offset
	
	orl $0x8000, 4(%edi)	# Set the Present bit
	
	popal
	xchgl %eax, (%esp)
	pop %es
	pop %ds
	addl $4, %esp
	
	iret

# Int 0C: Stack Fault (SS)
# This exception is handled by task-gate, because kernel stack fault would otherwise
# lead into double fault, tripple fault and hard reset.
# By switching tasks, we get fresh stack and an opportunity to fix the broken task.
# Action: Notify & jump to start
# Every problem with stack goes here. Problems with loading SS: CPL <> DPL, SS not present
# or ESP not within segment bounds. It is better to restart application

cpu_ex_0C:
	call okno_st    # takes last 3 ints on the stack
	subl $4, %esp
	movl $EX0C, (%esp)
	call okno_ex
	addl $4, %esp
	call read_enter

	# check if we were in kernel TSS
	movl _int0C_tss_base, %edx
	movzwl (%edx), %eax  # checking the previous link field
	cmpl %eax, _kern_tss
	jne _kill_task   # if not, then return to kernel task (thus killing the other task)

	# else we have to fix kernel stack (restart the application)
	movl _tss_base, %edx
	movl $_start, 32(%edx) # modify CS:EIP of the kernel TSS
	movl $8, 76(%edx)
	jmp _no_kill

_kill_task:
	movl _gdt_base, %edx
	addl %eax, %edx		# access the TSS descriptor
	andl $0xFFFFFDFF, 4(%edx) # change the busy flag to available

	movl _int0C_tss_base, %edx
	movl _kern_tss, %eax
	movw %ax, (%edx)	# modify previous link TSS to return to kernel TSS

_no_kill:	
	iret
	jmp cpu_ex_0C

# Int 0D: General Protection (GP)
# Action: Notify & jump to start
# The big trash dump of the protected mode. Anythig goes wrong in PM
# and it isn't in competention of other int's, is dumped here.
# That pityful exception handler must jump to a safe location (ignoring the problems)
# or try to analyze the code (lots of assembler coding, including disassembler, phew)

cpu_ex_0D:
#	movw $0x10, %ax
#	mov %ax, %ds
#	mov %ax, %es
#	movl $0x7BE0, %esp
	call okno_st    # takes last 3 ints on the stack
	add $4, %esp	# forget error code
	movl $EX0D, -4(%esp)
	jmp cpu_fault_nonfatal

# Int 0E: Page-Fault (PF)
# Action: Notify and jump to start
# Indicates a non-present page or generaly dereferencing the more privileged page

cpu_ex_0E:
#	movw $0x10, %ax
#	mov %ax, %ds
#	mov %ax, %es
	call okno_st    # takes last 3 ints on the stack
	pop %esi
	movl $EX0E, -4(%esp)
	jmp cpu_fault_nonfatal
	
# Int 0F: Reserved
cpu_ex_0F:
	iret

# Int 10: x87 FPU Floating Point Error (MF)
# Action: Notify, reset FPU and jump to start
# Math error. On the stack is the next FPU instruction. Faulty instruction is on FCS:FIP.
# Handler should issue a FNSAVE instruction to save contents of FPU, modify it and restore.

cpu_ex_10:
	fninit
	movl $EX10, -4(%esp)
	jmp cpu_fault_nonfatal
	
# Int 11: Alignment check (AC)
# Action: Notify and jump to start
# Occurs only when CR0.AM=1 & EFLAGS.AC=1 & CPL=3 & program tries to access unaligned
# memory reference. Just for bothering programmers. Handler should do the job as CPL=0
# but watch out! Memory must belong to CPL=3 process.

cpu_ex_11:
	call okno_st    # takes last 3 ints on the stack
	addl $4, %esp	# pop and forget
	movl $EX11, -4(%esp)
	jmp cpu_fault_nonfatal

# Int 12: Machine check (MC)
# Action: Notify & reboot
# External hardware detected HW problem (e.g. bus error) and asserted signal to CPU
# This can be fatal and there is no guarantee that the program will continue normally.

cpu_ex_12:
	movl $EX12, -4(%esp)
	jmp cpu_fault_fatal

# Int 13: SIMD Floating Point (XF)
# Action: Notify and jump to start
# SSE/SSE2/SSE3 SIMD FP exception. There should be none of these in this program.

cpu_ex_13:
	movl $EX13, -4(%esp)
	jmp cpu_fault_nonfatal

# This is generated by soft int
cpu_ex_1F:
	call okno_st    # takes last 3 ints on the stack
	subl $28, %esp
	movl $EX1F, 24(%esp)
	movl $GST0, 20(%esp)
	movl $31, 16(%esp)
	movl $3, 12(%esp)
	movl $9, 8(%esp)
	movl $45, 4(%esp)
	movl $3, (%esp)
	call okno
	addl $28, %esp

	call read_enter
	jmp end_fault_nonfatal

# Keyboard int handler
keyb_int:
	movb $0x20, %al
	outb %al, $0x20
	iret

# strings:

GST0:	.string "Exception"
GST1:	.string "Unknown / reserved CPU exception\nWe cannot continue safely\nPress enter to reboot"

EX00:	.string "CPU exception 00: Divide error\nInteger division by zero\nOperation aborted. Press enter."
EX02:	.string "CPU exception 02: NMI\nHardware fault or signal from APIC\nPress enter to reboot"
EX05:	.string "CPU exception 05: BOUND Range Exceeded\nRange check error, operation aborted\nPress enter"
EX06:	.string "CPU exception 06: Invalid Opcode\nThis is a serious problem.\nPress enter to reboot"
EX07:	.string "CPU exception 07: Device not Available\nFPU emulation or FPU after task switch\nPress enter"
EX08:	.string "Fatal: Double fault exception! System halted \001"
EX08l: .long . - EX08
EX09:	.string "CPU exception 09: FPU Segment Overrun\nPress enter"
EX0A:	.string "CPU exception 0A: Invalid TSS\nProblem during the task switch\nPress enter"
EX0B:	.string "CPU exception 0B: Segment Not Present\nPress enter to fix it"
EX0C:	.string "CPU exception 0C: Stack Fault\nSS not present or ESP out of limits\nPress enter"
EX0D:	.string "CPU exception 0D:\nGeneral Protection Fault\nPress enter"
EX0E:	.string "CPU exception 0E: Page Fault\nPage address translation problem\nPress enter"
EX10:	.string "CPU exception 10: FPU Error\nFloating point math error\nPress enter"
EX11:	.string "CPU exception 11: Alignment Check\nUnaligned memory access\nPress enter"
EX12:	.string "CPU exception 12: Machine Check\nFatal machine or bus error\nPress enter to reboot"
EX13:	.string "CPU exception 13: SIMD FP\nSSE instruction class error\nPress enter"
EX1F:	.string "Test interrupt 1F: OK\nExecution successful\nPress enter to continue"

.ex00:	.string "#DE Divide error"
.ex01:	.string "#DB Debug"
.ex02:	.string "    NMI"
.ex03:	.string "#BP Breakpoint"
.ex04:	.string "#OF Overflow"
.ex05:	.string "#BR BOUND Range Exceeded"
.ex06:	.string "#UD Invalid Opcode"
.ex07:	.string "#NM Device not Available"
.ex08:	.string "#DF Double fault"
.ex09:	.string "Reserved/FPU Segment Overrun"
.ex0a:	.string "#TS Invalid TSS"
.ex0b:	.string "#NP Segment Not Present"
.ex0c:	.string "#SS Stack Fault"
.ex0d:	.string "#GP General Protection Fault"
.ex0e:	.string "#PF Page Fault"
.ex0f:	.string "Reserved"
.ex10:	.string "#MF FPU Error"
.ex11:	.string "#AC Alignment Check"
.ex12:	.string "#MC Machine Check"
.ex13:	.string "#XF SIMD FP"

ex_names:
  .long .ex00
  .long .ex01
  .long .ex02
  .long .ex03
  .long .ex04
  .long .ex05
  .long .ex06
  .long .ex07
  .long .ex08
  .long .ex09
  .long .ex0a
  .long .ex0b
  .long .ex0c
  .long .ex0d
  .long .ex0e
  .long .ex0f
  .long .ex10
  .long .ex11
  .long .ex12
  .long .ex13
