# Prot386 Entry point and low-level functions
# (C) 2006-2007 Peter Ambroz
# This code is free software and is distributed under the terms of GNU GPL

	
	.text
	.globl _start
	.globl rm_reboot, reboot, sw_int
	.globl cpu_id
	.globl xdt_read, xdt_write, memcpy, _memcpy
	.globl load_ldtr, store_ldtr
	.globl load_gdtr, load_idtr, load_tr
	.globl getcpl, getcs, getds, getss
	.globl _gdt_base, _idt_base, _ldt_base, _ldt_orig_base, _tss_base
	.globl _utss_base, _int0C_tss_base, _supp_base, _kern_tss
	.globl _page_dir, _page_table, _memory_map, _flags_save

# Welcome to the new world. Everywere around, you can see vast ammounts of high memory.
# Setup your guards, populate the memory and live here in happiness until The Final Reset.

# This is were the protected mode code starts.
# bootloader did switch into PM by jumping here, now it's time to fire-up the application.

_start:
	movw $0x10, %ax
	movw %ax, %ds
	movw %ax, %es
	movw %ax, %fs
	movw %ax, %ss
	movl $0x7C00, %esp

	addw $8, %ax
	movw %ax, %gs	# Video-RAM segment
	
	movl $0x200002, _flags_save
	movl _flags_save, %eax
	push %eax
	popf
	
	# (re)initialize the stack pointer for saving the screen contents
	call init_screen_stack

	cld
	
	# setup keyboard repeat rate to maximum
	subl $8, %esp
	movl $0x00, (%esp)
	movl $0xF3, 4(%esp)
	call kbd_send
	addl $8, %esp

	# Check if the A20 gate is really enabled
	call check_a20

	# hide the screen cursor (-> crt.s)
	call hide_cursor

	# setup interrupt handlers (-> idt.s)
	call idt_setup
	
	sti

	# setup TSS (Intel says it is necessary) (-> tss.s)
	# It will be necessary when it comes to privilege level changing calls
	call tss_setup

	# setup LDT (for experimenting with LDT) (-> tss.s)
	# Must go AFTER tss_setup
	call ldt_setup

	# Turn on paging (-> page.s)
	# Call AFTER tss_setup
	call page_setup

	# setup supplementary User-space descriptors (-> tss.s)
	call extra_setup
	
	# setup TSS for handling Kernel stack faults (-> tss.s)
	call int0C_tss_setup

	# initialize modules (-> modules.c)
	call init_modules

	# main application function (-> menu.c)
	call app_main

# Fast reboot code
reboot:
	movb $0xFE, %al
	outb %al, $0x64

# Halt, in case we didn't succeed
halt:	
	cli
	hlt
	jmp halt	# NMI can wake the processor from halt state

# Invoke software interrupt (dangerous, but handlers should withstand it)
# Works only when CS.Base == DS.Base
sw_int:
	movb	4(%esp), %ah	# parameter - int no.
	movb	$0xCD, %al	# opcode for int
	movw	%ax, _sw_int_jmp	# put the instruction in front of us (this awfuly breaks the prediction)
_sw_int_jmp:
	.word 0x0000
	ret

# void kbd_send(data, reg)
kbd_send:
	push %ebp
	movl %esp, %ebp
	
	call	empty_8042
	movb	12(%ebp), %al
	outb	%al, $0x60
	movb	8(%ebp), %al
	outb	%al, $0x60
	call	empty_8042
	
	leave
	ret
	
empty_8042:
	.word	0x00eb,0x00eb
	inb	$0x64, %al	# 8042 status port
	testb	$2, %al		# is input buffer full?
	jnz	empty_8042	# yes - loop
	ret

cpu_id:
	enter $0,$0
	push %ebx
	push %edi

	movl 8(%ebp), %eax
	movl 12(%ebp), %edi
	cpuid
	movl %eax, (%edi)
	movl %ebx, 4(%edi)
	movl %ecx, 8(%edi)
	movl %edx, 12(%edi)

	pop %edi
	pop %ebx
	leave
	ret

# read 8 byte descriptor from xDT[selector] into data
# void xdt_read(int selector, int table, int *data)
xdt_read:
	enter $0,$0
	push %edi
	push %esi
	movl 8(%ebp), %eax
	movl 12(%ebp), %edx
	movl 16(%ebp), %edi

	cmpl $0, %edx
	je _gdt_read
	cmpl $1, %edx
	je _ldt_read
	cmpl $2, %edx
	je _idt_read
_gdt_read:
	movl _gdt_base, %esi
	jmp _xdt_read_continue
_ldt_read:
	movl _ldt_base, %esi
	jmp _xdt_read_continue
_idt_read:
	movl _idt_base, %esi

_xdt_read_continue:
	shll $3, %eax
	addl %eax, %esi
	movsl
	movsl

	pop %esi
	pop %edi
	leave
	ret

# write 8 byte descriptor into xDT[selector]
# void xdt_write(int selector, int table, int *data)
xdt_write:
	enter $0,$0
	push %edi
	push %esi
	movl 8(%ebp), %eax
	movl 12(%ebp), %edx
	movl 16(%ebp), %esi

	cmpl $0, %edx
	je _gdt_write
	cmpl $1, %edx
	je _ldt_write
	cmpl $2, %edx
	je _idt_write
_gdt_write:
	movl _gdt_base, %edi
	jmp _xdt_write_continue
_ldt_write:
	movl _ldt_base, %edi
	jmp _xdt_write_continue
_idt_write:
	movl _idt_base, %edi

_xdt_write_continue:
	shll $3, %eax
	addl %eax, %edi
	movsl
	movsl

	pop %esi
	pop %edi
	leave
	ret

# Execute sldt and return result
load_ldtr:
	xor %eax, %eax
	sldt %ax
	ret

# Execute lldt and update _ldtr_base pointer
# void store_ldtr(int selector)
store_ldtr:
	enter $0,$0

	movl 8(%ebp), %eax
	andl $0x7FC, %eax
	lldt %ax
	movl _tss_base, %edx
	movw %ax, 96(%edx)	# LDT field

	movl _gdt_base, %edx
	addl %eax, %edx
	movl (%edx), %eax
	movl 4(%edx), %ecx
	shrl $16, %eax
	roll $8, %ecx
	xchgb %cl, %ch
	shll $16, %ecx
	addl %ecx, %eax
	movl %eax, _ldt_base

	leave
	ret

# get the current task segment selector from TR (via the str instruction)
load_tr:
	xorl %eax, %eax
	str %ax
	ret

# Get the GDTR
load_gdtr:
	enter $0,$0

	movl 8(%ebp), %eax
	sgdt (%eax)
	movl (%eax), %ecx
	movw 4(%eax), %dx
	movw %cx, 4(%eax)
	andl $0xFFFF, 4(%eax)
	movw %dx, %cx
	roll $16, %ecx
	movl %ecx, (%eax)

	leave
	ret

# Get the IDTR
load_idtr:
	enter $0,$0

	movl 8(%ebp), %eax
	sidt (%eax)
	movl (%eax), %ecx
	movw 4(%eax), %dx
	movw %cx, 4(%eax)
	andl $0xFFFF, 4(%eax)
	movw %dx, %cx
	roll $16, %ecx
	movl %ecx, (%eax)

	leave
	ret

# get the current CPL
# int getcpl(void)
getcpl:
	movw %cs, %ax
	andl $0x03, %eax
	ret

getcs:
	xorl %eax, %eax
	movw %cs, %ax
	ret

getds:
	xorl %eax, %eax
	movw %ds, %ax
	ret

getss:
	xorl %eax, %eax
	movw %ss, %ax
	ret

# check the A20
check_a20:
	movl $0x7E00, %ecx
	movl $0x100000, %eax
	addl %ecx, %eax
	movl $0x0, (%eax)   # write above 1MB
	movl (%ecx), %edx	   # read from below 1MB
	movl $0x7FFFFFFF, (%eax)
	movl (%ecx), %eax
	cmpl %edx, %eax
	je _check_ok
	subl $4, %esp
	movl $_check_a20_bad, (%esp)
	call write
	addl $4, %esp
	jmp halt
_check_ok:
	ret

# This one is for compatibility with gcc compiler which is producing memcpy calls
# and expects them to be implemented somewhere in the library.
# Took me a lots of debugging to find this out, so here is wrapper
memcpy:
	enter $0,$0
	movl 8(%ebp), %eax
	movl 12(%ebp), %ecx
	movl 16(%ebp), %edx
	subl $20, %esp
	movl $0, (%esp)
	movl $0, 4(%esp)
	movl %eax, 8(%esp)
	movl %ecx, 12(%esp)
	movl %edx, 16(%esp)
	call _memcpy
	leave
	ret

# void *_memcpy(int dest_seg, int src_seg, void *dest, void *src_ofs, int n)
_memcpy:
	enter $0,$0
	push %esi
	push %edi

	movl 8(%ebp), %edx
	movl 12(%ebp), %eax
	movl 16(%ebp), %edi
	movl 20(%ebp), %esi
	movl 24(%ebp), %ecx
	cmpl $0, %edx
	je _memcpy_cont1
	push %es
	movw %dx, %es
_memcpy_cont1:
	cmpl $0, %eax
	je _memcpy_cont2
	push %ds
	movw %ax, %ds
_memcpy_cont2:
	movl %ecx, %edx

	andl $3, %edx
	shr $2, %ecx

	rep movsl
	movl %edx, %ecx
	rep movsb

	cmpl $0, %eax
	je _memcpy_c1
	pop %ds
_memcpy_c1:
	movl 8(%ebp), %edx
	cmpl $0, %edx
	je _memcpy_c2
	pop %es

_memcpy_c2:
	pop %edi
	pop %esi
	leave
	ret

.macro __memcpy dest src n
	subl $12, %esp
	movl \n, %eax
	movl %eax, 8(%esp)
	leal \src, %eax
	movl %eax, 4(%esp)
	leal \dest, %eax
	movl %eax, (%esp)
	call memcpy
	addl $12, %esp
.endm

# ACPI/APM powerdown with return to real-mode
# Return to real-mode. Say goodbye to high memory, goodbye descriptors, protection, privilege levels.. :(
rm_reboot:
	cli
 	__memcpy 0x7E00 rm_switch rm_switch_l
	__memcpy 0x8000 gdt_data $24
	lidt idt_48
	movw $0x10, %dx
	movw %dx, %ds
	movw %dx, %es
	movw %dx, %fs
	movw %dx, %gs
	movw %dx, %ss
	movl $0x7C00, %esp

	ljmp $0x8,$0x7E00

	hlt	
idt_48:
	.word	0x3FF	# idt limit=1024, appropriate for real-mode
	.long	0x0	# idt base

gdt_data:
	.long 0x00000000, 0x00000000	/* Null descriptor */
	.long 0x0000ffff, 0x00009a00	/* 16-bit real-mode 64k code */
	.long 0x0000ffff, 0x00009200	/* 16-bit real-mode 64k data */

# 16-bit real-mode switch and reboot
rm_switch:
.byte	0x66, 0x0f, 0x20, 0xc0		# movl  %cr0,%eax
.byte	0x66, 0x83, 0xe0, 0x00		# andl  $0x00000000,%eax
.byte	0x66, 0x0f, 0x22, 0xc0		# movl  %eax,%cr0
.byte	0xEA, 0x00, 0x00, 0xFF, 0xFF	# jmp 0xFFFF:0x0000 - BIOS reset code
rm_switch_l: .long . - rm_switch

_check_a20_bad: .string "\nUnable to setup A20 gate.\nSYSTEM HALTED"

_gdt_base:
	.long 0x8000
_idt_base:
	.long 0x8800
_ldt_base:
	.long 0
_ldt_orig_base:
	.long 0x9000
_tss_base:
	.long 0x9800
_utss_base:
	.long 0xA000
_int0C_tss_base:
	.long 0xA400
_supp_base:
	.long 0xA800
_kern_tss:
	.long 0x20
_page_dir:
	.long 0xB000
_page_table:
	.long 0x100000
_memory_map:
	.long 0xFC00
_flags_save:
	.long 0
