// Prot386 Page manipulation
// (C) 2008-2009 Peter Ambroz
// This code is free software and is distributed under the terms of GNU GPL

#include "prototypes.h"

char pg_info[363];
char *pg_txt0[2] = { "Page Directory","Page Table" };
char *pg_txt1[2] = { "No", "Yes" };
char *pg_txt2[2] = { "Read-only", "Read/Write" };
char *pg_txt3[2] = { "Supervisor only", "User+Supervisor" };
char *pg_txt4[4] = { "Not Accessed", "Read Accessed", "Dirty", "Write Accessed"};
char *pg_txt5[4] = { "Write-back", "Write-trhough", "No Cache", "No Cache" };
char *pg_txt6[2] = { " 4kB", " 4MB" };
char *pg_txt7[2] = { "", " Global" };

extern char *dlg1; // descriptor.c
extern unsigned int mem_protect;
extern int enforce_protect;

char *pg_dlg0 = "\nGo to entry:\n";
char *pg_dlg1 = "\nGo to lin.addr.\n";
char *pg_dlg2 = "Edit binary field\n|31  24|     16| 12|  8|  4|  0|\n";
char *nopage = "Warning\nKernel memory (0-2 MB)\nmust be mapped 1:1.\nChanges will be dropped.";
char *nopgdir = "Warning\nEntered base address\nis not allowed in this entry.\nChanges will be dropped.";

#define ep_x11 3
#define ep_x12 39
#define ep_x21 40
#define ep_x22 76
#define ep_y 6
#define ep_ro 10

int pg_old0;
int pg_old1 = 0;
int pg_old2 = 1;
int pg_old3 = 0;

char temp[40];

PAGE_ENTRY ptmp1, ptmp2;

void parse_page_from_raw(PAGE_ENTRY *p1) {
  unsigned int raw;

  if (!p1) return;
  raw = p1->raw;
  p1->p = raw & 0x1;
  p1->rw = (raw >> 1) & 0x1;
  p1->us = (raw >> 2) & 0x1;
  p1->wt = (raw >> 3) & 0x1;
  p1->cd = (raw >> 4) & 0x1;
  p1->a = (raw >> 5) & 0x1;
  p1->d = (raw >> 6) & 0x1;
  p1->s = (raw >> 7) & 0x1;
  p1->pat = (raw >> 7) & 0x1;
  p1->g = (raw >> 8) & 0x1;
  p1->avl = (raw >> 9) & 0x7;
  p1->base = raw & 0xFFFFF000;
}

PAGE_ENTRY *load_page_entry(int index1, int index2, PAGE_ENTRY *p1) {
  unsigned int data[2];
  int rv;

  if (!p1) return (PAGE_ENTRY *)0;
  rv = pxe_read(index1, index2, data);
  if (index2 == -1) {
    p1->type = TB_DIR;
    p1->index = index1;
    p1->parent = (PAGE_ENTRY *)0;
    p1->lin_base = index1 << 22;
    p1->lin_limit = p1->lin_base+0x3FFFFF;
  }
  else {
    p1->type = TB_TAB;
    p1->index = index2;
    p1->lin_base = (index1 << 22) + (index2 << 12);
    p1->lin_limit = p1->lin_base+0xFFF;
  }
  p1->raw = data[0];
  if (rv) p1->valid = 1;
  else p1->valid = 0;
  if (p1->parent) {
    p1->parent->type = TB_DIR;
    p1->parent->index = index1;
    p1->parent->parent = (PAGE_ENTRY *)0;
    p1->parent->raw = data[1];
    p1->parent->valid = 1;
    parse_page_from_raw(p1->parent);
  }

  parse_page_from_raw(p1);
  return p1;
}

void parse_page_to_raw(PAGE_ENTRY *p1) {
  unsigned int raw = 0;

  if (!p1) return;
  raw |= p1->p & 0x1;
  raw |= (p1->rw & 0x1) << 1;
  raw |= (p1->us & 0x1) << 2;
  raw |= (p1->wt & 0x1) << 3;
  raw |= (p1->cd & 0x1) << 4;
  raw |= (p1->a & 0x1) << 5;

  switch(p1->type) {
    case TB_DIR:
      // raw |= (p1->d & 0x0) << 6;
      raw |= (p1->s & 0x1) << 7;
      break;
    case TB_TAB:
      raw |= (p1->d & 0x1) << 6;
      raw |= (p1->pat & 0x1) << 7;
      break;
    default: ;
  }
  raw |= (p1->g & 0x1) << 8;
  raw |= (p1->avl & 0x7) << 9;
  raw |= p1->base & 0xFFFFF000;

  p1->raw = raw;
}

void store_page_entry(PAGE_ENTRY *p1) {
  parse_page_to_raw(p1);
  if (!p1) return;
  // Kernel memory protection:
  // - Physical memory under certain limit can only be mapped 1:1 with linear memory
  // - This "protected" memory cannot be mapped to other linear addresses

  switch(p1->type) {
    case TB_DIR:
      if (enforce_protect) {
       if (((p1->index == 0) && (p1->base != _page_table)) ||
           ((p1->index != 0) && (p1->base <= _page_table))) {
         okno(13, 45, 6, 4, 116, 0, nopgdir);
         _read_enter();
         return;
       }
      }
      pxe_write(p1->index, -1, p1->raw);
      invalidate_all();
      break;
    case TB_TAB:
      if (enforce_protect) {
       if (p1->base <= mem_protect) {
        if (p1->base != p1->lin_base) {
          okno(13, 40, 6, 3, 116, 0, nopage);
          _read_enter();
          return;
        }
       }
      }

      if (!p1->parent) return;
      pxe_write(p1->parent->index, p1->index, p1->raw);
      invalidate(p1->parent->index, p1->index);
      break;
    default: ;
  }
}

// helper function for highlighting portions of raw page
void page_hilight(int tab, int from, int to, int clear_other) {
  int x1, x2, y, h_x1, h_x2;

  switch(tab) {
    case 0:
      x1 = ep_x11+2;
      x2 = ep_x12-3;
      break;
    case 1:
      x1 = ep_x21+2;
      x2 = ep_x22-3;
      break;
  }
  y = ep_y+ep_ro;

  if (clear_other) highlight(x1, x2, y, 0x70);
  if (from == -1) return;

  h_x1 = max(from, to);
  h_x2 = min(from, to);

  highlight(x2-h_x1, x2-h_x2, y, 0x7F);
}

char *parse_page_info(PAGE_ENTRY *p1, char *info) {
  char tmps[33], tmps2[12];

  info[0] = '\0';
  if (!p1) return info;
  switch(p1->type) {
    case TB_DIR:
      strcat(info, pg_txt0[0], " at ", hextostr(load_pdbr(), tmps), 0);
      strcat(info, "\nEntry no.: ", inttostr(p1->index, tmps), 0);
      break;
    case TB_TAB:
      strcat(info, pg_txt0[1], " at ", hextostr(p1->parent->base, tmps), 0);
      strcat(info, "\nEntry no.: ", inttostr(p1->index, tmps), " (Dir.Ent. ", inttostr(p1->parent->index, tmps2), ")", 0);
    default: ;
  }
  strcat(info, "\nLinear Base: ", hextostr(p1->lin_base, tmps), 0);
  if (p1->type == TB_DIR)
    strcat(info, " (", inttostr(p1->lin_base >> 20, tmps), " MB)", 0);
  else
    strcat(info, " (+", inttostr((p1->lin_base & 0x3FFFFF) >> 10, tmps), " kB)", 0);
  strcat(info, "\nPresent: ", pg_txt1[p1->p], 0);
  if (p1->valid) {
   if (p1->p) {
    strcat(info, "\nPhysical Base: ", hextostr(p1->base, tmps), 0);
    strcat(info, "\nAccess: ", pg_txt2[p1->rw], 0);
    strcat(info, "\nAccess: ", pg_txt3[p1->us], 0);
    strcat(info, "\nStatus: ", pg_txt4[p1->a + 2*p1->d], 0);
    strcat(info, "\nFlags: ", pg_txt5[p1->wt + 2*p1->cd], 0);
    if (p1->type == TB_TAB)
      strcat(info, pg_txt6[p1->s], pg_txt7[p1->g], 0);
   } else strcat(info, "\n\n\n\n\n", 0);
   strcat(info, "\n|31 Raw data     12|  8|  4|  0|", 0);
   strcat(info, "\n", pad_right(32, bintostr(p1->raw,tmps), '0'), 0);
   if (p1->p) {
    if (p1->type == TB_DIR)
      strcat(info, "\nOpen this entry", 0);
    else
      strcat(info, "\nGo back", 0);
   }
  }
  return info;
}

int ep_nop(PAGE_ENTRY *p1, int action) {
  if (action == HLIGHT)
    page_hilight((int)p1,-1,-1,1); // only clear all highlights
  return NO_OP;
}

int ep_entry(PAGE_ENTRY *p1, int action) {
  int val;
  if (action == HLIGHT) {
    page_hilight((int)p1,-1,-1,1);
    return NO_OP;
  }
  val = DIALOG(2,pg_dlg0);
  if ((val < 0) || (val > 1023)) return NO_OP;
  switch(p1->type) {
    case TB_DIR: pg_old1 = val; break;
    case TB_TAB: pg_old3 = val; break;
    default: break;
  }
  return NO_OP;
}

int ep_lin_base(PAGE_ENTRY *p1, int action) {
  unsigned int val;
  if (action == HLIGHT) {
    page_hilight((int)p1,-1,-1,1);
    return NO_OP;
  }
  val = HEXDIALOG(2,pg_dlg1);
  if (val == -1) return NO_OP;
  switch(p1->type) {
    case TB_DIR: pg_old1 = val >> 22; break;
    case TB_TAB: pg_old3 = (val >> 12) & 0x3FF; break;
    default: break;
  }
  return NO_OP;
}

int ep_p(PAGE_ENTRY *p1, int action) {
  if (action == HLIGHT) {
    page_hilight((int)p1,0,0,1);
    return NO_OP;
  }
  p1->p ^= 1;
  return TO_RAW;
}

int ep_base(PAGE_ENTRY *p1, int action) {
  unsigned int val;
  if (action == HLIGHT) {
    page_hilight((int)p1,12,31,1);
    return NO_OP;
  }
  val = HEXDIALOG(3,dlg1);
  if (val == -1) return NO_OP;
  p1->base = val & 0xFFFFF000;
  return TO_RAW;
}

int ep_rw(PAGE_ENTRY *p1, int action) {
  if (action == HLIGHT) {
    page_hilight((int)p1,1,1,1);
    return NO_OP;
  }
  p1->rw ^= 1;
  return TO_RAW;
}

int ep_us(PAGE_ENTRY *p1, int action) {
  if (action == HLIGHT) {
    page_hilight((int)p1,2,2,1);
    return NO_OP;
  }
  p1->us ^= 1;
  return TO_RAW;
}

int ep_ad(PAGE_ENTRY *p1, int action) {
  int tab;
  if (action == HLIGHT) {
    tab = (int)p1;
    if (tab == 0) page_hilight(tab,5,5,1);
    else page_hilight(tab,5,6,1);
    return NO_OP;
  }
  p1->a = 0;
  p1->d = 0;
  return TO_RAW;
}

int ep_flags(PAGE_ENTRY *p1, int action) {
  int tab;
  if (action == HLIGHT)
    tab = (int)p1;
    page_hilight(tab,3,4,1);
    if (tab == 1) page_hilight(tab,7,8,0);
  return NO_OP;
}

int ep_raw(PAGE_ENTRY *p1, int action) {
  if (action == HLIGHT) return NO_OP;
  p1->raw = BINDIALOG(2, pg_dlg2, p1->raw);
  return FROM_RAW;
}

int (*pfx[12])(PAGE_ENTRY *, int);

void init_pfx(PAGE_ENTRY *p1) {
  int i;

  pfx[0] = ep_nop; pfx[1] = ep_entry; pfx[2] = ep_lin_base;
  pfx[3] = ep_p;   pfx[4] = ep_base;  pfx[5] = ep_rw;
  pfx[6] = ep_us;  pfx[7] = ep_ad;    pfx[8] = ep_flags;
  pfx[9] = ep_nop; pfx[10] = ep_raw;  pfx[11] = ep_nop;

  if (!p1) return;
  if (!p1->p)
    for (i=4; i<=8; i++) pfx[i] = ep_nop;
}

void pde_hilight_callback(int num) {
  pfx[num]((PAGE_ENTRY *)0, HLIGHT);
}

void pte_hilight_callback(int num) {
  pfx[num]((PAGE_ENTRY *)1, HLIGHT);
}

void submenu(void) {
  do {
    load_page_entry(pg_old1, pg_old3, &ptmp1);
    parse_page_info(&ptmp1, pg_info);
    init_pfx(&ptmp1);
    pg_old2 = menu(ep_x21,ep_x22,ep_y,11,0x70,pg_old2,0,AL_LEFT,pg_info,pte_hilight_callback);
    if (pg_old2 == -1) break;
    if (!menu_lr_dir) {
      if (pg_old2 == 11) break;
    }
    switch(menu_lr_dir) {
      case 0:
        save_screen();
        switch(pfx[pg_old2](&ptmp1, EDIT)) {
          case NO_OP: break;
          case TO_RAW: store_page_entry(&ptmp1); break;
          case FROM_RAW: parse_page_from_raw(&ptmp1); store_page_entry(&ptmp1); break;
        }
        load_screen();
        break;
      case K_LEFT: if (pg_old3 > 0) pg_old3--; else pg_old3 = 1023; break;
      case K_RIGHT: if (pg_old3 < 1023) pg_old3++; else pg_old3 = 0; break;
    }
  } while (1);
  return;
}

void pagedir_menu(void) {
  menu_left_right = 1;
  pg_old0 = 1;
  ptmp1.parent = &ptmp2;
  do {
    poz(0,0);
    load_page_entry(pg_old1, -1, &ptmp2);
    parse_page_info(&ptmp2, pg_info);
    init_pfx(&ptmp2);
    pg_old0 = menu(ep_x11,ep_x12,ep_y,11,0x70,pg_old0,0,AL_LEFT,pg_info,pde_hilight_callback);
    if (pg_old0 == -1) break;
    if (!menu_lr_dir) {
      if (pg_old0 == 11) submenu();
    }
    switch(menu_lr_dir) {
      case 0:
        switch(pfx[pg_old0](&ptmp2, EDIT)) {
          case NO_OP: break;
          case TO_RAW: store_page_entry(&ptmp2); break;
          case FROM_RAW: parse_page_from_raw(&ptmp2); store_page_entry(&ptmp2); break;
        }
        break;
      case K_LEFT: if (pg_old1 > 0) pg_old1--; else pg_old1 = 1023; pg_old3 = 0; break;
      case K_RIGHT: if (pg_old1 < 1023) pg_old1++; else pg_old1 = 0; pg_old3 = 0; break;
    }
  } while (1);
  menu_left_right = 0;
  return;
}

int lin2phys(unsigned int laddr, unsigned int *paddr) {
  PAGE_ENTRY ptmp;

  ptmp.parent = NULL;
  load_page_entry(laddr>>22, (laddr>>12)&0x3FF, &ptmp);
  if (!ptmp.valid || (ptmp.valid && !ptmp.p)) {
    *paddr = 0;
    return ADDR_NONPRESENT;
  }
  *paddr = ptmp.base + (laddr&0xFFF);
  return ADDR_OK;
}

int phys2lin(unsigned int paddr, unsigned int *laddr) {
  PAGE_ENTRY p1,p2;
  int i,j, found=0;
  unsigned int pbase;

  p1.parent = &p2;
  p2.parent = NULL;
  pbase = paddr&0xFFFFF000;
  for (i=0; i<1024; i++) {
    load_page_entry(i, -1, &p1);
    if (!p1.p) continue;
    p1.parent = &p2;
    for (j=0; j<1024; j++) {
      load_page_entry(i, j, &p1);
      if (!p1.p) continue;
      if (pbase == p1.base) {
	found = 1;
	break;
      }
    }
    if (found) break;
  }
  *laddr = 0;
  if (found) {
    *laddr = (p2.index << 22) + (p1.index << 12) + (paddr&0xFFF);
    return ADDR_OK;
  }
  return ADDR_NOTFOUND;
}
