# Prot386 TSS and LDT manipulation
# (C) 2006-2007 Peter Ambroz
# This code is free software and is distributed under the terms of GNU GPL


	.text
	.globl tss_setup
	.globl ldt_setup
	.globl extra_setup
	.globl tss_prep, tss_call
	.globl int0C_tss_setup
	.globl x_call

.macro _xdt_write table selector
	subl $20, %esp
	movl %edx, 16(%esp)
	movl %eax, 12(%esp)
	leal 12(%esp), %edx
	movl %edx, 8(%esp)
	movl \table, 4(%esp)
	movl \selector, (%esp)
	call xdt_write
	addl $20, %esp
.endm

.macro _xdt_read table selector
	subl $20, %esp
	leal 12(%esp), %edx
	movl %edx, 8(%esp)
	movl \table, 4(%esp)
	movl \selector, (%esp)
	call xdt_read
	movl 16(%esp), %edx
	movl 12(%esp), %eax
	addl $20, %esp
.endm

# Functions for manipulation with TSS and LDT structures.

# We're here to create the basic 104 bytes long TSS segment, fill it with appropriate information,
# create the corresponding TSS descriptor in GDT and load its selector into Task Register as the current task.

# 2 TSS structures will be created. First for Kernel-space (this application),
# second for User-space (executing code)

tss_setup:
	# Kernel TSS
	xorl %eax, %eax
	movl $26, %ecx
	movl _tss_base, %edi
	rep stosl	# Nothing special, just 104 zeroes.

	# Now the descriptor
	movl _tss_base, %eax
	shll $16, %eax
	movw $0x0067, %ax	# Base, Limit 0x67
	movl $0x00008900, %edx	# Available 386 TSS
	_xdt_write $0 $4

	# Finaly load TR
	movw _kern_tss, %ax
	ltr %ax

	# User TSS
	xorl %eax, %eax
	movl $26, %ecx
	movl _utss_base, %edi
	rep stosl	# Nothing special, just 104 zeroes.

	# Now the descriptor
	movl _utss_base, %eax
	shll $16, %eax
	movw $0x0067, %ax	# Base, Limit 0x67
	movl $0x00008900, %edx	# Available 386 TSS
	_xdt_write $0 $5

	ret

# This will setup first LDT.
# - Clear the whole LDT segment
# - Store LDT descriptor into GDT
# - Perform LLDT instruction (set our new LDT as Active)

ldt_setup:
	movl _ldt_orig_base, %eax
	movl %eax, _ldt_base
	push %edi
	xorl %eax, %eax
	movl $512, %ecx
	movl _ldt_base, %edi
	rep stosl	# Put 2048 zeroes.

	movl _ldt_base, %eax
	roll $16, %eax
	movl %eax, %edx
	xchgb %dl, %dh
	rorl $8, %edx
	andl $0xFF0000FF, %edx
	movw $0x07FF, %ax	# limit = 2kB
	addl $0x00008200, %edx	# LDT
	_xdt_write $0 $6

	# Load LDTR register
	movw $0x30, %ax
	lldt %ax
	movl _tss_base, %edi
	movw %ax, 96(%edi)	# LDT field

	pop %edi
	ret

# Setup supplementary code and data segment descriptors
# These will be referenced from within User-space TSS
# as the source descriptors for inter-segment calls
# 5 descriptors are needed: 1 for code, 4 for data/stacks at each privilege level
# they will all reference the same memory. Descriptors are stored in GDT[250-254]
# GDT[249] is 16-bit segment for returning to real-mode

extra_setup:
	# Code segment
	movl _supp_base, %eax
	shll $16, %eax
	movw $0x07FF, %ax	# limit = 2kB
	movl $0x0040FA00, %edx	# Code, DPL=3
	_xdt_write $0 $254
	
	# Stack for CPL=0
	movl $0xFFFF, %eax	# no limit
	movl $0x00CF9200, %edx	# Data, DPL=0
	_xdt_write $0 $250

	# Stack for CPL=1
	movl $0xFFFF, %eax	# no limit
	movl $0x00CFB200, %edx	# Data, DPL=1
	_xdt_write $0 $251

	# Stack for CPL=2
	movl $0xFFFF, %eax	# no limit
	movl $0x00CFD200, %edx	# Data, DPL=2
	_xdt_write $0 $252

	# Data segment / Stack for CPL=3
	movl $0xFFFF, %eax	# no limit
	movl $0x00CFF200, %edx	# Data, DPL=3
	_xdt_write $0 $253

	# Real mode 16-bit segment
	movl $0xFFFF, %eax
	movl $0x00009200, %edx
	_xdt_write $0 $249

	# Code and data segments for general purpose (unprotected)
	movl $0x0, %eax
	shll $16, %eax
	movw $0x01FF, %ax
	movl $0x00C0FA20, %edx
	_xdt_write $0 $7

	movl $0x0, %eax
	shll $16, %eax
	movw $0x01FF, %ax
	movl $0x00C0F220, %edx
	_xdt_write $0 $8

	ret

# Prepare user TSS for calling or jumping
# - Fill these fields: LDTR, CR3, CS:EIP, DS, ES, SS:ESP, EFLAGS, SS[x]:ESP[x] x={0,1,2}
# - All segment selectors do have RPL field == CPL of the calling process (user chooses the value)
# - Adjust DPL of the supplementary (calling) code segment. Must be equal to CPL.
# - Put the instruction for JMP or CALL at the CS:[EIP]
# Parameters:	opcode = select between CALL and JMP (0x9A or 0xEA)
#		sel = full segment selector (index*8 + dti*4 + rpl)
#		cpl = CPL of calling process
# void tss_prep(int opcode, int sel, int ofs, int cpl)

tss_prep:
	enter $0,$0
	push %ebx

	movl _utss_base, %edx
	
	movl $3, %ecx
_prep_ssx:		# Fill SS[x], ESP[x], x is in ecx
	jecxz _end_prep_ssx
	decl %ecx
	movl $0xB000, 4(%edx,%ecx,8)	# ESP[x] = B000
	movl %ecx, %eax
	addl $250, %eax		# segments are 250-253
	shll $3, %eax		# SS.IDX = CPL
	addl %ecx, %eax		# SS.RPL = x
	movw %ax, 8(%edx,%ecx,8)	# SS[x]
	jmp _prep_ssx
_end_prep_ssx:
	call load_pdbr
	movl %eax, 28(%edx)	# CR3 = PDBR
	movl $0, 32(%edx)	# EIP = 0 = start of segment

	pushf
	movl (%esp), %eax
	orl $0x3000, %eax	# IOPL = 3 (IO port access)
	movl %eax, 36(%edx)	# Copy our flags
	addl $4, %esp

	# This will select data segment with correct DPL (DPL=CPL)
	movl 20(%ebp), %ecx	# CPL
	movl %ecx, %eax
	addl $250, %eax
	shll $3, %eax
	addl %ecx, %eax		# RPL

	movl $0xB000, 56(%edx)	# ESP
	
	movw %ax, 72(%edx)	# ES
	movw $0x07F0, 76(%edx)	# CS (GDT[254])
	addw %cx, 76(%edx)
	movw %ax, 80(%edx)	# SS
	movw %ax, 84(%edx)	# DS
	movw %ax, 88(%edx)	# FS (available)
	movw %gs, %ax
	movw %ax, 92(%edx)	# GS (screen descriptor)

	call load_ldtr
	movw %ax, 96(%edx)	# LDTR

	# Dereference segment selector in case of gate
        movl 12(%ebp), %eax
	xorl %ecx, %ecx
	bt $2, %eax
	setc %cl		# Table indicator in ecx
	shrl $3, %eax		# Descriptor index in eax
	_xdt_read %ecx %eax	# values are in %edx, %eax
	bt $12, %edx		# system bit
	jc _prep_nogate
	bt $10, %edx		# gate indicator
	jnc _prep_nogate
	shrl $16, %eax		# EAX = segment pointed by the gate
	jmp _prep_gate
	
_prep_nogate:
	# (Re)create 4 data/stack segment descriptors for privilege levels
	# Match their base + limit to called segment
	movl 12(%ebp), %eax	# called CS
_prep_gate:
	xorl %ecx, %ecx
	bt $2, %eax
	setc %cl		# Table indicator in ecx
	shrl $3, %eax		# Descriptor index in eax
	_xdt_read %ecx %eax	# values are in %edx, %eax
	andl $0xFFFF90FF, %edx	# clear th DPL and Type fields
	orl $0x0200, %edx	# set the Type: writable data
	push %edx
	push %eax		# save %eax, %edx, because xdt_write clobbers them
	
	movl $0x8000, %ecx	# start with DPL=4 (decremented to 3 at the start of the loop)
	movl $254, %ebx
_prep_dsx:
	jecxz _end_prep_dsx	# end, if already done for DPL=0
	movl 4(%esp), %edx
	movl (%esp), %eax	# load saved %eax, %edx
	subl $0x2000, %ecx	# decrement PL
	decl %ebx
	orl %ecx, %edx
	_xdt_write $0 %ebx
	jmp _prep_dsx
_end_prep_dsx:
	addl $8, %esp		# pop %eax, %edx
	
	# Adjust code segment DPL to CPL
	_xdt_read $0 $254	# read code descriptor
	andl $0xFFFF9FFF, %edx	# clear the DPL field
	movl 20(%ebp), %ecx	# put the new DPL into ECX
	andl $0x03, %ecx	# Ensure the DPL is <= 3
	shll $13, %ecx		# move bits to the correct position (13-14)
	orl %ecx, %edx		# set the new DPL
	_xdt_write $0 $254	# write it back

	# Prepare the CALL/JMP instruction
	movzbl 8(%ebp), %eax
	movzwl 12(%ebp), %ecx
	leal _jmp_instr, %edx
	movb %al, (%edx)
	movw %cx, 5(%edx)
	movl 16(%ebp), %eax
	movl %eax, 1(%edx)

	# Move the instruction at the right place
	subl $12, %esp
	movl _supp_base, %eax
	movl %eax, (%esp)
	movl %edx, 4(%esp)
	movl $8, 8(%esp)
	call memcpy
	addl $12, %esp

	pop %ebx
	leave
	ret

_jmp_instr:
	.byte 0x9A
	.long 0x00000000
	.word 0x0123
	.byte 0x90
	

# Call TSS
# - Transfer control to the User TSS by the direct CALL (and hope there will be an IRET at some point)
# void tss_call(void)

tss_call:
	lcall $0x28,$0x0000
	ret

int0C_tss_setup:
	xorl %eax, %eax
	movl $26, %ecx
	movl _int0C_tss_base, %edi
	rep stosl

	# Now the descriptor
	movl _int0C_tss_base, %eax
	shll $16, %eax
	movw $0x0067, %ax	# Base, Limit 0x67
	movl $0x00008900, %edx	# Available 386 TSS
	_xdt_write $0 $255

	# Fill in necessary TSS fields. Assume that others are initialized to 0.
	movl _int0C_tss_base, %edx
	
	call load_pdbr
	movl %eax, 28(%edx)	# CR3 = PDBR
	movl $cpu_ex_0C, 32(%edx) # EIP = actual handler

	pushf
	movl (%esp), %eax
	movl %eax, 36(%edx)	# Copy our flags
	addl $4, %esp

	movl $0xA800, 56(%edx)	# ESP
	
	movl $0x10, %eax
	movw %ax, 72(%edx)	# ES
	movw $8,  76(%edx)	# CS (GDT[1])
	movw %ax, 80(%edx)	# SS
	movw %ax, 84(%edx)	# DS
	movw %ax, 88(%edx)	# FS (available)
	movw %gs, %ax
	movw %ax, 92(%edx)	# GS (screen descriptor)

	call load_ldtr
	movw %ax, 96(%edx)	# LDTR

	ret

x_call:
	movl $0x50, %eax
	mov %ax, %ss
	movl $0, %esp
	ret
