using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Text;
using System.Windows.Forms;

using AForge.Imaging;
using AForge.Imaging.Filters;
using System.Drawing.Imaging;


namespace Fiducial
{
    public partial class MainForm : Form
    {
        // constants
        private const int TEMPLATE_MATCHING = 1;
        private const int FIDUCIAL_MATCHING = 2;
        private const int FIDUCIAL_PLUS_MATCHING = 3;
        private const int MAX_FIDUCIALS = 4;
        private const int MAX_TEMPLATES = 4;
        private const int MAX_DEVIATION = 2;
        private const int MAX_DEVIATION_DAMAGED = 3;
        private const int MATRIX_ROWS = 9;
        private const double ANSWER_FILLED_THRESHOLD = 0.5;
        private const double SQUARE_FILLED_THRESHOLD = 0.5;
        private const int THRESHOLD = 128;
        private const double A4_PAPER_WIDTH_CM = 21;
        private const double A4_PAPER_HEIGHT_CM = 29.7;
        private const int FIRST_CIRCLE_OFFSET = 300;
        private const int CIRCLE_SPACING = 100;
        private const int CIRCLE_RADIUS = 25;

        // global variables
        private Bitmap sourceImage;
        private double hDPI;
        private double vDPI;
        private int fiducialCount;
        private int fiducialPlusCount;
        private int templateCount;
        private Fiducial[] fiducials;
        private Fiducial[] fiducialsPlus;
        private Template[] templates;
        private String statistics;
        private String resultMarks;
        private TimeSpan duration;
        List<Bitmap> templateImages = new List<Bitmap>();
        private FiltersSequence binaryFilter = new FiltersSequence(
            new GrayscaleBT709(),
            new Threshold(THRESHOLD),
            new Invert()
        );

        public MainForm()
        {
            InitializeComponent();
            loadTemplates();
        }

        private void loadTemplates()
        {
            try
            {
                Bitmap templateImage = (Bitmap)Bitmap.FromFile("templates/A.jpg");
                AForge.Imaging.Image.FormatImage(ref templateImage);
                templateImage = binaryFilter.Apply(templateImage);
                templateImages.Add(templateImage);

                templateImage = (Bitmap)Bitmap.FromFile("templates/B.jpg");
                AForge.Imaging.Image.FormatImage(ref templateImage);
                templateImage = binaryFilter.Apply(templateImage);
                templateImages.Add(templateImage);

                templateImage = (Bitmap)Bitmap.FromFile("templates/D.jpg");
                AForge.Imaging.Image.FormatImage(ref templateImage);
                templateImage = binaryFilter.Apply(templateImage);
                templateImages.Add(templateImage);

                templateImage = (Bitmap)Bitmap.FromFile("templates/G.jpg");
                AForge.Imaging.Image.FormatImage(ref templateImage);
                templateImage = binaryFilter.Apply(templateImage);
                templateImages.Add(templateImage);
            }
            catch
            {
                MessageBox.Show("Failed loading the templates", "Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
            }
        }

        private void exitToolStripMenuItem_Click(object sender, EventArgs e)
        {
            Application.Exit();
        }

        private void initializeVariables()
        {
            fiducialCount = 0;
            fiducialPlusCount = 0;
            templateCount = 0;
            fiducials = new Fiducial[MAX_FIDUCIALS];
            fiducialsPlus = new Fiducial[MAX_FIDUCIALS];
            templates = new Template[MAX_TEMPLATES];
            statistics = "";
            resultMarks = "";
            tabControl.TabPages[3].Text = "Template";
            label1.Text = "";
            tabControl.TabPages[4].Text = "Fiducial";
            label5.Text = "";
            tabControl.TabPages[5].Text = "Fiducial+";
            label4.Text = "";
            binarisedPictureBox.Image = null;
            erosionPictureBox.Image = null;
            fiducialPictureBox.Image = null;
            fiducialPlusPictureBox.Image = null;
            templatePictureBox.Image = null;
            toolStripButton1.Enabled = true;
            toolStripButton2.Enabled = true;
            toolStripButton3.Enabled = true;
        }

        private void templateMatching(int offset, Bitmap cropImage, Rectangle rectangle, float similarity)
        {
            ExhaustiveTemplateMatching templateMatching = new ExhaustiveTemplateMatching(similarity);
            Crop cropFilter2 = new Crop(new Rectangle(offset, offset, cropImage.Width - offset, cropImage.Height - offset));
            Bitmap cropImage2 = cropFilter2.Apply(cropImage);
            for (int i = 0; i < templateImages.Count; i++)
            {
                TemplateMatch[] matchingsFound = templateMatching.ProcessImage(cropImage2, templateImages[i]);
                if (matchingsFound.Length == 1)
                {
                    Rectangle rect = new Rectangle(rectangle.X + matchingsFound[0].Rectangle.X + offset, rectangle.Y + matchingsFound[0].Rectangle.Y + offset, matchingsFound[0].Rectangle.Width, matchingsFound[0].Rectangle.Height);
                    templates[i] = new Template(i, rect);
                    templateCount++;
                    break;
                }
            }
        }


        private int fiducialMatching(Bitmap cropImage, BlobCounter topologyBlobCounter, int deviation)
        {
            int ID = -1;
            Boolean isFiducial = false;
            Boolean isFilled = false;

            topologyBlobCounter.ProcessImage(cropImage);
            Rectangle[] topologyRectangles = topologyBlobCounter.GetObjectRectangles();
            if (topologyBlobCounter.ObjectsCount == 2 || topologyBlobCounter.ObjectsCount == 3)
            {
                Point center = new Point(cropImage.Width / 2, cropImage.Height / 2);
                isFiducial = true;
                for (int i = 1; i < topologyRectangles.Length; i++)
                {
                    isFiducial &= (System.Math.Abs((topologyRectangles[i].X + topologyRectangles[i].Width / 2) - center.X) <= deviation) && (System.Math.Abs((topologyRectangles[i].Y + topologyRectangles[i].Height / 2) - center.Y) <= deviation);
                    if (!isFiducial)
                        break;
                    if (i == topologyRectangles.Length - 1)
                    {
                        if (topologyRectangles[i].Width > 1 && topologyRectangles[i].Height > 1)
                        {
                            Rectangle rect = new Rectangle(topologyRectangles[i].X + topologyRectangles[i].Width / 4, topologyRectangles[i].Y + topologyRectangles[i].Height / 4, topologyRectangles[i].Width / 2, topologyRectangles[i].Height / 2);
                            Crop cropFilter2 = new Crop(rect);
                            Bitmap cropImage2 = cropFilter2.Apply(cropImage);
                            double threshold = SQUARE_FILLED_THRESHOLD * cropImage2.Width * cropImage2.Width;
                            isFilled = countArea(cropImage2) > threshold;
                        }
                    }
                }
            }

            if (isFiducial)
            {
                ID = (isFilled) ? topologyBlobCounter.ObjectsCount - 2 : topologyBlobCounter.ObjectsCount;
                foreach (Fiducial fiducial in fiducials)
                {
                    if (fiducial != null && fiducial.Id == ID)
                        return -1;
                }
            }

            return ID;
        }

        private int fiducialPlusMatching(Bitmap cropImage, Rectangle[] topologyRectangles, int deviation)
        {
            int ID = -1;
            Boolean isFiducial = false;
            Boolean isFilled = false;
            if (topologyRectangles.Length == 2)
            {
                Point center = new Point(cropImage.Width / 2, cropImage.Height / 2);
                isFiducial = (System.Math.Abs((topologyRectangles[1].X + topologyRectangles[1].Width / 2) - center.X) <= deviation) && (System.Math.Abs((topologyRectangles[1].Y + topologyRectangles[1].Height / 2) - center.Y) <= deviation);
                if (isFiducial && topologyRectangles[1].Width > 1 && topologyRectangles[1].Height > 1)
                {
                    Rectangle rect = new Rectangle(topologyRectangles[1].X + topologyRectangles[1].Width / 4, topologyRectangles[1].Y + topologyRectangles[1].Height / 4, topologyRectangles[1].Width / 2, topologyRectangles[1].Height / 2);
                    Crop cropFilter2 = new Crop(rect);
                    Bitmap cropImage2 = cropFilter2.Apply(cropImage);
                    double threshold = SQUARE_FILLED_THRESHOLD * cropImage2.Width * cropImage2.Width;
                    isFilled = countArea(cropImage2) > threshold;
                }
            }
            else if (topologyRectangles.Length == 3)
            {
                Point center = new Point(cropImage.Width / 2, cropImage.Height / 2);
                isFiducial = (System.Math.Abs((topologyRectangles[1].X + topologyRectangles[1].Width / 2 + topologyRectangles[2].X + topologyRectangles[2].Width / 2) / 2 - center.X) <= MAX_DEVIATION) && (System.Math.Abs((topologyRectangles[1].Y + topologyRectangles[1].Height / 2 + topologyRectangles[2].Y + topologyRectangles[2].Height / 2) / 2 - center.Y) <= MAX_DEVIATION);
                if (isFiducial && topologyRectangles[1].Width > 1 && topologyRectangles[1].Height > 1)
                {
                    Rectangle rect = new Rectangle(topologyRectangles[1].X + topologyRectangles[1].Width / 4, topologyRectangles[1].Y + topologyRectangles[1].Height / 4, topologyRectangles[1].Width / 2, topologyRectangles[1].Height / 2);
                    Crop cropFilter2 = new Crop(rect);
                    Bitmap cropImage2 = cropFilter2.Apply(cropImage);
                    double threshold = SQUARE_FILLED_THRESHOLD * cropImage2.Width * cropImage2.Width;
                    isFilled = countArea(cropImage2) > threshold;
                    if (isFilled)
                    {
                        if (topologyRectangles[2].Width > 1 && topologyRectangles[2].Height > 1)
                        {
                            rect = new Rectangle(topologyRectangles[2].X + topologyRectangles[2].Width / 4, topologyRectangles[2].Y + topologyRectangles[2].Height / 4, topologyRectangles[2].Width / 2, topologyRectangles[2].Height / 2);
                            cropFilter2 = new Crop(rect);
                            cropImage2 = cropFilter2.Apply(cropImage);
                            threshold = SQUARE_FILLED_THRESHOLD * cropImage2.Width * cropImage2.Width;
                            isFilled = countArea(cropImage2) > threshold;
                        }
                    }
                }
            }
            if (isFiducial)
            {
                ID = (isFilled) ? topologyRectangles.Length - 2 : topologyRectangles.Length;
                foreach (Fiducial fiducial in fiducialsPlus)
                {
                    if (fiducial != null && fiducial.Id == ID)
                        return -1;
                }
            }
            return ID;
        }

        private void searchCandidates(Bitmap image, Rectangle[] rectangles, Boolean isNormalSearch, int algorithm)
        {
            BlobCounter topologyBlobCounter = new BlobCounter();
            topologyBlobCounter.ObjectsOrder = ObjectsOrder.Size;
            foreach (Rectangle rectangle in rectangles)
            {

                if (rectangle.Width > 1 && rectangle.Height > 1)
                {
                    Crop cropFilter = new Crop(rectangle);
                    Bitmap cropImage = cropFilter.Apply(image);
                    topologyBlobCounter.ProcessImage(cropImage);
                    Rectangle[] topologyRectangles = topologyBlobCounter.GetObjectRectangles();
                    switch (algorithm)
                    {
                        case TEMPLATE_MATCHING:
                            int offset = checkSquare(cropImage);
                            if (offset > 0)
                            {
                                float similarity = isNormalSearch ? 0.9f : 0.2f;
                                templateMatching(offset, cropImage, rectangle, similarity);
                            }
                            if (templateCount >= MAX_TEMPLATES)
                                return;
                            break;
                        case FIDUCIAL_MATCHING:
                            int deviation = isNormalSearch ? MAX_DEVIATION : MAX_DEVIATION_DAMAGED;
                            int ID = fiducialMatching(cropImage, topologyBlobCounter, deviation);
                            if (ID > -1)
                            {
                                fiducials[ID] = new Fiducial(ID, rectangle);
                                fiducialCount++;
                            }
                            if (fiducialCount >= MAX_FIDUCIALS)
                                return;
                            break;
                        case FIDUCIAL_PLUS_MATCHING:
                            deviation = isNormalSearch ? MAX_DEVIATION : MAX_DEVIATION_DAMAGED;
                            ID = fiducialPlusMatching(cropImage, topologyRectangles, deviation);
                            if (ID > -1)
                            {
                                fiducialsPlus[ID] = new Fiducial(ID, rectangle);
                                fiducialPlusCount++;
                            }
                            if (fiducialPlusCount >= MAX_FIDUCIALS)
                                return;
                            break;
                    }
                }
            }
        }

        private Rectangle[] getConnectedComponentsRectangles(Bitmap image, int minWidth, int minHeight)
        {
            BlobCounter blobCounter = new BlobCounter();
            blobCounter.ObjectsOrder = ObjectsOrder.Size;
            blobCounter.FilterBlobs = true;
            blobCounter.MinWidth = minWidth;
            blobCounter.MinHeight = minHeight;
            blobCounter.ProcessImage(image);
            return blobCounter.GetObjectRectangles();
        }

        private void openToolStripMenuItem_Click(object sender, EventArgs e)
        {
            loadImage();
        }

        private void loadImage()
        {
            try
            {
                if (openFileDialog.ShowDialog() == DialogResult.OK)
                {
                    this.Text = "Fiducial - " + System.IO.Path.GetFileName(openFileDialog.FileName);
                    sourceImage = (Bitmap)Bitmap.FromFile(openFileDialog.FileName);
                    AForge.Imaging.Image.FormatImage(ref sourceImage);
                    hDPI = sourceImage.Width / A4_PAPER_WIDTH_CM * 2.54;
                    vDPI = sourceImage.Height / A4_PAPER_HEIGHT_CM * 2.54;
                    sourcePictureBox.Image = sourceImage;
                    initializeVariables();
                    tabControl.SelectTab(0);
                }
            }
            catch
            {
                MessageBox.Show("Failed loading the image", "Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
            }
        }

        // MAIN PROGRAM
        private void processImage(Bitmap image, int algorhitm)
        {
            try
            {
                Bitmap binarisedImage = binaryFilter.Apply(image);
                Bitmap erosionImage = null;
                DateTime startTime = DateTime.Now;
                // -------- computation -------- //
                Rectangle[] rectangles = getConnectedComponentsRectangles(binarisedImage, (int)(vDPI * 20 / 300), (int)(vDPI * 20 / 300));
                // normal search
                searchCandidates(binarisedImage, rectangles, true, algorhitm);
                if ((fiducialPlusCount < MAX_FIDUCIALS && algorhitm == FIDUCIAL_PLUS_MATCHING)||
                    (fiducialCount < MAX_FIDUCIALS && algorhitm == FIDUCIAL_MATCHING))
                {
                    Erosion erosionFilter = new Erosion();
                    erosionImage = erosionFilter.Apply(binarisedImage);
                    rectangles = getConnectedComponentsRectangles(erosionImage, (int)(vDPI * 20 / 300), (int)(vDPI * 20 / 300));
                    // damage search
                    searchCandidates(erosionImage, rectangles, false, algorhitm);
                }
                DateTime stopTime = DateTime.Now;
                if (fiducialPlusCount == 4)
                    evaluateTest(binarisedImage);
                // -------- end of computation -------- //
                duration = stopTime - startTime;
                showResults(binarisedImage, erosionImage, algorhitm);
            }
            catch
            {
                MessageBox.Show("Failed processing the image", "Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
            }
        }

        private void evaluateTest(Bitmap image)
        {
            Dilatation dilatation = new Dilatation();
            image = dilatation.Apply(image);
            
            int firstCircleOffset = (int) (FIRST_CIRCLE_OFFSET * Math.Sqrt((Math.Abs(fiducialsPlus[0].Rectangle.Y - fiducialsPlus[1].Rectangle.Y) * Math.Abs(fiducialsPlus[0].Rectangle.Y - fiducialsPlus[1].Rectangle.Y)) + (Math.Abs(fiducialsPlus[0].Rectangle.Y - fiducialsPlus[2].Rectangle.Y) * Math.Abs(fiducialsPlus[0].Rectangle.Y - fiducialsPlus[2].Rectangle.Y))) / 3000);
            int circleSpacing = (int)(CIRCLE_SPACING * Math.Sqrt((Math.Abs(fiducialsPlus[0].Rectangle.Y - fiducialsPlus[1].Rectangle.Y) * Math.Abs(fiducialsPlus[0].Rectangle.Y - fiducialsPlus[1].Rectangle.Y)) + (Math.Abs(fiducialsPlus[0].Rectangle.Y - fiducialsPlus[2].Rectangle.Y) * Math.Abs(fiducialsPlus[0].Rectangle.Y - fiducialsPlus[2].Rectangle.Y))) / 3000);
            
            // sign of rotation
            int sign = (fiducialsPlus[0].Rectangle.Y > fiducialsPlus[1].Rectangle.Y) ? -1 : 1;
            if (fiducialsPlus[0].Rectangle.X != fiducialsPlus[1].Rectangle.X)
            {
                // test is rotated
                double result = (double)Math.Abs(fiducialsPlus[0].Rectangle.X - fiducialsPlus[1].Rectangle.X) / (double)Math.Abs(fiducialsPlus[0].Rectangle.Y - fiducialsPlus[1].Rectangle.Y);
                double radians = Math.Atan(result);
                double angle = radians * (180 / Math.PI);
                int offsetX;
                int offsetY;
                int lineA = (int)(Math.Sin(radians) * firstCircleOffset);
                int lineB = (int)(Math.Cos(radians) * firstCircleOffset);
                int lineAA = (int)(Math.Sin(radians) * circleSpacing);
                int lineBB = (int)(Math.Cos(radians) * circleSpacing);

                if ((fiducialsPlus[0].Rectangle.X < fiducialsPlus[1].Rectangle.X && sign == 1)||(fiducialsPlus[0].Rectangle.X > fiducialsPlus[1].Rectangle.X && sign == -1))
                {
                    // rotation left
                    offsetX = lineA + lineB;
                    offsetY = lineB - lineA;
                    for (int i = 0; i < MATRIX_ROWS; i++)
                        resultMarks += " " + checkMarks(image, fiducialsPlus[0].Rectangle.X + fiducialsPlus[0].Rectangle.Width / 2 + (offsetX + i * lineAA) * sign, fiducialsPlus[0].Rectangle.Y + fiducialsPlus[0].Rectangle.Height / 2 + (offsetY + i * lineBB) * sign, lineBB * sign, -lineAA * sign);
                }
                else
                {
                    // rotation right
                    offsetX = lineB - lineA;
                    offsetY = lineA + lineB;
                    for (int i = 0; i < MATRIX_ROWS; i++)
                        resultMarks += " " + checkMarks(image, fiducialsPlus[0].Rectangle.X + fiducialsPlus[0].Rectangle.Width / 2 + (offsetX - i * lineAA) * sign, fiducialsPlus[0].Rectangle.Y + fiducialsPlus[0].Rectangle.Height / 2 + (offsetY + i * lineBB) * sign, lineBB * sign, lineAA * sign);
                }
            } else {
                for (int i = 0; i < MATRIX_ROWS; i++)
                    resultMarks += " " + checkMarks(image, fiducialsPlus[0].Rectangle.X + fiducialsPlus[0].Rectangle.Width / 2 + firstCircleOffset * sign, fiducialsPlus[0].Rectangle.Y + fiducialsPlus[0].Rectangle.Width / 2 + (firstCircleOffset + i * circleSpacing) * sign, circleSpacing * sign, 0);
            }
        }

        private char checkMarks(Bitmap image, int pointX, int pointY, int nextX, int nextY)
        {
            int circleRadius = (int) (vDPI * CIRCLE_RADIUS / 300);
            for (char i = 'A'; i <= 'D'; i++)
            {
                Rectangle rectangle = new Rectangle(pointX - circleRadius, pointY - circleRadius, circleRadius * 2, circleRadius * 2);
                Crop cropFilter = new Crop(rectangle);
                Bitmap markImage = cropFilter.Apply(image);
                if (countArea(markImage) > ANSWER_FILLED_THRESHOLD * Math.PI * circleRadius * circleRadius)
                {
                    return i;
                }
                pointX += nextX;
                pointY += nextY;
            }
            return 'X';
        }

        private void showResults(Bitmap binarisedImage, Bitmap erosionImage, int algorithm)
        {
            // draw rectangles
            Bitmap rectangleImage = AForge.Imaging.Image.Clone(sourceImage);
            // show statistics
            switch (algorithm)
            {
                case TEMPLATE_MATCHING:
                    drawRectangles(rectangleImage, "template", Color.Red);
                    templatePictureBox.Image = rectangleImage;
                    tabControl.TabPages[3].Text = "Template (" + templateCount + ")";
                    tabControl.SelectTab(3);
                    for (int i = 0; i < MAX_TEMPLATES; i++)
                    {
                        if (templates[i] != null)
                            statistics += "TEMPLATE " + templates[i].Id + " (" + (templates[i].Rectangle.X + templates[i].Rectangle.Width / 2) + ", " + (templates[i].Rectangle.Y + templates[i].Rectangle.Height / 2) + ")     ";
                    }
                    label4.Text = "Found: " + (statistics == "" ? "nothing" : statistics) + "\nDuration: " + duration;
                    toolStripButton3.Enabled = false;
                    break;
                case FIDUCIAL_MATCHING:
                    drawRectangles(rectangleImage, "fiducial", Color.Red);
                    fiducialPictureBox.Image = rectangleImage;
                    tabControl.TabPages[4].Text = "Fiducial (" + fiducialCount + ")";
                    tabControl.SelectTab(4);
                    for (int i = 0; i < MAX_FIDUCIALS; i++)
                    {
                        if (fiducials[i] != null)
                            statistics += "FIDUCIAL " + fiducials[i].Id + " (" + (fiducials[i].Rectangle.X + fiducials[i].Rectangle.Width / 2) + ", " + (fiducials[i].Rectangle.Y + fiducials[i].Rectangle.Height / 2) + ")     ";
                    }
                    label1.Text = "Found: " + (statistics == "" ? "nothing" : statistics) + "\nDuration: " + duration;
                    toolStripButton1.Enabled = false;
                    break;
                case FIDUCIAL_PLUS_MATCHING:
                    drawRectangles(rectangleImage, "fiducial+", Color.Red);
                    fiducialPlusPictureBox.Image = rectangleImage;
                    tabControl.TabPages[5].Text = "Fiducial+ (" + fiducialPlusCount + ")";
                    tabControl.SelectTab(5);
                    for (int i=0; i < MAX_FIDUCIALS; i++)
                    {
                        if (fiducialsPlus[i] != null)
                            statistics += "FIDUCIAL+ " + fiducialsPlus[i].Id + " (" + (fiducialsPlus[i].Rectangle.X + fiducialsPlus[i].Rectangle.Width / 2) + ", " + (fiducialsPlus[i].Rectangle.Y + fiducialsPlus[i].Rectangle.Height / 2) + ")     ";
                    }
                    label5.Text = "Found: " + (statistics == "" ? "nothing" : statistics) + "\nDuration: " + duration + "     " + (resultMarks == "" ? "" : "Result:" + resultMarks);
                    toolStripButton2.Enabled = false;
                    break;
            }

            // show pictures in tabs
            binarisedPictureBox.Image = binarisedImage;
            erosionPictureBox.Image = erosionImage;
            statistics = "";
        }

        private void drawRectangles(Bitmap image, String type, Color color)
        {
            BitmapData data = image.LockBits(
                new Rectangle(0, 0, image.Width, image.Height),
                ImageLockMode.ReadWrite, image.PixelFormat);
            switch (type)
            {
                case "fiducial":
                    foreach (Fiducial fiducial in fiducials)
                    {
                        if (fiducial != null)
                        {
                            Rectangle r = fiducial.Rectangle;
                            for (int i = 0; i < 11; i++)
                            {
                                Drawing.Rectangle(data, r, color);
                                r.Inflate(1,1);
                            }
                        }
                    }
                    break;
                case "fiducial+":
                    foreach (Fiducial fiducial in fiducialsPlus)
                    {
                        if (fiducial != null)
                        {
                            Rectangle r = fiducial.Rectangle;
                            for (int i = 0; i < 11; i++)
                            {
                                Drawing.Rectangle(data, r, color);
                                r.Inflate(1, 1);
                            }
                        }
                    }
                    break;
                case "template":
                    foreach (Template template in templates)
                    {
                        if (template != null)
                        {
                            Rectangle r = template.Rectangle;
                            for (int i = 0; i < 11; i++)
                            {
                                Drawing.Rectangle(data, r, color);
                                r.Inflate(1, 1);
                            }
                        }
                    }
                    break;
            }

            image.UnlockBits(data);
        }

        private int checkSquare(Bitmap image)
        {
            int row = 0;
            BitmapData imageData = image.LockBits(
                new Rectangle(0, 0, image.Width, image.Height),
                ImageLockMode.ReadOnly, PixelFormat.Format8bppIndexed);
            
            int width = imageData.Width;
            int height = imageData.Height;
            int dstOffset = imageData.Stride - width;

            unsafe
            {
                byte* dst = (byte*)imageData.Scan0.ToPointer();
                for (int y = 0; y < height; y++)
                {
                    for (int x = 0; x < width; x++, dst++)
                    {
                        // black pixel
                        if (*dst == 0)
                        {    
                            if(y == 1 || y == height - 2 || x == 1 || x == width - 2)
                            {
                                image.UnlockBits(imageData);
                                return 0;
                            } else {
                                if(row == 0)
                                    row = y;
                            }
                        }
                    }
                    dst += dstOffset;
                }
            }

            image.UnlockBits(imageData);
            return row;
        }

        private int countArea(Bitmap image)
        {
            int area = 0;
            BitmapData imageData = image.LockBits(
                new Rectangle(0, 0, image.Width, image.Height),
                ImageLockMode.ReadOnly, PixelFormat.Format8bppIndexed);

            int width = imageData.Width;
            int height = imageData.Height;
            int dstOffset = imageData.Stride - width;

            unsafe
            {
                byte* dst = (byte*)imageData.Scan0.ToPointer();
                for (int y = 0; y < height; y++)
                {
                    for (int x = 0; x < width; x++, dst++)
                    {
                        // white pixel
                        if (*dst != 0)
                        {
                            area++;
                        }
                    }
                    dst += dstOffset;
                }
            }

            image.UnlockBits(imageData);
            return area;
        }

        private void toolStripButton4_Click(object sender, EventArgs e)
        {
            loadImage();
        }

        private void toolStripButton3_Click(object sender, EventArgs e)
        {
            processImage(sourceImage, TEMPLATE_MATCHING);
        }

        private void toolStripButton1_Click(object sender, EventArgs e)
        {
            processImage(sourceImage, FIDUCIAL_MATCHING);
        }

        private void toolStripButton2_Click(object sender, EventArgs e)
        {
            processImage(sourceImage, FIDUCIAL_PLUS_MATCHING);
        }

        private void openToolStripButton_Click(object sender, EventArgs e)
        {
            loadImage();
        }

    }

    public class Fiducial
    {
        public int Id;
        public Rectangle Rectangle;
        public Fiducial( int id, Rectangle rect)
        {
            this.Rectangle = rect;
            this.Id = id;
        }
    }

    public class Template
    {
        public int Id;
        public Rectangle Rectangle;
        public float similarity;
        public Template( int id, Rectangle rect)
        {
            this.Rectangle = rect;
            this.Id = id;
        }
    }

}