#include "stdafx.h"
#include "FibonacciSphere.h"
#include "CollisionDetector.h"
#include "CameraCone.h"

CameraCone::CameraCone(
	CapturingReality::CoordinateSystemPoint point,
	CapturingReality::CoordinateSystemPoint normal,
	CollisionDetector *ColDet,
	double minFlightHeight,
	double safeRadius,
	double minShootDist,
	double MaxShootDist,
	double maxViewAngle)
{
	cd = ColDet;
	M = new double[12];
	K = new double[9];
	double angle = 90 - maxViewAngle;
	minHeight = minFlightHeight;
	radius = safeRadius;
	minDist = minShootDist;
	maxDist = MaxShootDist;


	bitmapSize = 50; // 50x50 is a good resolution
	bitmap = new bool*[bitmapSize];
	for (int x = 0; x < bitmapSize; x++)
	{
		bitmap[x] = new bool[bitmapSize];
		for (int y = 0; y < bitmapSize; y++)
		{
			if (isInCircle(bitmapSize / 2, bitmapSize / 2, bitmapSize / 2, x, y))
			{
				bitmap[x][y] = 1;
			}else bitmap[x][y] = 0;
		}
	}

	lookAt = point;

	ortho = new CapturingReality::CoordinateSystemPoint[3];

	ortho[0] = normal;

	ortho[1] = normal;
	ortho[1].x += 1; // make up some vector

	ortho[2] = normal;
	ortho[2].y += 1; // another one

	gram_schimdt();

	M[0] = ortho[1].x;
	M[1] = ortho[1].y;
	M[2] = ortho[1].z;
	M[3] = -Vec3Dot(&ortho[1].x, &lookAt.x);

	M[4] = ortho[2].x;
	M[5] = ortho[2].y;
	M[6] = ortho[2].z;
	M[7] = -Vec3Dot(&ortho[2].x, &lookAt.x);

	M[8] = ortho[0].x;
	M[9] = ortho[0].y;
	M[10] = ortho[0].z;
	M[11] = -Vec3Dot(&ortho[0].x, &lookAt.x);


	K[0] = bitmapSize / 2 * tan(angle * DEG_TO_RAD);
	K[1] = 0;
	K[2] = bitmapSize / 2;

	K[3] = 0;
	K[4] = bitmapSize / 2 * tan(angle * DEG_TO_RAD);
	K[5] = bitmapSize / 2;

	K[6] = 0;
	K[7] = 0;
	K[8] = 1;

	Vec3Scale(&ortho[1].x, 0.5);
	
	GeneratePoints();
}

CameraCone::~CameraCone()
{
}

void CameraCone::GeneratePoints()
{
	std::vector<CapturingReality::CoordinateSystemPoint> sphere;
	for (double r = radius*2; r <= maxDist; r += radius) {
		sphere = FibonacciSphere(500, lookAt, r);
		std::vector<CapturingReality::CoordinateSystemPoint> pointsRad;
		#pragma omp parallel for
		for (int i = 0; i < sphere.size(); i++)
		{
			if (sphere[i].y < minHeight) continue;
			if (!isColliding(sphere[i])) { //updating the bitmask
				#pragma omp critical
				{
					if (r > minDist) pointsRad.push_back(sphere[i]);
					all_points.push_back(sphere[i]);
				}
			}
		}
		if (pointsRad.size() > 0) points.push_back(pointsRad);
	}
}

bool CameraCone::isColliding(CapturingReality::CoordinateSystemPoint X) {
	CapturingReality::CoordinateSystemPoint tmp, tmp2, off, off2, tmp3;

	MulM34V3(&M[0], &X.x, &tmp.x);
	MulM33V3(&K[0], &tmp.x, &tmp2.x);

	if (tmp2.z < 0) return true;

	int x = std::round(tmp2.x / tmp2.z);
	int y = std::round(tmp2.y / tmp2.z);

	if (x < 0 || x > bitmapSize-1 || y < 0 || y > bitmapSize-1) return true;

	if (!bitmap[x][y]) return true;

	if (cd->SpherePointCollision(X, 0.5)) {

		Vec3Add(&X.x, &ortho[1].x, &off.x);

		MulM34V3(&M[0], &off.x, &off2.x);
		MulM33V3(&K[0], &off2.x, &off.x);

		double rad = abs(std::round(off.x / off.z) - x);

		for (int _x = 0; _x < bitmapSize; _x++)
		{
			for (int _y = 0; _y < bitmapSize; _y++)
			{
				if (isInCircle(x, y, rad, _x, _y))
				{
					bitmap[_x][_y] = 0;
				}
			}
		}

		return true;
	}
	
	return !bitmap[x][y];
}


void CameraCone::gram_schimdt() {

	int i, j, k;
	for (i = 1; i < 3; ++i) {
		for (j = 0; j < i; ++j) {
			double scaling_factor = Vec3Dot(&ortho[j].x, &ortho[i].x)
				/ Vec3Dot(&ortho[j].x, &ortho[j].x);

			/* Subtract each scaled component of q_j from q_i */
			ortho[i].x -= scaling_factor * ortho[j].x;
			ortho[i].y -= scaling_factor * ortho[j].y;
			ortho[i].z -= scaling_factor * ortho[j].z;
		}
	}

	/* Now normalize all the 'n' orthogonal vectors */
	for (i = 0; i < 3; ++i)
		Vec3Normalize(&ortho[i].x);
}

std::vector<CapturingReality::CoordinateSystemPoint> CameraCone::getPoints() {
	return points.back();
}

std::vector<CapturingReality::CoordinateSystemPoint> CameraCone::getAllPoints() {
	return all_points;
}

std::vector<CapturingReality::CoordinateSystemPoint> CameraCone::getRandomPoints(int count)
{
	std::vector<std::vector<CapturingReality::CoordinateSystemPoint>> p = points;
	std::vector<CapturingReality::CoordinateSystemPoint> p2;
	if (p.size() == 0) return p2;
	std::vector<CapturingReality::CoordinateSystemPoint> pBack = p.back();
	//if (pBack.size() <= count) return pBack;
	while (p2.size() != count) {
		int rand = std::rand() % pBack.size();
		p2.push_back(pBack[rand]);
		pBack.erase(pBack.begin() + rand);
		if (pBack.size() == 0) {
			if (p.size() == 0) break;
			pBack = p.back();
			p.pop_back();
		}
	}
	return p2;
}
