#include "stdafx.h"
#include <algorithm>
#include "CollisionDetector.h"


HRESULT CollisionDetector::LoadModel(CapturingReality::Mvs::IMvsModel *pMvsModel, CapturingReality::CoordinateSystemResidual transform)
{
	HRESULT hr = S_OK;

	CComPtr< CapturingReality::Mvs::IMvsModel > spMvsModel = pMvsModel;

	UINT partsCount = spMvsModel->GetPartsCount();

	//find part with the most amount of triangles
	UINT maxCount = 0;
	UINT partIndexToCalc = partsCount > 0 ? 0 : 0xffffffff;
	for (UINT partIndex = 0; partIndex < min(partsCount, 10); partIndex++)
	{
		if (spMvsModel->GetPartParamsPtr(partIndex)->trianglesCount > maxCount)
		{
			maxCount = spMvsModel->GetPartParamsPtr(partIndex)->trianglesCount;
			partIndexToCalc = partIndex;
		};
	};

	if (partIndexToCalc != 0xffffffff)
	{

		CComPtr< CapturingReality::Mvs::IMvsModelPart > spPart;
		hr = spMvsModel->GetPart(partIndexToCalc, &spPart);
		if (SUCCEEDED(hr))
		{
			CapturingReality::Mvs::MvsModelPartParameters *pPartParams = spPart->GetParamsPtr();

			if (pPartParams->hasTriangulation)
			{
				CComPtr< CapturingReality::Geometry::ISceneTriangulation > spTriangulation;
				hr = spPart->LoadTriangulation(&spTriangulation);
				if (SUCCEEDED(hr))
				{
					CapturingReality::Geometry::LocalPointInfo	*pPtsInfoStart;
					CapturingReality::LocalPoint				*pPtsStart;
					CapturingReality::Geometry::Triangle		*pTrisStart;

					CapturingReality::Geometry::SceneTriangulationPointsStructureInfo pointsStructureInfo;
					hr = spTriangulation->GetPointsEx(&pPtsStart, &pPtsInfoStart, &pointsStructureInfo);
					if (SUCCEEDED(hr))
					{
						CapturingReality::CoordinateSystemAnchor anchor = *spTriangulation->GetAnchor();

						CapturingReality::Geometry::SceneTriangulationTrianglesStructureInfo trianglesStructureInfo;
						hr = spTriangulation->GetTrianglesEx(&pTrisStart, &trianglesStructureInfo);
						if (SUCCEEDED(hr))
						{
							CapturingReality::CoordinateSystemPoint t;
							CapturingReality::CoordinateSystemPoint tmp;
							for (UINT p = 0; p < pPartParams->pointsCount; p++)
							{
								t.x = pPtsStart[p].pt.x;
								t.y = pPtsStart[p].pt.y;
								t.z = pPtsStart[p].pt.z;

								if (transform.s != 0)
								{
									Vec3Scale(&t.x, transform.s);
									MulM33V3(&transform.R[0], &t.x, &tmp.x);
									Vec3Add(&tmp.x, &transform.t[0], &t.x);
								}

								x_max = max(x_max, t.x);
								x_min = min(x_min, t.x);
								z_max = max(z_max, t.z);
								z_min = min(z_min, t.z);
							}

							x_range = x_max - x_min;
							x_size = std::round(x_range / cell_size_x);
							cell_size_x = x_range / x_size;

							z_range = z_max - z_min;
							z_size = std::round(z_range / cell_size_z);
							cell_size_z = z_range / z_size;

							//triangles = new std::vector<CapturingReality::CoordinateSystemPoint*>*[x_size];
							points = new std::vector<CapturingReality::CoordinateSystemPoint>*[x_size];
							for (int i = 0; i < x_size; i++)
							{
								//triangles[i] = new std::vector<CapturingReality::CoordinateSystemPoint*>[z_size];
								points[i] = new std::vector<CapturingReality::CoordinateSystemPoint>[z_size];
							}

							int face[3];
							for (UINT p = 0; p < pPartParams->trianglesCount; p++)
							{
								face[0] = pTrisStart[p].vertices[0]; //pos of the first point in triangle
								face[1] = pTrisStart[p].vertices[1]; //second
								face[2] = pTrisStart[p].vertices[2]; //third

								CapturingReality::CoordinateSystemPoint *v = new CapturingReality::CoordinateSystemPoint[3];

								v[0].x = pPtsStart[face[0]].pt.x;
								v[0].y = pPtsStart[face[0]].pt.y;
								v[0].z = pPtsStart[face[0]].pt.z;
								
								v[1].x = pPtsStart[face[1]].pt.x;
								v[1].y = pPtsStart[face[1]].pt.y;
								v[1].z = pPtsStart[face[1]].pt.z;
								
								v[2].x = pPtsStart[face[2]].pt.x;
								v[2].y = pPtsStart[face[2]].pt.y;
								v[2].z = pPtsStart[face[2]].pt.z;

								if (transform.s != 0)
								{
									Vec3Scale(&v[0].x, transform.s);
									MulM33V3(&transform.R[0], &v[0].x, &tmp.x);
									Vec3Add(&tmp.x, &transform.t[0], &v[0].x);

									Vec3Scale(&v[1].x, transform.s);
									MulM33V3(&transform.R[0], &v[1].x, &tmp.x);
									Vec3Add(&tmp.x, &transform.t[0], &v[1].x);

									Vec3Scale(&v[2].x, transform.s);
									MulM33V3(&transform.R[0], &v[2].x, &tmp.x);
									Vec3Add(&tmp.x, &transform.t[0], &v[2].x);
								}

								std::vector<CapturingReality::CoordinateSystemPoint> subPoints = Subdivide(0.5, v[0], v[1], v[2]);
								for (UINT k = 0; k < subPoints.size(); k++)
								{
									int x = MapX(subPoints[k].x);
									int z = MapZ(subPoints[k].z);

									points[x][z].push_back(subPoints[k]);
								}
							}

							/*for (UINT p = 0; p < pPartParams->pointsCount; p++)
							{
								CapturingReality::CoordinateSystemPoint point;
								point.x = (double)pPtsStart[p].pt.x;
								point.y = (double)pPtsStart[p].pt.y;
								point.z = (double)pPtsStart[p].pt.z;

								if (transform.s != 0)
								{
									Vec3Scale(&point.x, transform.s);
									MulM33V3(&transform.R[0], &point.x, &tmp.x);
									Vec3Add(&tmp.x, &transform.t[0], &point.x);
								}

								int x = MapX(point.x);
								int z = MapZ(point.z);

								points[x][z].push_back(point);

							}*/

						};
					};
				};
			};
		};
	};

	RemoveDuplicates();

	return hr;
}

//Midpoint subdivision until triangle is not the desired size
std::vector<CapturingReality::CoordinateSystemPoint> CollisionDetector::Subdivide(
	const double maxTriangleSize,
	CapturingReality::CoordinateSystemPoint &t1,
	CapturingReality::CoordinateSystemPoint &t2,
	CapturingReality::CoordinateSystemPoint &t3)
{
	std::vector<CapturingReality::CoordinateSystemPoint> points;

	if (EuclideanDistance(t1, t2) > maxTriangleSize ||
		EuclideanDistance(t2, t3) > maxTriangleSize ||
		EuclideanDistance(t3, t1) > maxTriangleSize) {

		CapturingReality::CoordinateSystemPoint t12, t23, t31;
		std::vector<CapturingReality::CoordinateSystemPoint> new_t;
		t12 = Midpoint(t1, t2);
		t23 = Midpoint(t2, t3);
		t31 = Midpoint(t3, t1);
		new_t = Subdivide(maxTriangleSize, t1, t12, t31);
		points.insert(points.end(), new_t.begin(), new_t.end());

		new_t = Subdivide(maxTriangleSize, t2, t12, t23);
		points.insert(points.end(), new_t.begin(), new_t.end());

		new_t = Subdivide(maxTriangleSize, t3, t23, t31);
		points.insert(points.end(), new_t.begin(), new_t.end());

		new_t = Subdivide(maxTriangleSize, t12, t23, t31);
		points.insert(points.end(), new_t.begin(), new_t.end());
	}
	else {
		points.push_back(t1);
		points.push_back(t2);
		points.push_back(t3);
	}

	return points;
}


void CollisionDetector::RemoveDuplicates() {
	
	for (int x = 0; x < x_size; x++)
	{
		for (int z = 0; z < z_size; z++)
		{
			std::sort(points[x][z].begin(), points[x][z].end(),
				[](const CapturingReality::CoordinateSystemPoint &a, const CapturingReality::CoordinateSystemPoint &b) {
					return (a.x < b.x);
				}
			);

			points[x][z].erase(unique(points[x][z].begin(), points[x][z].end(),
				[](const CapturingReality::CoordinateSystemPoint &a, const CapturingReality::CoordinateSystemPoint &b) {
					return (a.x == b.x && a.y == b.y && a.z == b.z);
				}),
				points[x][z].end());
		}
	}
}

bool CollisionDetector::SpherePointCollision(CapturingReality::CoordinateSystemPoint C, double radius)
{
	if (radius > cell_size_x || radius > cell_size_z)
	{
		std::cout << "Detection Not possible. Cell size too small" << std::endl;
		return false;
	}

	if (C.x + 20 < x_min || C.x - 20 > x_max || C.z + 20 < z_min || C.z - 20 > z_max) return true;

	int x = MapX(C.x);
	int z = MapZ(C.z);

	bool map[3][3] = { false };

	if (C.x > x_min &&
		C.x < x_max &&
		C.z > z_min &&
		C.z < z_max) map[1][1] = true;
	

	for (double x_off = -radius; x_off <= radius; x_off += radius * 2)
	{	
		if (C.x + x_off > x_min &&
			C.x + x_off < x_max &&
			C.z > z_min &&
			C.z < z_max)
		{
			int _x = MapX(C.x + x_off);
			if (x != _x)
			{
				if(x_off < 0) map[0][1] = true;
				else map[2][1] = true;
			}
		}

	}

	for (double z_off = -radius; z_off <= radius; z_off += radius * 2)
	{
		if (C.z + z_off > z_min &&
			C.z + z_off < z_max &&
			C.x > x_min &&
			C.x < x_max)
		{
			int _z = MapZ(C.z + z_off);
			if (z != _z)
			{
				if (z_off < 0) map[1][0] = true;
				else map[1][2] = true;
			}
		}
	}

	if (map[0][1] && map[1][0]) map[0][0] = true;
	if (map[0][1] && map[1][2]) map[0][2] = true;
	if (map[1][0] && map[2][1]) map[2][0] = true;
	if (map[1][2] && map[2][1]) map[2][2] = true;
	
	for (int x_map = 0; x_map < 3; x_map++)
	{
		for (int z_map = 0; z_map < 3; z_map++)
		{
			if (map[x_map][z_map])
			{
				int x = MapX(C.x + radius * (x_map - 1));
				int z = MapZ(C.z + radius * (z_map - 1));

				for (int i = 0; i < points[x][z].size(); i++)
				{
					double dist = EuclideanDistance(C, points[x][z][i]);
					if (dist < radius) return true;			
				}
			}
		}
	}
	
	return false;
}

void CollisionDetector::GivePoints(
	std::vector<CapturingReality::CoordinateSystemPoint> &pts,
	std::vector<int> &ids)
{
	int id = 0;
	for (int x = 0; x < x_size; x++) {
		for (int z = 0; z < z_size; z++) {
			for (int i = 0; i < points[x][z].size(); i++)
			{
				pts.push_back(points[x][z][i]);
				ids.push_back(id);
			}
			id++;
		}
	}
}

CollisionDetector::~CollisionDetector()
{
}
