//
// Capturing Reality s.r.o.
//
// This file 'SfmHelpers.cpp' is a part of the RC Engine Samples. You may use it freely.
//

#include "stdafx.h"
#include "CommonHelpersLib.h"                             
#include <tchar.h> 
#include <stdio.h>
#include <strsafe.h>
#pragma comment(lib, "User32.lib")


HRESULT AlignImages( __in_z const wchar_t** ppImageList, __in UINT imagesCount, 
                     __in CapturingReality::IResourceCache *pCache, __in CapturingReality::IConfig *pConfig,
                     __deref_out CapturingReality::Sfm::IStructureFromMotion **ppSfm )
{
    HRESULT hr;

	//the Pipeline can use a database with camera sensors to get rough prior information.
    //We can ignore missing database of sensors, the pipeline works ok without exif focal but is faster with it.
	//You can find this database file included with RealityCapture application executables.
    CComPtr< CapturingReality::Sfm::ISfmSensorDatabase > spSensorsDb;
    CreateSfmSensorDatabase( L"..\\Resources\\metadata\\sensorsdb.xml", &spSensorsDb );
    
    // This is optional if you do not plan to use geo-referenced cameras and you can simply pass NULL
    // to CreateSfmPipeline creator.
    // Coordinates table servers as a container for storing definitions of particular coordinate systems, e.g., WGS 84 (GPS)
    // If table was provided, system will be able to understand GPS coordinates and read them from EXIF.
    // If table was not provided, GPS stored in the image EXIF will be ignored
    CComPtr<CapturingReality::Sfm::ICoordinateSystemTable> spCoordinateSystemTable;
    hr = CreateCoordinateSystemTable( &spCoordinateSystemTable );
    if (FAILED(hr))
        return hr;

    CComPtr<CapturingReality::Sfm::IStructureFromMotion> spSfm;
    hr = CapturingReality::Sfm::CreateSfmPipeline( pConfig, spCoordinateSystemTable, NULL, pCache, spSensorsDb, NULL, &spSfm);
    if (FAILED(hr))
        return hr;

    //Add images
    for ( UINT i = 0; i < imagesCount; i++ )
    {
        hr = spSfm->AddImage(ppImageList[i], NULL, 0, 0); 
        if (FAILED(hr)) 
            return hr;
    }

    printf("Calculating camera parameters\n");

    //If you wish to receive progress information or add an ability to abort the registration process, implement IProgressReport and pass it to RegisterImages.
    hr = spSfm->RegisterImages(NULL);
    if (FAILED(hr) )
        return hr;

	hr = UnlockReconstructionsIfNeeded(spSfm);
    if (FAILED(hr))
        return hr;

    return spSfm.CopyTo(ppSfm);
}

HRESULT SetSfmInputCalibrationPrior( 
    __in CapturingReality::Sfm::IStructureFromMotion *pSfm, 
    __in UINT inputIndex,
    __in CapturingReality::SfmCameraCalibration *pK, 
	__in CapturingReality::Sfm::SfmCameraCalibrationFlags flags,
	__in CapturingReality::SfmLensDistortionModel distortionModel )
{
    CComPtr< ISfmStreamInput > spInput;
    HRESULT hr = pSfm->GetImage( inputIndex, &spInput );
	if ( FAILED(hr) )
    {
		return hr;
    }

    // cast to image as it can be something else, e.g., laser scan input
    // whose calibration is fixed
    CComQIPtr< ISfmImage > spImate = spInput;
    if ( spImate )
    {
        return E_NOINTERFACE;
    }

    return spImate->SetCalibration( pK, flags, distortionModel );
}

HRESULT SetSfmInputPositionPrior( 
    __in CapturingReality::Sfm::IStructureFromMotion *pSfm, 
    __in UINT inputIndex,
    __in CapturingReality::Sfm::SfmCoordinateSystem *pPose, 
	__in CapturingReality::Sfm::SfmCoordinateSystemType type,
	__in CapturingReality::Sfm::ICoordinateSystem *pCoordinateSystem )
{
    CComPtr< ISfmStreamInput > spInput;
    HRESULT hr = pSfm->GetImage( inputIndex, &spInput );
	if ( FAILED(hr) )
    {
		return hr;
    }

    return spInput->SetPriorPoseAbsolute( pPose, type, pCoordinateSystem ); 
}

HRESULT UnlockReconstructionsIfNeeded( __in CapturingReality::Sfm::IStructureFromMotion *pSfm )
{
	// various licenses scenes are either locked or unlocked and
	// a different unlocking mechanism applies, e.g., PPI, subscriptions,
	// cloud licenses and so on

	using namespace CapturingReality::RealityCaptureServices;

	// unlock reconstructions
	ServiceConnectionContext connectionContext;
	strcpy_s( connectionContext.appToken, SDKSAMPLE_APP_TOKEN );

	CComPtr< IOnlineLicensingManager > spLicensingManager;
	HRESULT hr = CreateOnlineLicensingManager( &connectionContext, NULL, &spLicensingManager );

	if ( SUCCEEDED(hr) )
	{
		ServiceCallContext callContext = {0};

        // uncomment following line if you want UX with external browser
        // which informs user what happens.
        callContext.tokenType[0] = STT_GET;
        callContext.token[0] = "onerror";

		UINT requestEKU[] = {2, 0};
        
		InputsLicensingContext datasetContext = {0};
		datasetContext.pEku = requestEKU;
		datasetContext.pReconstruction = NULL;
		datasetContext.pProgress = NULL;
		datasetContext.pModel = NULL;
		datasetContext.pControlPoints = NULL;
		datasetContext.pSfm = pSfm;

        // initialize input licenses storage
        static CComPtr< IInputsLicensingManager > spPPIManager;
        if ( !spPPIManager )
        {
            // the licensing manager is static and hence all
            // gained licenses
            hr = CreateInputsLicensingManager( &spPPIManager );
        }
        
		InputsLicensingResultContext outContext = {0};
		if ( SUCCEEDED(hr) )
		{
            printf( "Unlocking sfm ... \n" );
    		hr = spLicensingManager->UnlockReconstructions( &callContext, &datasetContext, spPPIManager, "", InputsLicensingPinType::PT_NONE, &outContext );
        }
		if ( SUCCEEDED(hr) )
		{
			if ( outContext.authContext.tokenType[0] == STT_INTERCHANGE_TOKEN )
			{
				// user interaction / authorization is requested
				// this is done through the secured web-based interface
				WCHAR urlText[1024];
				hr = spLicensingManager->FormatRequestURL( &outContext.authContext, urlText, _countof(urlText) );
				if ( SUCCEEDED(hr) )
				{
					// Now we have a URL which can be opened in an Internet browser component. 
					// The user will follow on-screen instructions to complete the process
                    printf( "Opening browser ... \n" );
					hr = OpenWebBrowser( urlText );
				}

				if ( SUCCEEDED(hr) )
				{
					// wait for the user to finish
					callContext = outContext.authContext;
					callContext.token[0] = callContext.scratchMem;

					// wait for the user's response
					while ( SUCCEEDED(hr) && (outContext.authContext.tokenType[0] == STT_INTERCHANGE_TOKEN) )
					{
						printf( "Unlocking sfm: Waiting for a user response ... \n" );
						Sleep(1100);
						hr = spLicensingManager->UnlockReconstructions( &callContext, &datasetContext, spPPIManager, "", InputsLicensingPinType::PT_NONE, &outContext );
					}

					// the user has finished the task or closed the browser
				}
			}
		}
	}
	
	return hr;
}

HRESULT ExportRCComponent( 
	__in_z const wchar_t *pFileName, 
	__in CapturingReality::Sfm::IStructureFromMotion *pSfm,
	__in CapturingReality::Sfm::ISfmReconstruction *pReconstruction )
{
	UINT nImages = pSfm->GetImagesCount();

	UINT *pImages = new UINT[nImages];
    if ( !pImages )
    {
        return E_OUTOFMEMORY;
    }

	for ( UINT i = 0; i < nImages; i++ )
	{
		pImages[i] = i;
	};

	CComPtr<ISfmReconstructionExported> spComponent;
	HRESULT hr = pSfm->CreateSfmReconstructionExported(
			pReconstruction, 
			0,
			NULL,
			nImages, 
			pImages,
			NULL,
			&spComponent);

	if ( SUCCEEDED( hr ) )
	{
        CComPtr< IStream > spStream;
        hr = SHCreateStreamOnFile( pFileName, STGM_CREATE | STGM_WRITE, &spStream );
		if ( SUCCEEDED( hr ) )
		{
			hr = spComponent->SaveToStream( spStream );
		};
	};

	delete[] pImages;

	return hr;
}

HRESULT ImportRCComponent( __in_z const wchar_t *pComponentFileName, __in CapturingReality::Sfm::IStructureFromMotion *pSfm,
                     __deref_out CapturingReality::Sfm::ISfmReconstruction **ppReconstruction )
{
    CComPtr< CapturingReality::Sfm::ISfmReconstructionExported > spComponent;
    HRESULT hr = CapturingReality::Sfm::CreateSfmReconstructionImporter( pComponentFileName, &spComponent );
    if ( FAILED(hr) )
        return hr;

    CComPtr<ISfmReconstruction> spRec;
    hr = spComponent->QueryInterface(&spRec);
    if ( FAILED(hr) )
        return hr;

    // register images
	CapturingReality::SfmCamera *pCameras;
    UINT cameraCount;
    hr = spRec->GetCameras(&pCameras, &cameraCount);
    if ( SUCCEEDED(hr) )
    {
        for (UINT i = 0; i < cameraCount; i++)
        {
            const SfmReconstructionImageInfo *pImageInfo;
            hr = spComponent->GetImageInfo(i, &pImageInfo);
            if ( SUCCEEDED(hr) )
            {
			    const WCHAR *imageFileName = spComponent->GetString(pImageInfo->fileNameIndex);
			    if ( imageFileName )
			    {
					CComPtr< ISfmStreamInput > spStreamInput;
					hr = pSfm->AddImage( imageFileName, NULL, 0, &spStreamInput );
                    if ( FAILED(hr) )
                        return hr;
                    
                    //relocate the sfm index
					pCameras[i].sfmImage = spStreamInput->GetSfmIndex();

                    //optionally, you should copy flags stored in pImageInfo into the sfm image
                }
            }
        }
    }

    if ( SUCCEEDED(hr) )
    {
        hr = spRec.CopyTo( ppReconstruction );
    }

    return hr;
}

bool IsImageFile( __in_z const wchar_t *pFileName )
{
#define NUM_ALLOWED_EXTS 5
    wchar_t exts[NUM_ALLOWED_EXTS][5] = { L"jpg", L"jpeg", L"png", L"tif", L"tiff" };

    size_t len = wcslen( pFileName );
    size_t i = len - 2;
    while ( ( i > 0 ) && ( pFileName[i] != L'.' ) )
    {
        i--;
    }

    if ( ( i > 0 ) && ( pFileName[i] == L'.' ) )
    {
        const wchar_t *pExt = &pFileName[i + 1];
        len = len - i - 1;

        for ( UINT j = 0; j < NUM_ALLOWED_EXTS; j++ )
        {
            bool isOk = true;
            size_t extsLen = wcslen( exts[j] );
            if ( extsLen == len )
            {
                for ( size_t k = 0; k < extsLen; k++ )
                {
                    if ( towlower( pExt[k] ) != exts[j][k] )
                    {
                        isOk = false;
                        break;
                    }
                };

                if ( isOk )
                {
                    return true;
                };
            }
        }
    }

    return false;
};

HRESULT CreateSfmFromDirectory(
    __in_z      const wchar_t*											pImagesDirectory,
    __in        CapturingReality::IResourceCache*                       pCache,
    __in        CapturingReality::IConfig*                              pConfig,
    __deref_out CapturingReality::Sfm::IStructureFromMotion**           ppSfm )
{
    if ( ( pCache == NULL ) || ( pConfig == NULL ) || ( pImagesDirectory == NULL ) )
    {
        return E_INVALIDARG;
    }

    //the Pipeline can use a database with camera sensors to get rough prior information.
    //We can ignore missing database of sensors, the pipeline works ok without exif focal but is faster with it.
    //You can find this database file included with RealityCapture application executables.
    CComPtr< CapturingReality::Sfm::ISfmSensorDatabase > spSensorsDb;
    CreateSfmSensorDatabase( L"..\\Resources\\metadata\\sensorsdb.xml", &spSensorsDb );

    // This is optional if you do not plan to use geo-referenced cameras and you can simply pass NULL
    // to CreateSfmPipeline creator.
    // Coordinates table servers as a container for storing definitions of particular coordinate systems, e.g., WGS 84 (GPS)
    // If table was provided, system will be able to understand GPS coordinates and read them from EXIF.
    // If table was not provided, GPS stored in the image EXIF will be ignored
    CComPtr<CapturingReality::Sfm::ICoordinateSystemTable> spCoordinateSystemTable;
    HRESULT hr = CreateCoordinateSystemTable( &spCoordinateSystemTable );
    if ( FAILED( hr ) )
        return hr;

    //
    // Create a SFM structure. We register the images,
    // but don't align yet. It is done in CreateReconstruction() in case
    // there is aligned reconstruction found from previous runs.
    //

    CComPtr<CapturingReality::Sfm::IStructureFromMotion> spSfm;
    hr = CapturingReality::Sfm::CreateSfmPipeline( pConfig, spCoordinateSystemTable, NULL, pCache, spSensorsDb, NULL, &spSfm );
    if ( SUCCEEDED( hr ) )
    {
        // Add images.
        WIN32_FIND_DATA ffd;
        TCHAR szDir[MAX_PATH];
        size_t length_of_arg;
        HANDLE hFind = INVALID_HANDLE_VALUE;
        DWORD dwError = 0;

        // Check that the input path plus 3 is not longer than MAX_PATH.
        // Three characters are for the "\*" plus NULL appended below.
        StringCchLength( pImagesDirectory, MAX_PATH, &length_of_arg );

        if ( length_of_arg > ( MAX_PATH - 3 ) )
        {
            _tprintf( TEXT( "\nDirectory path is too long.\n" ) );
            return ( -1 );
        }

        _tprintf( TEXT( "\nTarget directory is %s\n\n" ), pImagesDirectory );

        // Prepare string for use with FindFile functions.  First, copy the
        // string to a buffer, then append '\*' to the directory name.

        StringCchCopy( szDir, MAX_PATH, pImagesDirectory );
        StringCchCat( szDir, MAX_PATH, TEXT( "\\*" ) );

        // Find the first file in the directory.

        hFind = FindFirstFile( szDir, &ffd );

        if ( INVALID_HANDLE_VALUE == hFind )
        {
            _tprintf( TEXT( "Error in FindFirstFile\n\n" ) );
            return E_FAIL;
        }

        // List all the files in the directory with some info about them.
        do
        {
            if ( ( !( ffd.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY ) ) && ( IsImageFile( ffd.cFileName ) ) )
            {
                _tprintf( TEXT( "Adding image :  %s \n" ), ffd.cFileName );

                TCHAR fileNameWithPath[2 * MAX_PATH];
                StringCchCopy( fileNameWithPath, MAX_PATH, pImagesDirectory );
                StringCchCat( fileNameWithPath, MAX_PATH, TEXT( "\\" ) );
                StringCchCat( fileNameWithPath, MAX_PATH, ffd.cFileName );

                hr = spSfm->AddImage( ( fileNameWithPath ), NULL, NULL, 0 ); 
                if ( FAILED( hr ) ) 
                {
                    break;
                }
            }
        } while ( FindNextFile( hFind, &ffd ) != 0 );
    }

    if ( SUCCEEDED( hr ) )
    {
        *ppSfm = spSfm.Detach( );
    }

    return hr;
}

HRESULT Align(
    __in        CapturingReality::Sfm::IStructureFromMotion*    pSfm,
    __deref_out CapturingReality::Sfm::ISfmReconstruction**     ppReconstruction )
{
    if ( pSfm == NULL )
    {
        return E_INVALIDARG;
    }

    HRESULT hr = S_OK;

    //
    // Align SFM structures with registered input images to obtain a reconstruction.
    //

    if ( SUCCEEDED( hr ) )
    {
        printf( "Calculating camera parameters\n" );
        hr = pSfm->RegisterImages( NULL );
    }

    //
    // More components can be reconstructed. We select the largest one,
    // which is, most likely, the most interesting one.
    //

    UINT nReconstructions = 0;

    if ( SUCCEEDED( hr ) )
    {
        printf( "Unlocking reconstructions\n" );
        hr = UnlockReconstructionsIfNeeded( pSfm );
    }

    if ( SUCCEEDED( hr ) )
    {
        nReconstructions = pSfm->GetReconstructionsCount( );
        if ( nReconstructions == 0 )
        {
            *ppReconstruction = NULL;
            return hr;
        }
    }

    UINT largestComponentIndex = 0;

    if ( SUCCEEDED( hr ) )
    {
        UINT largestComponentCameraCount = 0;

        printf( "Calculated %d components\n", nReconstructions );

        for ( UINT ireconstruction = 0; ireconstruction < nReconstructions; ireconstruction++ )
        {
            CComPtr< CapturingReality::Sfm::ISfmReconstruction > spReconstruction;
            UINT structureSize;

            hr = pSfm->GetReconstruction( ireconstruction, &spReconstruction );
            if ( SUCCEEDED( hr ) )
            {
                hr = spReconstruction->GetStructureSize( &structureSize );
            }
            if ( SUCCEEDED( hr ) )
            {
                UINT cameraCount = spReconstruction->GetCamerasCount( );
                printf( "Component %d consists of %d cameras and %d global points\n", ireconstruction, cameraCount, structureSize );

                if ( cameraCount >= largestComponentCameraCount )
                {
                    largestComponentIndex = ireconstruction;
                    largestComponentCameraCount = cameraCount;
                }
            }
            else
            {
                break;
            }
        }
    }

    // Select the component with the largest number of aligned cameras.

    if ( SUCCEEDED( hr ) )
    {
        hr = pSfm->GetReconstruction( largestComponentIndex, ppReconstruction );
    }

    return hr;
}


const double RAD_TO_DEG = ( 180 / 3.1415926535897932384626433832795 );

void PrintGpsLatitude( __in double src )
{
    double angle, min, sec;
    sec = abs( src ) * RAD_TO_DEG;
    angle = floor( sec );
    sec = ( sec - angle ) * 60;
    min = floor( sec );
    sec = ( sec - min ) * 60;
    printf( "%c%.0f,%.0f,%.2f", src >= 0 ? 'N' : 'S', angle, min, sec );
}

void PrintGpsLongitude( __in double src )
{
    double angle, min, sec;
    sec = abs( src ) * RAD_TO_DEG;
    angle = floor( sec );
    sec = ( sec - angle ) * 60;
    min = floor( sec );
    sec = ( sec - min ) * 60;
    printf( "%c%.0f,%.0f,%.2f", src >= 0 ? 'E' : 'W', angle, min, sec );
}

HRESULT PrintCalculatedCameraPositionsInGps( __in ISfmReconstruction *pReconstruction )
{
    HRESULT hr = S_OK;

    if ( pReconstruction->GetReconstructionFlags( ) & ( SCCF_GROUND_CONTROL | SCCF_METRIC ) )
    {
        CComPtr< CapturingReality::CoordinateSystems::ICoordinateSystem > spGpsCoordSystem;
        const wchar_t* g_gps_name = L"GPS (WGS 84)";
        const wchar_t* g_gps_definition = L"+proj=longlat +datum=WGS84 +no_defs";
        hr = CapturingReality::CoordinateSystems::CreateCoordinateSystem( g_gps_name, g_gps_definition, &spGpsCoordSystem );
        if ( SUCCEEDED( hr ) )
        {
            ISfmCameraModel *pCameraModel = pReconstruction->GetCameraModel( );
            CapturingReality::CoordinateSystemAnchor anchor = pReconstruction->GetAnchor( );
            CapturingReality::CoordinateSystemGroundPlane groundPlane = pReconstruction->GetGroundPlane( );

            UINT count;
            CapturingReality::SfmCamera *pCameras;
            hr = pReconstruction->GetCameras( &pCameras, &count );
            if ( SUCCEEDED( hr ) )
            {
                for ( UINT i = 0; i < count; i++, pCameras++ )
                {
                    double cameraCentre[4];
                    pCameraModel->GetCameraCentre( *pCameras, cameraCentre );

                    CapturingReality::CoordinateSystemPoint point;
                    MulV3M33( cameraCentre, groundPlane.R, &point.x );
                    Vec3Add( &point.x, &anchor.x, &point.x );

                    hr = spGpsCoordSystem->ToModel( 1, &point );

                    if ( SUCCEEDED( hr ) )
                    {
                        double longitude = ( point.x / XM_PI ) * 0.5;
                        double latitude = -( log( tan( point.y ) + ( 1.0 / cos( point.y ) ) ) / XM_PI ) * 0.5;
                        double altitude = point.z;

                        printf( " Calculated position latitude: " );
                        PrintGpsLatitude( latitude );

                        printf( " longitude: " );
                        PrintGpsLongitude( latitude );

                        printf( " altitude:%.2f\n", altitude );
                    }


                    if ( FAILED( hr ) )
                    {
                        break;
                    };
                }
            }
        }
    }
    else
    {
        hr = S_FALSE; //the component is not geo-refrerenced
    };

    return hr;
}

HRESULT PrintAlignmentStats( __in IStructureFromMotion *pSfm )
{
    using namespace CapturingReality;
    using namespace CapturingReality::Sfm;

    UINT recCount = pSfm->GetReconstructionsCount( );
    printf( "Calculated %d components\n", recCount );
    for ( UINT i = 0; i < recCount; i++ )
    {
        CComPtr< ISfmReconstruction > spReconstruction;
        pSfm->GetReconstruction( i, &spReconstruction );

        UINT structureSize;
        HRESULT hr = spReconstruction->GetStructureSize( &structureSize );
        if ( FAILED( hr ) )
        {
            return hr;
        }

        UINT cameraCount = spReconstruction->GetCamerasCount( );
        printf( "Component %d consists of %d cameras and %d global points\n", i, cameraCount, structureSize );

        // now let us print some camera information
        if ( cameraCount > 0 )
        {
            int cameraIndex = 0;

            printf( "Here is some info about Camera %d\n", cameraIndex );

            SfmCamera camera;
            hr = spReconstruction->GetCamera( cameraIndex, &camera );
            if ( FAILED( hr ) )
            {
                return hr;
            }

            // get camera input file; it is not necessary an image
            // following information are available even prior to registration
            // and are can be read the same way

            CComPtr< ISfmStreamInput > spInput;
            hr = pSfm->GetImage( camera.sfmImage, &spInput );
            if ( FAILED( hr ) )
            {
                return hr;
            }

            // Uncomment to setup calibration groups, i.e., cameras with the same
            // group ID will share the same calibrations/lens parameters
            // CComQIPtr< ISfmImage > spImage = spInput;
            // if ( spImage )
            // {
            //     SfmImageParametersGroups p;
            //     p.calibrationGroup = 1;
            //     p.distortionGroup = 1;
            //     spImage->SetParameterGrouping( p );
            // }

            UINT width, height;
            hr = spInput->GetImageSize( &width, &height );
            if ( FAILED( hr ) )
            {
                return hr;
            }

            const wchar_t *pInputFileName = spInput->GetFileName( );
            printf( " Source file name: \"%S\"\n", pInputFileName );
            printf( " Width: %d pixels\n", width );
            printf( " Height: %d pixels\n", height );

            // check if the input is geo-referenced
            // it can be in any coordinate system
            SfmCoordinateSystem csPose;
            SfmCoordinateSystemType csType;
            CComPtr< ICoordinateSystem > spCS;
            hr = spInput->GetPriorPoseAbsolute( &csPose, &csType, &spCS );
            if ( FAILED( hr ) )
            {
                return hr;
            }

            if ( csType & SFMCST_REGISTERED )
            {
                printf( " Image is geo-referenced\n" );

                if ( spCS->IsLatitudeLongitude( ) )
                {
                    double longitude = csPose.position[0];
                    double latitude = csPose.position[1];
                    double altitude = csPose.position[2];

                    printf( " Prior position latitude: " );
                    PrintGpsLatitude( latitude );

                    printf( " longitude: " );
                    PrintGpsLongitude( latitude );

                    printf( " altitude:%.2f\n", altitude );
                }
            }

            // now print some registration outcomes
            printf( " Focal length: %.1fmm\n", camera.K.K.focalLength * 36.0 );

            // all information are stored relative to the unit-sized image (the bigger side of image is 1)
            // this way image resolution is not important and you can freely downscale/upscale images in 
            // processes
            double unit2Pixel = max( width, height );
            printf( " Principal point: [%.1fpx, %.1fpx]\n", camera.K.K.principalU * unit2Pixel + width / 2.0, camera.K.K.principalU * unit2Pixel + height / 2.0 );

            // distortion values have meaning only w.r.t. selected camera model; this is defined in config prior to 
            // registration and by default it is set to Brown model

            // image is distorted in general and camera model serves to project
            // between image and 3D space
            ISfmCameraModel *pModel = spReconstruction->GetCameraModel( );
            _ASSERT( pModel );

            UINT cameraModelId = pModel->GetModelId( );

            printf( " Distortion model: " );
            switch ( cameraModelId & 0xff )
            {
            case SFMLDM_DIVISION:
                printf( "Division Model\n" );
                break;
            case SFMLDM_BROWN_R3:
                printf( "Brown 3 param\n" );
                break;
            case SFMLDM_BROWN_R3_T2:
                printf( "Brown 3 param with tangential distortion\n" );
                break;
            case SFMLDM_BROWN_R4:
                printf( "Brown 4 param\n" );
                break;
            case SFMLDM_BROWN_R4_T2:
                printf( "Brown 4 param with tangential distortion\n" );
                break;

            }

            printf( " Radial distortion: [%.1f, %.1f, %.1f, %.1f] tangential distortion: [%.1f, %.1f]\n", camera.K.rd.radial1, camera.K.rd.radial2, camera.K.rd.radial3, camera.K.rd.radial4, camera.K.rd.tangential1, camera.K.rd.tangential2 );

            // let us project some 3D point to the image space

            UINT pointsCount;
            SfmReconstructionPoint *pPoints;
            hr = spReconstruction->GetStructure( &pPoints, &pointsCount );
            _ASSERT( SUCCEEDED( hr ) );

            WorldPoint X;
            X.x = pPoints[0].X.x;
            X.y = pPoints[0].X.y;
            X.z = pPoints[0].X.z;
            X.w = pPoints[0].X.w;

            double projection[3];
            pModel->GetProjection( camera, X, projection );

            // de-homogenize 3D point 
            double x = X.x / X.w;
            double y = X.y / X.w;
            double z = X.z / X.w;

            if ( projection[2] < 0 )
            {
                printf( " The projected 3D point [%.2f, %.2f, %.2f] is behind the camera\n", x, y, z );
            }
            else
            {
                printf( " The point [%.2f, %.2f, %.2f] projects to [%.2fpx, %.2fpx] \n", x, y, z, projection[0] * unit2Pixel + width / 2, projection[1] * unit2Pixel + height / 2 );
            }

            hr = PrintCalculatedCameraPositionsInGps( spReconstruction );
        }
    }

    return S_OK;
}

